// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgExPolicyExpr.h"

#include "MtgEngine.h"

#include "MtgEvGenerator.h"

#include "MtgFactor.h"

#include "MtgNumericalExpr.h"

#include "MtgPortfolio.h"

#include "MtgParser.h"



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tExPolicyExpr::init()



{

    m_nType = xExPolicy;

    m_nExPolicy = xDontExercise;

    m_pCond = 0;

    m_Child[0] = 0;

    m_Child[1] = 0;

}





//

//   s e t T y p e

//



void tExPolicyExpr::setType( tType nType, tExPolicy nExPolicy )



{

    MTG_ASSERT( nType == xExPolicy );



    reset();



    m_nType = nType;

    m_nExPolicy = nExPolicy;

}





//

//   s e t T y p e

//



void tExPolicyExpr::setType( tType nType, tNumericalExpr *pCond,

    tExPolicyExpr* pChild )



{

    MTG_ASSERT( nType == xIfThen );



    reset();



    m_nType = nType;

    m_pCond = pCond;

    m_Child[0] = pChild;

}





//

//   s e t T y p e

//



void tExPolicyExpr::setType( tType nType, tNumericalExpr *pCond,

    tExPolicyExpr *pChild1, tExPolicyExpr *pChild2 )



{

    MTG_ASSERT( nType == xIfThenElse );



    reset();



    m_nType = nType;

    m_pCond = pCond;

    m_Child[0] = pChild1;

    m_Child[1] = pChild2;

}





//

//   p a r s e E x P o l i c y

//



tRetCode tExPolicyExpr::parseExPolicy( tParser& Parser, bool& bGo )



{

    tRetCode nRet;

    tExPolicy nExPolicy;



    bGo = false;

    switch( Parser.curToken() ) {

        case xTokDontExercise :

            nExPolicy = xDontExercise;

            break;



        case xTokMayExercise :

            nExPolicy = xMayExercise;

            break;



        case xTokForceExercise :

            nExPolicy = xForceExercise;

            break;



        default :

            bGo = true;

            break;

    }



    if( ! bGo ) {

        if( ( nRet = Parser.readToken() ) != OK )

            return nRet;

        setType( xExPolicy, nExPolicy );

    }



    return OK;

}





//

//   p a r s e I f T h e n E l s e

//



tRetCode tExPolicyExpr::parseIfThenElse( tParser& Parser )



{

    MTG_ASSERT( Parser.curToken() == xTokIf );



    tRetCode nRet;



    if( ( nRet = Parser.readToken() ) != OK )

        return OK;



    tNumericalExpr* pCond = new tNumericalExpr;

    tExPolicyExpr* pBr1 = new tExPolicyExpr;

    tExPolicyExpr* pBr2 = 0;



    if( ( nRet = pCond->parse1( Parser ) ) != OK ||

        ( nRet = Parser.swallowToken( xTokThen, "then" ) ) != OK ||

        ( nRet = pBr1->parse1( Parser ) ) != OK ) {

        goto error;

    }



    switch( Parser.curToken() ) {

        case xTokElse :

            pBr2 = new tExPolicyExpr;



            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = pBr2->parse1( Parser ) ) != OK ||

                ( nRet = Parser.swallowToken( xTokEndif,

                            "endif" ) ) != OK ) {

                goto error;

            }

            setType( xIfThenElse, pCond, pBr1, pBr2 );

            break;



        case xTokEndif :

            if( ( nRet = Parser.readToken() ) != OK )

                goto error;

            setType( xIfThen, pCond, pBr1 );

            break;



        default :

            nRet = Parser.setError( "token 'else' or 'endif' expected" );

            goto error;

    }



    return OK;



error:



    delete pCond;

    delete pBr1;

    if( pBr2 != 0 )

        delete pBr2;



    return nRet;

}





//

//   p a r s e 1

//



tRetCode tExPolicyExpr::parse1( tParser& Parser )



{

    tRetCode nRet;

    bool bGo;



    if( ( nRet = parseExPolicy( Parser, bGo ) ) != OK || ! bGo )

        return nRet;



    if( Parser.curToken() == xTokIf )

        return parseIfThenElse( Parser );



    return Parser.setError( "malformed expression" );

}





//

//   t E x P o l i c y E x p r

//



tExPolicyExpr::tExPolicyExpr()



{

    init();

}





//

//   t E x P o l i c y E x p r

//



tExPolicyExpr::tExPolicyExpr( tExPolicyExpr& Expr )



{

    init();

    copyFrom( Expr );

}





//

//   ~ t E x P o l i c y E x p r

//



tExPolicyExpr::~tExPolicyExpr()



{

    reset();

}





//

//   r e s e t

//



void tExPolicyExpr::reset()



{

    if( m_pCond != 0 ) {

        delete m_pCond;

        m_pCond = 0;

    }



    for( int k = 0; k < 2; ++k ) {

        if( m_Child[k] != 0 ) {

            delete m_Child[k];

            m_Child[k] = 0;

        }

    }



    m_nType = xExPolicy;

    m_nExPolicy = xDontExercise;

}







//

//   o p e r a t o r =

//



tExPolicyExpr& tExPolicyExpr::operator=( tExPolicyExpr& Expr )



{

    if( &Expr != this )

        copyFrom( Expr );

    return *this;

}





//

//   c o p y F r o m

//



void tExPolicyExpr::copyFrom( tExPolicyExpr &Expr )



{

    if( &Expr == this )

        return;



    reset();



    m_nType = Expr.m_nType;

    m_nExPolicy = Expr.m_nExPolicy;



    if( Expr.m_pCond != 0 )

        m_pCond = new tNumericalExpr( *Expr.m_pCond );



    for( int k = 0; k < 2; ++k ) {

        if( Expr.m_Child[k] != 0 )

            m_Child[k] = new tExPolicyExpr( *Expr.m_Child[k] );

    }

}





//

//   r e s o l v e

//



tRetCode tExPolicyExpr::resolve( int nThisId, tLookup<int>& Claim,

    tLookup<int>& Factor, tHeap<tFactor*>& FactorPtr, tPortfolio& Pf )



{

    tRetCode nRet;



    if( m_pCond != 0 ) {

        if( ( nRet = m_pCond->resolve( nThisId, Claim,

                        Factor, FactorPtr, Pf ) ) != OK ) {

            return nRet;

        }

    }



    for( int k = 0; k < 2 && m_Child[k] != 0; ++k ) {

        if( ( nRet = m_Child[k]->resolve( nThisId, Claim,

                        Factor, FactorPtr, Pf ) ) != OK ) {

            return nRet;

        }

    }  



    return OK;

}





//

//   g e t E v e n t s

//



void tExPolicyExpr::getEvents( tSeqEvGenerator& EvGen ) const



{

    if( m_pCond != 0 )

        m_pCond->getEvents( EvGen );



    for( int k = 0; k < 2 && m_Child[k] != 0; ++k )

        m_Child[k]->getEvents( EvGen );

}





//

//   g e t B a r r i e r s

//



void tExPolicyExpr::getBarriers( const tFactor* pFactor,

    tHeap<double>& Barrier ) const



{

    if( m_pCond != 0 )

        m_pCond->getBarriers( pFactor, Barrier );



    for( int k = 0; k < 2 && m_Child[k] != 0; ++k )

        m_Child[k]->getBarriers( pFactor, Barrier );

}





//

//   a p p l y

//



tExPolicy tExPolicyExpr::apply( tEngine& Engine,

    double gUnitValue, bool& bValid )



{

    bool bValid1;



        // bValid is set to false only if the missing branch of an

        // if..then..endif expression is triggered.



    bValid = true;



    if( m_nType == xExPolicy )

        return m_nExPolicy;



    MTG_ASSERT( m_pCond != 0 );

    double gCond = m_pCond->apply( Engine, gUnitValue, bValid1 );



    switch( m_nType ) {

        case xIfThen :

            if( gCond != 0 )

                return m_Child[0]->apply( Engine, gUnitValue, bValid );

            bValid = false;     // forced to use default value

            return xDontExercise;



        case xIfThenElse :

            if( gCond != 0 )

                return m_Child[0]->apply( Engine, gUnitValue, bValid );

            return m_Child[1]->apply( Engine, gUnitValue, bValid );



        default :

            throw INTERNAL_ERROR;

    }



    return xDontExercise;

}





//

//   p a r s e

//



tRetCode tExPolicyExpr::parse( tParser& Parser )



{

    tRetCode nRet;

    bool bGo;



        // First try the most basic expression without '{', '}'.



    if( ( nRet = parseExPolicy( Parser, bGo ) ) != OK || ! bGo )

        return nRet;



        // Now try obligatory '{', '}'.



    if( ( nRet = Parser.scanBeginOfObj() ) != OK ||

        ( nRet = parse1( Parser ) ) != OK ||

        ( nRet = Parser.scanEndOfObj() ) != OK ) {

        return nRet;

    }



    return OK;

}



MTG_END_NAMESPACE

