// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgGeoSpaceAxis.h"

#include "MtgGeoExplicit.h"

#include "MtgGeoImplicit.h"

#include "MtgPortfolio.h"



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tGeoSpaceAxis::init()



{

}





//

//   s p l i t A l i g n A t

//



void tGeoSpaceAxis::splitAlignAt( double gRoot, tHeap<double>& aAlignAt,

    tHeap<double>& aUpAlignAt, tHeap<double>& aDownAlignAt )



{

    for( int k = 0; k < aAlignAt.numOfElems(); ++k ) {

        if( aAlignAt[k] > gRoot ) {   

            aUpAlignAt.sortInAsc( aAlignAt[k], true );

        }

        else

            if( aAlignAt[k] < gRoot && aAlignAt[k] > 0 ) {

                aDownAlignAt.sortInDesc( aAlignAt[k], true );

            }

    }

}





//

//   r e g R e g i o n

//



void tGeoSpaceAxis::regRegion( double gSigmaBar, int nFromLevel,

    int nToLevel, bool bAligned, double gAlignAt )



{

    ++m_aRegion;

    tRegion& M = m_aRegion.last();



    M.m_gUpSigmaBar = gSigmaBar;

    M.m_gDownSigmaBar = gSigmaBar;

    M.m_nFromLevel = nFromLevel;

    M.m_nToLevel = nToLevel;

    M.m_bAligned = bAligned;

    M.m_gAlignAt = gAlignAt;

}





//

//   r e f i n e R e g i o n

//



tRetCode tGeoSpaceAxis::refineRegion( double S, double gAlignAt,

    double &gSigmaBar, double &gMaxDt, int &nSkip )



{

    double gMuBar = fabs( m_gMaxMu );

    if( fabs( m_gMinMu ) > gMuBar )

        gMuBar = fabs( m_gMinMu );



    if( gMuBar != 0 && m_gMinVol == 0 || m_gMaxVol == 0 )

        return INVALID_VOLATILITY;



    bool bIgnFactor = ( gAlignAt == S );

    bool bRecheck = true;



    double gNewSigmaBar;



    do{

        gNewSigmaBar = gSigmaBar;

        nSkip = 0;



        if( ! bIgnFactor ) {

            double dt2 = sqrt( gMaxDt );

            double gFrac = log( gAlignAt / S );



            if( gFrac > 0.0 ) {

                nSkip = (int) floor( gFrac / ( gNewSigmaBar * dt2 ) );

                if( nSkip == 0 )

                    bIgnFactor = true;

            }

            else {

                nSkip = (int) -floor( gFrac / ( -gNewSigmaBar * dt2 ) );

                if( nSkip == 0 )

                    bIgnFactor = true;

            }



            if( nSkip != 0 )

                gNewSigmaBar = gFrac / ( nSkip * dt2 );

        }



        bRecheck = false;



        if( isExplicit() ) {

                // Only if an explicit finite difference method

                // is used do we have to check for stability.



            double dt = 2 * m_gMinVol * m_gMinVol /

                ( gNewSigmaBar * ( m_gMaxVol * m_gMaxVol + 2 * gMuBar ) );

      

            MTG_ASSERT( gMuBar == 0.0 || m_gMinVol > 0.0 );

            MTG_ASSERT( dt >= 0.0 );



            if( dt != 0.0 )

                dt *= dt;

            else

                dt = 4.0 / ( gNewSigmaBar * gNewSigmaBar );



            dt *= 0.95;



            if( gMaxDt >= dt ) {

                double gNewMaxDt = dt;



                if( gNewMaxDt > 0 || bIgnFactor ) {

                    gMaxDt = gNewMaxDt;

                    if( ! bIgnFactor )

                        bRecheck = true;

                }

                else {

                    bIgnFactor = true;

                    bRecheck = true;

                }

            }

        }

    } while( bRecheck );



    gSigmaBar = gNewSigmaBar;

    return OK;

}





//

//   f i n i s h R e g i o n s

//



void tGeoSpaceAxis::finishRegions()



{

    for( int k = 0; k < m_aRegion.numOfElems(); ++k ) {

        double uu = m_aRegion[k].m_gUpSigmaBar * m_aRegion[k].m_gUpSigmaBar;

        double dd = m_aRegion[k].m_gDownSigmaBar * m_aRegion[k].m_gDownSigmaBar;

        double ud = m_aRegion[k].m_gUpSigmaBar * m_aRegion[k].m_gDownSigmaBar;



        if( isExplicit() ) {

            double dt2 = sqrt( m_gMaxDt );



            m_aRegion[k].m_gUpProbA =

                ( 1.0 - m_aRegion[k].m_gDownSigmaBar * dt2 / 2.0 ) / ( ud + uu );

            m_aRegion[k].m_gUpProbB =

                ( m_aRegion[k].m_gDownSigmaBar * dt2 ) / ( ud + uu );



            m_aRegion[k].m_gDownProbA =

                ( 1.0 + m_aRegion[k].m_gUpSigmaBar * dt2 / 2.0 ) / ( ud + dd );

            m_aRegion[k].m_gDownProbB =

                -( m_aRegion[k].m_gUpSigmaBar * dt2 ) / ( ud + dd );

        }

        else {

            m_aRegion[k].m_gD2 = dd;

            m_aRegion[k].m_gU2 = uu;

            m_aRegion[k].m_gD2U2 = dd - uu;

            m_aRegion[k].m_gDU =

                m_aRegion[k].m_gUpSigmaBar + m_aRegion[k].m_gDownSigmaBar;

            m_aRegion[k].m_gUD2DU2 = ud * m_aRegion[k].m_gDU;

        }

    }

}





//

//   s o r t R e g i o n s

//



void tGeoSpaceAxis::sortRegions()



{

    for( int k = 1; k < m_aRegion.numOfElems(); ++k ) {

        tRegion R = m_aRegion[k];

        int j = k;



        while( j > 0 && m_aRegion[j - 1].m_nFromLevel > R.m_nFromLevel ) {

            m_aRegion[j] = m_aRegion[j - 1];

            --j;

        }

        m_aRegion[j] = R;

    }



#if defined(_DEBUG)

    /*MTG_TRACE( "GeoSpaceAxis {\n" );

    for( MTG_FOR_INIT( int ) k = 0; k < m_aRegion.numOfElems(); ++k ) {

        if( k > 0 ) {

            MTG_ASSERT( m_aRegion[k].m_gDownSigmaBar ==

                        m_aRegion[k - 1].m_gUpSigmaBar );

        }

        MTG_TRACE( "    Region %d/%d: %lg up %lg down",

            m_aRegion[k].m_nFromLevel, m_aRegion[k].m_nToLevel,

            m_aRegion[k].m_gUpSigmaBar, m_aRegion[k].m_gDownSigmaBar );

        if( m_aRegion[k].m_bAligned )

            MTG_TRACE( ", aligned at %lg", m_aRegion[k].m_gAlignAt );

        MTG_TRACE( "\n" );

    }

    MTG_TRACE( "}\n" );*/

#endif

}





//

//   p r e p a r e

//



tRetCode tGeoSpaceAxis::prepare( tHeap<double>& aUpAlignAt,

    tHeap<double>& aDownAlignAt )



{

    tRetCode nRet;



    double gStdSigmaBar = m_gMaxVol * 1.41;



    m_aRegion.reset();

    m_aLevel.reset();



loop:



    int k = 0;

    int nUpLevel = 0;

    double S = m_gRoot;



    int nSkip;

    double gUpSigmaBar;

    

    do{

        double gAlignAt;



        if( k < aUpAlignAt.numOfElems() )

            gAlignAt = aUpAlignAt[k];

        else

            gAlignAt = S;



        double gSigmaBar = gStdSigmaBar;

        double gMaxDt = m_gMaxDt;



        if( ( nRet = refineRegion( S, gAlignAt, gSigmaBar,

                gMaxDt, nSkip ) ) != OK ) {

            return nRet;

        }



        if( m_aRegion.numOfElems() == 0 && k + 1 >= aUpAlignAt.numOfElems() ||

                nSkip != 0 ) {

            if( gMaxDt != m_gMaxDt ) {

                m_gMaxDt = gMaxDt;

                if( m_aRegion.numOfElems() > 0 ) {

                    m_aRegion.reset();

                    goto loop;

                }

            }



            if( m_aRegion.numOfElems() == 0 )

                regRegion( gSigmaBar, 0, 0, true, S );

            else

                m_aRegion.last().m_gUpSigmaBar = gSigmaBar;



            if( nSkip != 0 ) {

                if( nSkip > 1 )

                    regRegion( gSigmaBar, nUpLevel + 1, nUpLevel + nSkip - 1 );



                nUpLevel += nSkip;

                regRegion( gSigmaBar, nUpLevel, nUpLevel, true, gAlignAt );



                S = gAlignAt;

            }



            gUpSigmaBar = gSigmaBar;

        }

    } while( ++k < aUpAlignAt.numOfElems() );



    k = 0;

    S = m_gRoot;



    int nDownLevel = 0;

    double gDownSigmaBar = m_aRegion[0].m_gDownSigmaBar;



    for( k = 0; k < aDownAlignAt.numOfElems(); ++k ) {

        double gAlignAt = aDownAlignAt[k];

        double gSigmaBar = gStdSigmaBar;

        double gMaxDt = m_gMaxDt;



        if( ( nRet = refineRegion( S, gAlignAt, gSigmaBar,

                gMaxDt, nSkip ) ) != OK ) {

            return nRet;

        }



        if( nSkip != 0 ) {

            if( gMaxDt != m_gMaxDt ) {

                m_gMaxDt = gMaxDt;

                m_aRegion.reset();

                goto loop;

            }



            if( nDownLevel == 0 ) {

                m_aRegion[0].m_gDownSigmaBar = gSigmaBar;

                if( nUpLevel == 0 ) {

                    m_aRegion[0].m_gUpSigmaBar = gSigmaBar;

                    gUpSigmaBar = gSigmaBar;

                }

            }

            else {

                m_aRegion.last().m_gDownSigmaBar = gSigmaBar;

            }



            if( nSkip < -1 )

                regRegion( gSigmaBar, nDownLevel - 1, nDownLevel + nSkip + 1 );



            nDownLevel += nSkip;

            regRegion( gSigmaBar, nDownLevel, nDownLevel, true, gAlignAt );



            gDownSigmaBar = gSigmaBar;

    

            S = gAlignAt;

        }

    }



    regRegion( gUpSigmaBar, nUpLevel + 1, INT_MAX / 2 );

    regRegion( gDownSigmaBar, nDownLevel - 1, INT_MIN / 2 );



    finishRegions();

    sortRegions();



    m_gStableDt = m_gMaxDt;



    return OK;

}





//

//   f i n a l i z e

//



void tGeoSpaceAxis::finalize( int nNumOfUpLevels, int nNumOfDownLevels )



{

    MTG_ASSERT( nNumOfUpLevels >= 0 && nNumOfDownLevels >= 0 );



    m_nNumOfUpLevels = nNumOfUpLevels + 1;

    m_nNumOfDownLevels = nNumOfDownLevels + 1;

    m_nRootLevel = m_nNumOfDownLevels;



    m_aLevel.numOfElems( m_nNumOfUpLevels + m_nNumOfDownLevels + 1 );



    for( int k = 0; k < m_aRegion.numOfElems(); ++k ) {

        int j1 = m_nRootLevel + m_aRegion[k].m_nFromLevel;

        int j2 = m_nRootLevel + m_aRegion[k].m_nToLevel;



        if( j1 > j2 ) {

            int t = j1; j1 = j2; j2 = t;

        }



        if( j1 < 0 )

            j1 = 0;

        if( j2 >= m_aLevel.numOfElems() )

            j2 = m_aLevel.numOfElems() - 1;



        for( int j = j1; j <= j2; ++j )

            m_aLevel[j].m_nRegion = k;

    }



    double dt2 = sqrt( m_gMaxDt );

    double f;



    for( int j = m_nRootLevel; j < m_aLevel.numOfElems(); ++j ) {

        int i = m_aLevel[j].m_nRegion;

        if( m_aRegion[i].m_bAligned ) {

            m_aLevel[j].m_gFactor = m_aRegion[i].m_gAlignAt;

            f = exp( m_aRegion[i].m_gUpSigmaBar * dt2 );

        }

        else {

            m_aLevel[j].m_gFactor = f * m_aLevel[j - 1].m_gFactor;

        }

    }



    for( MTG_FOR_INIT( int ) j = m_nRootLevel; j >= 0; --j ) {

        int i = m_aLevel[j].m_nRegion;

        if( m_aRegion[i].m_bAligned ) {

            m_aLevel[j].m_gFactor = m_aRegion[i].m_gAlignAt;

            f = exp( -m_aRegion[i].m_gDownSigmaBar * dt2 );

        }

        else {

            m_aLevel[j].m_gFactor = f * m_aLevel[j + 1].m_gFactor;

        }

    }



    for( MTG_FOR_INIT( int ) j = 1; j < m_aLevel.numOfElems() - 1; ++j ) {

        double hu = m_aLevel[j + 1].m_gFactor - m_aLevel[j].m_gFactor;

        double hd = m_aLevel[j].m_gFactor - m_aLevel[j - 1].m_gFactor;

        double hm = m_aLevel[j + 1].m_gFactor - m_aLevel[j - 1].m_gFactor;



        m_aLevel[j].m_gUpDeltaMod = hd / ( hu * hm );

        m_aLevel[j].m_gDownDeltaMod = hu / ( hd * hm );



        m_aLevel[j].m_gUpGammaMod = 2 / ( hu * hm );

        m_aLevel[j].m_gDownGammaMod = -2 / ( hd * hm );

    }



#if defined(_DEBUG)

    //for( MTG_FOR_INIT( int ) k = 0; k < m_aLevel.numOfElems(); ++k ) {

    //    MTG_TRACE( "Level %d at %.20lg, region %d\n",

    //        k - m_nRootLevel, m_aLevel[k].m_gFactor, m_aLevel[k].m_nRegion );

    //}

#endif

}





//

//   t G e o S p a c e A x i s

//



tGeoSpaceAxis::tGeoSpaceAxis()



{

    init();

}





//

//   t G e o S p a c e A x i s

//



tGeoSpaceAxis::tGeoSpaceAxis( const tGeoSpaceAxis& Axis )



{

    init();

    copyFrom( Axis );

}





//

//   ~ t G e o S p a c e A x i s

//



tGeoSpaceAxis::~tGeoSpaceAxis()



{

}





//

//   o p e r a t o r =

//



tGeoSpaceAxis& tGeoSpaceAxis::operator=( const tGeoSpaceAxis& Axis )



{

    if( this != &Axis )

        copyFrom( Axis );

    return *this;

}





//

//   c o p y F r o m

//



void tGeoSpaceAxis::copyFrom( const tGeoSpaceAxis& Axis )



{

    if( this == &Axis )

        return;



    m_aRegion = Axis.m_aRegion;

    m_aLevel = Axis.m_aLevel;

    

    m_gRoot = Axis.m_gRoot;

    m_gMinMu = Axis.m_gMinMu;

    m_gMaxMu = Axis.m_gMaxMu;

    m_gMinVol = Axis.m_gMinVol;

    m_gMaxVol = Axis.m_gMaxVol;

    m_gMaxDt = Axis.m_gMaxDt;



    m_nNumOfUpLevels = Axis.m_nNumOfUpLevels;

    m_nNumOfDownLevels = Axis.m_nNumOfDownLevels;

    m_nRootLevel = Axis.m_nRootLevel;



    super::copyFrom( Axis );  // Super copies m_bFinalized.

}





//

//   c l o n e

//



tSpaceAxis* tGeoSpaceAxis::clone() const



{

    return new tGeoSpaceAxis( *this );

}





//

//   p r e p a r e

//



tRetCode tGeoSpaceAxis::prepare( tFDMethod nMethod, double gRoot,

    double gMinVol, double gMaxVol, double gMinMu, double gMaxMu,

    double gMaxDt )



{

    tHeap<double> Empty;



    MTG_ASSERT( gRoot > 0 );

    MTG_ASSERT( gMinVol >= 0 && gMaxVol >= gMinVol );



    tHeap<double> aUpAlignAt, aDownAlignAt;

        

    setMethod( nMethod );



    m_gRoot = gRoot;

    m_gMinMu = gMinMu;

    m_gMaxMu = gMaxMu;

    m_gMinVol = gMinVol;

    m_gMaxVol = gMaxVol;

    m_gMaxDt = gMaxDt;



    return prepare( Empty, Empty );

}





//

//   p r e p a r e

//



tRetCode tGeoSpaceAxis::prepare( tFDMethod nMethod, double gRoot,

    double gMinVol, double gMaxVol, double gMinMu, double gMaxMu,

    double gMaxDt, tHeap<double>& aAlignAt )



{

    MTG_ASSERT( gRoot > 0 );

    MTG_ASSERT( gMinVol >= 0 && gMaxVol >= gMinVol );



    tHeap<double> aUpAlignAt, aDownAlignAt;

        

    setMethod( nMethod );



    m_gRoot = gRoot;

    m_gMinMu = gMinMu;

    m_gMaxMu = gMaxMu;

    m_gMinVol = gMinVol;

    m_gMaxVol = gMaxVol;

    m_gMaxDt = gMaxDt;



        // Building the Partition must proceed from the root upwards

        // and downwards in two sweeps, because the root must be aligned

        // in all cases, whereas other barriers may be missed.



    splitAlignAt( gRoot, aAlignAt, aUpAlignAt, aDownAlignAt );

    return prepare( aUpAlignAt, aDownAlignAt );

}





//

//   a d j u s t S t a b l e D t

//



void tGeoSpaceAxis::adjustStableDt( double gOrgDt, double gDayDt,

    double gDayUnit )



{

    if( m_gMaxDt == gOrgDt )

        m_gStableDt = gDayDt;

    else

        m_gStableDt = m_gMaxDt / gDayUnit;

}





//

//   c r e a t e O F S o l v e r

//



tOFSolver* tGeoSpaceAxis::createOFSolver()



{

    MTG_ASSERT( isFinalized() );



    if( isImplicit() )

        return new tGeoImplicit( this );

    if( isExplicit() )

        return new tGeoExplicit( this );



    throw new tException( INTERNAL_ERROR );

    return 0;

}





//

//   c a l c P r o b

//



void tGeoSpaceAxis::calcProb( int nLevel, double gVol, double gDrift,

    double gRelDuration, double& gPU, double& gPD ) const



{

    MTG_ASSERT( isExplicit() && isFinalized() );



    tRegion& R = m_aRegion[m_aLevel[m_nRootLevel + nLevel].m_nRegion];



    gVol *= gVol;



    gPU = R.m_gUpProbA * gVol + R.m_gUpProbB * gDrift;

    gPD = R.m_gDownProbA * gVol + R.m_gDownProbB * gDrift;



    gPU *= gRelDuration;

    gPD *= gRelDuration;

}





//

//   c a l c D e l t a

//



double tGeoSpaceAxis::calcDelta( int nLevel, double gVU, double gVM,

    double gVD ) const



{

    MTG_ASSERT( nLevel + m_nRootLevel > 0 &&

                nLevel + m_nRootLevel < m_aLevel.numOfElems() - 1 );



    tLevel& L = m_aLevel[m_nRootLevel + nLevel];



    return ( gVU - gVM ) * L.m_gUpDeltaMod +

           ( gVM - gVD ) * L.m_gDownDeltaMod;

}





//

//   c a l c G a m m a

//



double tGeoSpaceAxis::calcGamma( int nLevel, double gVU, double gVM,

    double gVD ) const



{

    MTG_ASSERT( nLevel + m_nRootLevel > 0 &&

                nLevel + m_nRootLevel < m_aLevel.numOfElems() - 1 );



    tLevel& L = m_aLevel[m_nRootLevel + nLevel];



    return ( gVU - gVM ) * L.m_gUpGammaMod +

           ( gVM - gVD ) * L.m_gDownGammaMod;

}



MTG_END_NAMESPACE

