// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#if ! defined(_MTG_GEO_SPACE_AXIS_)

#define _MTG_GEO_SPACE_AXIS_



#include "MtgHeap.h"

#include "MtgSpaceAxis.h"

#include "MtgOFImplicit.h"



// A axis of space for the process dS = S(\mu dt + \sigma dW),

// aligned to barriers, if necessary.



MTG_BEGIN_NAMESPACE



class tPortfolio;





//

//   t G e o S p a c e A x i s

//



class tGeoSpaceAxis : public tSpaceAxis {



    typedef tSpaceAxis super;



    struct tRegion {

        int m_nFromLevel;

        int m_nToLevel;

        double m_gUpSigmaBar;

        double m_gDownSigmaBar;

        bool m_bAligned;

        double m_gAlignAt;



            // For explicit finite difference methods:



        double m_gUpProbA;

        double m_gUpProbB;

        double m_gDownProbA;

        double m_gDownProbB;



            // For implicit finite difference methods:



        double m_gD2;       // m_gDownSigmaBar^2

        double m_gU2;       // m_gUpSigmaBar^2

        double m_gD2U2;     // m_gDownSigmaBar^2 - m_gUpSigmaBar^2

        double m_gDU;       // m_gDownSigmaBar + m_gUpSigmaBar

        double m_gUD2DU2;   // m_gUpSigmaBar m_gDownSigmaBar^2 +

                            // m_gDownSigmaBar m_gUpSigmaBar^2

    };



    struct tLevel {

        double m_gFactor;

        double m_gUpDeltaMod;   

        double m_gDownDeltaMod;                                 

        double m_gUpGammaMod;   

        double m_gDownGammaMod;                                 

        int m_nRegion;

    };



private:



    double m_gMinMu;

    double m_gMaxMu;

    double m_gMinVol;

    double m_gMaxVol;



    double m_gRoot;



    double m_gMaxDt;    // scale: day or year



    tHeap<tRegion> m_aRegion;

    tHeap<tLevel> m_aLevel;



    int m_nNumOfUpLevels;

    int m_nNumOfDownLevels;

    int m_nRootLevel;



    void init();



    void splitAlignAt( double gRoot, tHeap<double>& aAlignAt,

        tHeap<double>& aUpAlignAt, tHeap<double>& aDownAlignAt );

    

    void regRegion( double gSigmaBar, int nFromLevel,

        int nToLevel, bool bAligned = false, double gAlignAt = 0 );



    tRetCode refineRegion( double S, double gAlignAt, double &gSigmaBar,

        double &gMaxDt, int &nSkip );



    void finishRegions();

    void sortRegions();



    tRetCode prepare( tHeap<double>& aUpAlignAt,

        tHeap<double>& aDownAlignAt );



    void finalize( int nNumOfUpLevels, int nNumOfDownLevels );



    tRegion& region( int nLevel ) {

        return m_aRegion[m_aLevel[m_nRootLevel + nLevel].m_nRegion];

    }



    friend class tGeoImplicit;

    friend class tGeoExplicit;



public:



    tGeoSpaceAxis();

    tGeoSpaceAxis( const tGeoSpaceAxis& Axis );



    ~tGeoSpaceAxis();



    tGeoSpaceAxis& operator=( const tGeoSpaceAxis& Axis );

    void copyFrom( const tGeoSpaceAxis& Axis );



    tSpaceAxis* clone() const;



        // All these parameters are either day-based or year-based!

        // However, if they are year-based, adjustStableDt() has to be

        // called after prepare() has returned, in order to adjust

        // the scale of m_gStableDt, which has to be day-based.



    tRetCode prepare( tFDMethod nMethod, double gRoot, double gMinVol,

        double gMaxVol, double gMinMu, double gMaxMu, double gMaxDt );



    tRetCode prepare( tFDMethod nMethod, double gRoot, double gMinVol,

        double gMaxVol, double gMinMu, double gMaxMu, double gMaxDt,

        tHeap<double>& aAlignAt );



        // gOrgDt is the original gMaxDt parameter passed to prepare().

        // gDayDt is the corresponding day-based value (could be the

        // same). If m_gStableDt != gOrgDt, gDayUnit is used for

        // conversion (this scheme avoids unnecessary arithmetic).



    void adjustStableDt( double gOrgDt, double gDayDt, double gDayUnit );



    tOFSolver* createOFSolver();



    double factor( int nLevel ) const {

        return m_aLevel[nLevel + m_nRootLevel].m_gFactor;

    }



    void calcProb( int nLevel, double gVol, double gDrift,

        double gRelDuration, double& gPU, double& gPD ) const;



        // A finite difference approximation of gamma:



    double calcDelta( int nLevel, double gVU, double gVM, double gVD ) const;



        // A finite difference approximation of gamma:



    double calcGamma( int nLevel, double gVU, double gVM, double gVD ) const;

};



MTG_END_NAMESPACE



#endif

