// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgHJMTermStruct.h"

#include "MtgInterestSpline.h"

#include "MtgStepDrift.h"



MTG_BEGIN_NAMESPACE





//

//   p r o g r e s s

//



tRetCode tHJMTermStruct::tEvolutionStub::progress( 

    int nStep, int nNumOfSamples, const tHeap<tSample>& Initial,

    const tHeap<tSample>& Last, tHeap<tSample>& Future )



{

    for( int i = nStep; i < nNumOfSamples; ++i )

        Future[i] = Last[i];

    return OK;

}





//

//   i n i t

//



void tHJMTermStruct::init()



{

    m_nNumOfSamples = 0;

    m_nMaxStart = -1;

    m_bIsFinalized = false;

}





//

//   a l l o c S p a c e

//



void tHJMTermStruct::allocSpace( int nNumOfSamples )



{

    m_nNumOfSamples = nNumOfSamples;



    m_Initial.numOfElems( nNumOfSamples );

    m_Forward.numOfElems( nNumOfSamples );

    m_Discount.numOfElems( nNumOfSamples );

}





//

//   a l l o c S p a c e

//



void tHJMTermStruct::allocSpace( const tDrift& Drift )



{

    MTG_ASSERT( compatibleDateBaseWith( Drift ) );



    int nNumOfSamples = Drift.maturity();



    if( scale() == xYear && Drift.numOfPeriods() != numOfPeriods() ) {

        double f = (double) numOfPeriods() / Drift.numOfPeriods();

        nNumOfSamples = (int) ceil( nNumOfSamples * f );

    }



    allocSpace( nNumOfSamples  );

}





//

//   l o a d D r i f t

//



tRetCode tHJMTermStruct::loadDrift( const tDrift& Drift, int nNumOfSamples )



{

    tRetCode nRet;

    tStepDrift NewDrift;

    const tDrift* pDrift;



    if( sameDateBaseAs( Drift ) ) {

        pDrift = &Drift;

    }

    else {

            // need to convert first

        static_cast<tDateBase&>( NewDrift ) = *this;



        if( ( nRet = NewDrift.addDrift( Drift ) ) != OK ||

            ( nRet = NewDrift.finalize() ) != OK ) {

            return nRet;

        }

        pDrift = &NewDrift;

    }



    if( nNumOfSamples <= 0 )

        allocSpace( *pDrift );

    else

        allocSpace( nNumOfSamples );



        // load initial forward rate curve one-to-one

    for( int i = 0; i < m_nNumOfSamples; ++i )

        m_Initial[i] = (tSample) pDrift->forwardSI( i );



    return OK;

}





//

//   c o p y F r o m

//



void tHJMTermStruct::copyFrom( const tDrift& Drift )



{

    tRetCode nRet;



        // inherit date base

    super::copyFrom( Drift );



    m_bIsFinalized = false;

    if( ( nRet = loadDrift( Drift, -1 ) ) != OK )

        throw tException( nRet );



    // evolution is missing

}





//

//   c o p y F r o m

//



void tHJMTermStruct::copyFrom( const tHJMTermStruct& HJM )



{

    if( this == &HJM )

        return;



    m_nNumOfSamples = HJM.m_nNumOfSamples;

    m_nMaxStart = HJM.m_nMaxStart;



    m_Initial = HJM.m_Initial;

    m_Forward = HJM.m_Forward;

    m_Discount = HJM.m_Discount;



    m_bIsFinalized = HJM.m_bIsFinalized;



    super::copyFrom( HJM );

}





//

//   t H J M T e r m S t r u c t

//



tHJMTermStruct::tHJMTermStruct()



{

    init();

}





//

//   t H J M T e r m S t r u c t

//



tHJMTermStruct::tHJMTermStruct( const tDrift& Drift )



{

    init();

    copyFrom( Drift );

}





//

//   t H J M T e r m S t r u c t

//



tHJMTermStruct::tHJMTermStruct( tScale nScale )

    : tDateBase( nScale )



{

    init();

}





//

//   t H J M T e r m S t r u c t

//



tHJMTermStruct::tHJMTermStruct( tScale nScale, int nNumOfPeriods )

    : tDateBase( nScale, nNumOfPeriods )



{

    init();

}





//

//   t H J M T e r m S t r u c t

//



tHJMTermStruct::tHJMTermStruct( tScale nScale, const tDayCount& DayCount )

    : tDateBase( nScale, DayCount )



{

    init();

}





//

//   t H J M T e r m S t r u c t

//



tHJMTermStruct::tHJMTermStruct( tScale nScale, int nNumOfPeriods,

    const tDayCount& DayCount )

    : tDateBase( nScale, nNumOfPeriods, DayCount )



{

    init();

}





//

//   t H J M T e r m S t r u c t

//



tHJMTermStruct::tHJMTermStruct( tScale nScale, const tDayCount& DayCount, tDate Base )

    : tDateBase( nScale, DayCount, Base )



{

    init();

}





//

//   t H J M T e r m S t r u c t

//



tHJMTermStruct::tHJMTermStruct( tScale nScale, int nNumOfPeriods,

    const tDayCount& DayCount, tDate Base )

    : tDateBase( nScale, nNumOfPeriods, DayCount, Base )



{

    init();

}





//

//   t H J M T e r m S t r u c t

//



tHJMTermStruct::tHJMTermStruct( const tHJMTermStruct& HJM )



{

    init();

    copyFrom( HJM );

}





//

//   ~ t H J M T e r m S t r u c t

//



tHJMTermStruct::~tHJMTermStruct()



{

}





//

//   o p e r a t o r =

//



tHJMTermStruct& tHJMTermStruct::operator=( const tHJMTermStruct& HJM )



{

    if( this != &HJM )

        copyFrom( HJM );

    return *this;

}





//

//   o p e r a t o r =

//



tHJMTermStruct& tHJMTermStruct::operator=( const tDrift& Drift )



{

    copyFrom( Drift );

    return *this;

}





//

//   l o a d I n i t i a l

//



tRetCode tHJMTermStruct::loadInitial( const tDrift& Drift,

    int nNumOfSamples )



{

    tRetCode nRet;



    m_bIsFinalized = false;

    if( ( nRet = loadDrift( Drift, nNumOfSamples ) ) != OK )

        return nRet;

    return OK;

}





//

//   l o a d I n i t i a l

//



tRetCode tHJMTermStruct::loadInitial( tInterestSpline& Spline,

    int nNumOfSamples )



{

    tRetCode nRet;



        // create temporary tStepDrift object:

    tStepDrift Drift( scale(), numOfPeriods(), dayCount(), base() );

        // load drift object with interest spline:

    

    if( ( nRet = Drift.addSpline( Spline ) ) != OK ||

        ( nRet = Drift.finalize() ) != OK ) {

        return nRet;

    }



        // and use the result:

    return loadInitial( Drift, nNumOfSamples );

}





//

//   l o a d I n i t i a l

//



tRetCode tHJMTermStruct::loadInitial( double gConstInitial,

    int nNumOfSamples )



{

    MTG_ASSERT( nNumOfSamples > 0 );



    m_bIsFinalized = false;

    allocSpace( nNumOfSamples );



    for( int i = 0; i < m_nNumOfSamples; ++i )

        m_Initial[i] = (tSample) gConstInitial;



    return OK;

}





//

//   l i m i t E v o l u t i o n

//



void tHJMTermStruct::limitEvolution( int nMaxStart )



{

        // Is there a restriction currently?

    if( m_nMaxStart >= 0 && m_nMaxStart < m_nNumOfSamples - 1 ) {

            // Is the new restriction less restrictive?

        if( nMaxStart < 0 || nMaxStart > m_nMaxStart )

            m_bIsFinalized = false;

    }

    m_nMaxStart = nMaxStart;

}





//

//   f i n a l i z e

//



tRetCode tHJMTermStruct::finalize( tEvolutionStub& Evolution, int nMaxStart )



{

    limitEvolution( nMaxStart );

    return finalize( Evolution );

}





//

//   f i n a l i z e

//



tRetCode tHJMTermStruct::finalize( tEvolutionStub& Evolution )



{

    if( m_nNumOfSamples <= 0 ) {

        m_bIsFinalized = true;

        return OK;

    }



    int m = m_nMaxStart < 0 ? m_nNumOfSamples - 1 : m_nMaxStart;



    if( m >= m_nNumOfSamples )

        m = m_nNumOfSamples - 1;



        // First propagate the initial forwardForward rate curve over

        // time, by calling evolution() for each row.



    tHeap<tSample> Row0, Row1;



    Evolution.init( *this, m_nNumOfSamples, m_Initial );



    for( int k = 1; k <= m; ++k ) {

        tRetCode nRet;



        if( k == 1 ) {

            m_Discount.getFullRow( k, Row1 );

            nRet = Evolution.progress( k, m_nNumOfSamples,

                m_Initial, m_Initial, Row1 );

        }

        else {

            m_Discount.getFullRow( k - 1, Row0 );

            m_Discount.getFullRow( k, Row1 );

            nRet = Evolution.progress( k, m_nNumOfSamples,

                m_Initial, Row0, Row1 );

        }



        if( nRet != OK )

            return nRet;

    }



    Evolution.cleanup();



#if defined(_DEBUG)

    /*

    MTG_TRACE( "Forward:\nRow 0:" );

    for( int l = 0; l < m_nNumOfSamples; ++l )

        MTG_TRACE( " % .3lf", (double) m_Initial[l] );



    MTG_TRACE( "\n" );

    for( MTG_FOR_INIT( int ) l = 1; l < m_nNumOfSamples; ++l ) {

        MTG_TRACE( "Row %d:", l );

        for( int j = 0; j < l; ++j )

            MTG_TRACE( "       " );

        for( MTG_FOR_INIT( int ) j = 0; j < m_nNumOfSamples - l; ++j )

            MTG_TRACE( " % .3lf", (double) m_Discount[l][j] );

        MTG_TRACE( "\n" );

    }

    */

#endif



        // Now compute the cumulative accrual and discount factors.



    m_Forward[0] = m_Initial[0];

    for( int i = 1; i <= m; ++i )

        m_Forward[i] = m_Forward[i - 1] + m_Discount[i][0];



    m_Discount[0][0] = m_Initial[0];

    for( MTG_FOR_INIT( int ) i = 1; i < m_nNumOfSamples; ++i )

        m_Discount[0][i] = m_Discount[0][i - 1] + m_Initial[i];



    for( MTG_FOR_INIT( int ) k = 1; k <= m; ++k ) {

        tSample* F = m_Discount[k] - k;



        for( int i = k + 1; i < m_nNumOfSamples; ++i )

            F[i] += F[i - 1];

    }



#if defined(_DEBUG)

/*

    MTG_TRACE( "Discount:\n" );

    for( MTG_FOR_INIT( int ) i = 0; i < m_nNumOfSamples; ++i ) {

        MTG_TRACE( "Row %d:", i );

        for( int j = 0; j < i; ++j )

            MTG_TRACE( "       " );

        for( MTG_FOR_INIT( int ) j = 0; j < m_nNumOfSamples - i; ++j ) {

            MTG_TRACE( " % .3lf", (double) m_Discount[i][j] );

            MTG_ASSERT( m_Discount[i][j] == D( i, i + j ) );

        }

        MTG_TRACE( "\n" );

    }



    MTG_TRACE( "Acc forward:" );

    for( MTG_FOR_INIT( int ) i = 1; i < m_nNumOfSamples; ++i )

        MTG_TRACE( " % .3lf", (double) m_Forward[i] );

    MTG_TRACE( "\n" );

*/                        

#endif



    m_bIsFinalized = true;

    return OK;

}





//

//   i s C o v e r e d S I

//



bool tHJMTermStruct::isCoveredSI( int nSub ) const



{

    return nSub >= 0 && nSub < m_nNumOfSamples;

}





//

//   i s C o v e r e d S I

//



bool tHJMTermStruct::isCoveredSI( double gSub ) const



{

    return gSub >= 0 && gSub < m_nNumOfSamples;

}





//

//   i s C o v e r e d T U

//



bool tHJMTermStruct::isCoveredTU( int nUnit ) const



{

    return isCoveredSI( nUnit * numOfPeriods() );

}





//

//   i s C o v e r e d T U

//



bool tHJMTermStruct::isCoveredTU( double gUnit ) const



{

    return isCoveredSI( gUnit * numOfPeriods() );

}





//

//   i s C o v e r e d

//



bool tHJMTermStruct::isCovered( tDate Date ) const



{

    if( scale() == xDay ) {

        int n;



        getSI( Date, n );

        return isCoveredSI( n );

    }



    return isCoveredSI( getSI( Date ) );

}





//

//   f o r w a r d T U

//



double tHJMTermStruct::forwardTU( int nFromUnit, int nToUnit ) const



{

    if( scale() == xDay )

        return forwardSI( nFromUnit, nToUnit );

    return forwardSI( nFromUnit * numOfPeriods(), nToUnit * numOfPeriods() );

}





//

//   f o r w a r d T U

//



double tHJMTermStruct::forwardTU( double gFromUnit, double gToUnit ) const



{

    if( scale() == xDay )

        return forwardSI( gFromUnit, gToUnit );

    return forwardSI( gFromUnit * numOfPeriods(), gToUnit * numOfPeriods() );

}





//

//   f o r w a r d F o r w a r d T U

//



double tHJMTermStruct::forwardForwardTU( int n ) const



{

    if( scale() == xDay )

        return forwardForwardSI( n );

    return forwardForwardSI( n * numOfPeriods() );

}





//

//   f o r w a r d F o r w a r d T U

//



double tHJMTermStruct::forwardForwardTU( int n, int nUnit ) const



{

    if( scale() == xDay )

        return forwardForwardSI( n, nUnit );

    return forwardForwardSI( n * numOfPeriods(), nUnit * numOfPeriods() );

}





//

//   f o r w a r d F o r w a r d T U

//



double tHJMTermStruct::forwardForwardTU( int n,

    int nFromUnit, int nToUnit ) const



{

    if( scale() == xDay )

        return forwardForwardSI( n, nFromUnit, nToUnit );

    return forwardForwardSI( n * numOfPeriods(),

        nFromUnit * numOfPeriods(), nToUnit * numOfPeriods() );

}





//

//   f o r w a r d F o r w a r d T U

//



double tHJMTermStruct::forwardForwardTU( int n, double gUnit ) const



{

    if( scale() == xDay )

        return forwardForwardSI( n, gUnit );

    return forwardForwardSI( n * numOfPeriods(), gUnit * numOfPeriods() );

}





//

//   f o r w a r d F o r w a r d T U

//



double tHJMTermStruct::forwardForwardTU( int n,

    double gFromUnit, double gToUnit ) const



{

    if( scale() == xDay )

        return forwardForwardSI( n, gFromUnit, gToUnit );

    return forwardForwardSI( n * numOfPeriods(),

        gFromUnit * numOfPeriods(), gToUnit * numOfPeriods() );

}





//

//   f o r w a r d F o r w a r d T U

//



double tHJMTermStruct::forwardForwardTU( double g ) const



{

    if( scale() == xDay )

        return forwardForwardSI( g );

    return forwardForwardSI( g * numOfPeriods() );

}





//

//   f o r w a r d F o r w a r d T U

//



double tHJMTermStruct::forwardForwardTU( double g, int nUnit ) const



{

    if( scale() == xDay )

        return forwardForwardSI( g, nUnit );

    return forwardForwardSI( g * numOfPeriods(), nUnit * numOfPeriods() );

}





//

//   f o r w a r d F o r w a r d T U

//



double tHJMTermStruct::forwardForwardTU( double g,

    int nFromUnit, int nToUnit ) const



{

    if( scale() == xDay )

        return forwardForwardSI( g, nFromUnit, nToUnit );

    return forwardForwardSI( g * numOfPeriods(),

        nFromUnit * numOfPeriods(), nToUnit * numOfPeriods() );

}





//

//   f o r w a r d F o r w a r d T U

//



double tHJMTermStruct::forwardForwardTU( double g, double gUnit ) const



{

    if( scale() == xDay )

        return forwardForwardSI( g, gUnit );

    return forwardForwardSI( g * numOfPeriods(), gUnit * numOfPeriods() );

}





//

//   f o r w a r d F o r w a r d T U

//



double tHJMTermStruct::forwardForwardTU( double g,

    double gFromUnit, double gToUnit ) const



{

    if( scale() == xDay )

        return forwardForwardSI( g, gFromUnit, gToUnit );

    return forwardForwardSI( g * numOfPeriods(),

        gFromUnit * numOfPeriods(), gToUnit * numOfPeriods() );

}





//

//   f o r w a r d S I

//



double tHJMTermStruct::forwardSI( int nFromSub, int nToSub ) const



{

    MTG_ASSERT( m_bIsFinalized );

    MTG_ASSERT( nFromSub <= nToSub );



    if( nFromSub == nToSub )

        return forwardForwardSI( nFromSub );

    if( nFromSub == 0 ) 

        return m_Forward[nToSub - 1] / nToSub;

    return ( m_Forward[nToSub - 1] - m_Forward[nFromSub - 1] ) / 

                ( nToSub - nFromSub );

}





//

//   f o r w a r d S I

//



double tHJMTermStruct::forwardSI( double gFromSub, double gToSub ) const



{

    MTG_ASSERT( m_bIsFinalized );

    MTG_ASSERT( gFromSub <= gToSub );



    if( gFromSub == gToSub )

        return forwardForwardSI( gFromSub );



    int nCeilFrom = (int) ceil( gFromSub );

    int nFloorTo = (int) floor( gToSub );



    if( nCeilFrom > nFloorTo ) {

            // interval is within the same sub interval

        return forwardForwardSI( gFromSub );

    }



    double gRate = 0;



    if( nCeilFrom < nFloorTo ) {

            // interval spans several sub intervals

        if( nCeilFrom == 0 )

            gRate += m_Forward[nFloorTo - 1];

        else

            gRate += m_Forward[nFloorTo - 1] - m_Forward[nCeilFrom - 1];

    }



        // now look at head and tail

    if( nCeilFrom > gFromSub )

        gRate += ( nCeilFrom - gFromSub ) * forwardForwardSI( gFromSub );

    if( nFloorTo < gToSub )

        gRate += ( gToSub - nFloorTo ) * forwardForwardSI( nFloorTo );



    return gRate / ( gToSub - gFromSub );

}





//

//   f o r w a r d F o r w a r d S I

//



double tHJMTermStruct::forwardForwardSI( int n ) const



{

    MTG_ASSERT( m_bIsFinalized );



    if( m_nNumOfSamples == 0 )

        return 0;

    if( n >= m_nNumOfSamples )

        return D( m_nNumOfSamples - 1 );

    if( n <= 0 ) 

        return D( 0 );

    return D( n );

}





//

//   f o r w a r d F o r w a r d S I

//



double tHJMTermStruct::forwardForwardSI( int n, int nSub ) const



{

    MTG_ASSERT( m_bIsFinalized );

    MTG_ASSERT( n <= nSub );



    if( m_nNumOfSamples == 0 )

        return 0;



    if( nSub >= m_nNumOfSamples ) {

        if( n >= m_nNumOfSamples )

            n = m_nNumOfSamples - 1;

        nSub = m_nNumOfSamples - 1;

    }



    if( n == nSub )

        return forwardForwardSI( n );            // simply a lookup



    return D( n, nSub ) - D( n, nSub - 1 );

}





//

//   f o r w a r d F o r w a r d S I

//



double tHJMTermStruct::forwardForwardSI( int n,

    int nFromSub, int nToSub ) const



{

    MTG_ASSERT( m_bIsFinalized );

    MTG_ASSERT( n <= nFromSub && nFromSub <= nToSub );



    if( nFromSub == nToSub )

        return forwardForwardSI( n, nFromSub );

    if( nFromSub == n )

        return D( n, nToSub - 1 ) / ( nToSub - nFromSub );

    return ( D( n, nToSub - 1 ) - D( n, nFromSub - 1 ) ) / 

                ( nToSub - nFromSub );

}





//

//   f o r w a r d F o r w a r d S I

//



double tHJMTermStruct::forwardForwardSI( int n, double gSub ) const



{

    MTG_ASSERT( n <= gSub );



    return forwardForwardSI( n, (int) floor( gSub ) );

}





//

//   f o r w a r d F o r w a r d S I

//



double tHJMTermStruct::forwardForwardSI( int n,

    double gFromSub, double gToSub ) const



{

    MTG_ASSERT( n <= gFromSub && gFromSub <= gToSub );



    if( gFromSub == gToSub )

        return forwardForwardSI( n, gFromSub );



    int nCeilFrom = (int) ceil( gFromSub );

    int nFloorTo = (int) floor( gToSub );



    if( nCeilFrom > nFloorTo ) {

            // interval is within the same sub interval

        return forwardForwardSI( n, nFloorTo );

    }



    double gRate = 0;



    if( nCeilFrom < nFloorTo ) {

            // interval spans several sub intervals

        if( nCeilFrom == n )

            gRate += D( n, nFloorTo - 1 );

        else

            gRate += D( n, nFloorTo - 1 ) - D( n, nCeilFrom - 1 );

    }



        // now look at head and tail

    if( nCeilFrom > gFromSub ) {

        gRate += ( nCeilFrom - gFromSub ) *

                    forwardForwardSI( n, nCeilFrom - 1 );

    }

    if( nFloorTo < gToSub ) {

        gRate += ( gToSub - nFloorTo ) *

                    forwardForwardSI( n, nFloorTo );

    }



    return gRate / ( gToSub - gFromSub );

}





//

//   f o r w a r d F o r w a r d S I

//



double tHJMTermStruct::forwardForwardSI( double g ) const



{

    int n = (int) floor( g );



    if( n == g )

        return forwardForwardSI( n );



    double e = g - n;



    return forwardForwardSI( n ) +

            e * ( forwardForwardSI( n + 1 ) - forwardForwardSI( n, n + 1 ) );

}





//

//   f o r w a r d F o r w a r d S I

//



double tHJMTermStruct::forwardForwardSI( double g, int nSub ) const



{

    MTG_ASSERT( g <= nSub );



    int n = (int) floor( g );



    if( n == g )

        return forwardForwardSI( n, nSub );



    double e = g - n;



    return (1 - e ) * forwardForwardSI( n, nSub ) +

                  e * forwardForwardSI( n + 1, nSub );

}





//

//   f o r w a r d F o r w a r d S I

//



double tHJMTermStruct::forwardForwardSI( double g,

    int nFromSub, int nToSub ) const



{

    MTG_ASSERT( g <= nFromSub && nFromSub <= nToSub );



    if( nFromSub == nToSub )

        return forwardForwardSI( g, nFromSub );



    int n = (int) floor( g );



    if( n == g )

        return forwardForwardSI( n, nFromSub, nToSub );



    double e = g - n;



    return ( 1 - e ) * forwardForwardSI( n, nFromSub, nToSub ) +

                   e * forwardForwardSI( n + 1, nFromSub, nToSub );

}





//

//   f o r w a r d F o r w a r d S I

//



double tHJMTermStruct::forwardForwardSI( double g, double gSub ) const



{

    MTG_ASSERT( g <= gSub );



    int n = (int) floor( g );



    if( n == g )

        return forwardForwardSI( n, gSub );

    if( gSub < n + 1 )

        return forwardForwardSI( g );



    double e = g - n;



    return ( 1 - e ) * forwardForwardSI( n, gSub ) +

                   e * forwardForwardSI( n + 1, gSub );

}





//

//   f o r w a r d F o r w a r d S I

//



double tHJMTermStruct::forwardForwardSI( double g,

    double gFromSub, double gToSub ) const



{

    MTG_ASSERT( g <= gFromSub && gFromSub <= gToSub );



    if( gFromSub == gToSub )

        return forwardForwardSI( g, gFromSub );



    int n = (int) floor( g );



    if( n == g )

        return forwardForwardSI( n, gFromSub, gToSub );

    if( gToSub < n + 1 )

        return forwardForwardSI( g );



    double e = g - n;



    if( gFromSub >= n + 1 ) {

        return ( 1 - e ) * forwardForwardSI( n, gFromSub, gToSub ) +

                       e * forwardForwardSI( n + 1, gFromSub, gToSub );

    }



    double a =

        ( 1 - e ) * forwardForwardSI( n, (double) ( n + 1 ), gToSub ) +

                e * forwardForwardSI( n + 1, (double) ( n + 1 ), gToSub );



    a *= gToSub - ( n + 1 );

     

    double b = forwardForwardSI( g ) * ( ( n + 1 ) - gFromSub );



    return ( a + b ) / ( gToSub - gFromSub );

}





//

//   f o r w a r d

//



double tHJMTermStruct::forward( tDate Start, tDate End ) const



{

    if( scale() == xDay ) {

        int nFromSub, nToSub;



        getSI( Start, nFromSub );

        getSI( End, nToSub );

        return forwardSI( nFromSub, nToSub );

    }



    return forwardSI( getSI( Start ), getSI( End ) );

}





//

//   f o r w a r d F o r w a r d

//



double tHJMTermStruct::forwardForward( tDate Ref ) const



{

    if( scale() == xDay ) {

        int n;



        getSI( Ref, n );

        return forwardForwardSI( n );

    }



    return forwardForwardSI( getSI( Ref ) );

}





//

//   f o r w a r d F o r w a r d

//



double tHJMTermStruct::forwardForward( tDate Ref, tDate Date ) const



{

    if( scale() == xDay ) {

        int n, nSub;



        getSI( Ref, n );

        getSI( Date, nSub );

        return forwardForwardSI( n, nSub );

    }



    return forwardForwardSI( getSI( Ref ), getSI( Date ) );

}





//

//   f o r w a r d F o r w a r d

//



double tHJMTermStruct::forwardForward( tDate Ref,

    tDate Start, tDate End ) const



{

    if( scale() == xDay ) {

        int n, nFromSub, nToSub;



        getSI( Ref, n );

        getSI( Start, nFromSub );

        getSI( End, nToSub );

        return forwardForwardSI( n, nFromSub, nToSub );

    }



    return forwardForwardSI( getSI( Ref ),

            getSI( Start ), getSI( End ) );

}





//

//   f o r w a r d F o r w a r d 0

//



double tHJMTermStruct::forwardForward0( tDate Date ) const



{

    if( scale() == xDay ) {

        int nSub;



        getSI( Date, nSub );

        return forwardForwardSI( 0, nSub );

    }



    return forwardForwardSI( 0, getSI( Date ) );

}





//

//   f o r w a r d F o r w a r d 0

//



double tHJMTermStruct::forwardForward0( tDate Start, tDate End ) const



{

    if( scale() == xDay ) {

        int nFromSub, nToSub;



        getSI( Start, nFromSub );

        getSI( End, nToSub );

        return forwardForwardSI( 0, nFromSub, nToSub );

    }



    return forwardForwardSI( 0, getSI( Start ), getSI( End ) );

}





//

//   p r e s e n t V a l u e T U

//



double tHJMTermStruct::presentValueTU( int nFromUnit, int nToUnit ) const



{

    double gFwd = forwardTU( nFromUnit, nToUnit );

    return exp( -gFwd * (double) ( nToUnit - nFromUnit ) * dtTU() );

}





//

//   f u t u r e V a l u e T U

//



double tHJMTermStruct::futureValueTU( int nFromUnit, int nToUnit ) const



{

    return 1 / presentValueTU( nFromUnit, nToUnit );

}





//

//   d i s c o u n t F a c t o r T U

//



double tHJMTermStruct::discountFactorTU( int n,

    int nFromUnit, int nToUnit ) const



{

    double gFwd = forwardForwardTU( n, nFromUnit, nToUnit );

    return exp( -gFwd * (double) ( nToUnit - nFromUnit ) * dtTU() );

}





//

//   a c c r u a l F a c t o r T U

//



double tHJMTermStruct::accrualFactorTU( int n,

    int nFromUnit, int nToUnit ) const



{

    return 1 / discountFactorTU( n, nFromUnit, nToUnit );

}





//

//   d i s c o u n t F a c t o r T U

//



double tHJMTermStruct::discountFactorTU( double g,

    int nFromUnit, int nToUnit ) const



{

    double gFwd = forwardForwardTU( g, nFromUnit, nToUnit );

    return exp( -gFwd * (double) ( nToUnit - nFromUnit ) * dtTU() );

}





//

//   a c c r u a l F a c t o r T U

//



double tHJMTermStruct::accrualFactorTU( double g,

    int nFromUnit, int nToUnit ) const



{

    return 1 / discountFactorTU( g, nFromUnit, nToUnit );

}





//

//   p r e s e n t V a l u e S I

//



double tHJMTermStruct::presentValueSI( int nFromSub, int nToSub ) const



{

    double gFwd = forwardSI( nFromSub, nToSub );

    return exp( -gFwd * (double) ( nToSub - nFromSub ) * dtSI() );

}





//

//   f u t u r e V a l u e S I

//



double tHJMTermStruct::futureValueSI( int nFromSub, int nToSub ) const



{

    return 1 / futureValueSI( nFromSub, nToSub );

}





//

//   d i s c o u n t F a c t o r S I

//



double tHJMTermStruct::discountFactorSI( int n,

    int nFromSub, int nToSub ) const



{

    double gFwd = forwardForwardSI( n, nFromSub, nToSub );

    return exp( -gFwd * (double) ( nToSub - nFromSub ) * dtSI() );

}





//

//   a c c r u a l F a c t o r S I

//



double tHJMTermStruct::accrualFactorSI( int n,

    int nFromSub, int nToSub ) const



{

    return 1 / discountFactorSI( n, nFromSub, nToSub );

}





//

//   d i s c o u n t F a c t o r S I

//



double tHJMTermStruct::discountFactorSI( double g,

    int nFromSub, int nToSub ) const



{

    double gFwd = forwardForwardSI( g, nFromSub, nToSub );

    return exp( -gFwd * (double) ( nToSub - nFromSub ) * dtSI() );

}





//

//   a c c r u a l F a c t o r S I

//



double tHJMTermStruct::accrualFactorSI( double g,

    int nFromSub, int nToSub ) const



{

    return 1 / discountFactorSI( g, nFromSub, nToSub );

}





//

//   p r e s e n t V a l u e T U

//



double tHJMTermStruct::presentValueTU( double gFromUnit, double gToUnit ) const



{

    double gFwd = forwardTU( gFromUnit, gToUnit );

    return exp( -gFwd * ( gToUnit - gFromUnit ) * dtTU() );

}





//

//   f u t u r e V a l u e T U

//



double tHJMTermStruct::futureValueTU( double gFromUnit, double gToUnit ) const



{

    return 1 / presentValueTU( gFromUnit, gToUnit );

}





//

//   d i s c o u n t F a c t o r T U

//



double tHJMTermStruct::discountFactorTU( int n,

    double gFromUnit, double gToUnit ) const



{

    double gFwd = forwardForwardTU( n, gFromUnit, gToUnit );

    return exp( -gFwd * ( gToUnit - gFromUnit ) * dtTU() );

}





//

//   a c c r u a l F a c t o r T U

//



double tHJMTermStruct::accrualFactorTU( int n,

    double gFromUnit, double gToUnit ) const



{

    return 1 / discountFactorTU( n, gFromUnit, gToUnit );

}





//

//   d i s c o u n t F a c t o r T U

//



double tHJMTermStruct::discountFactorTU( double g,

    double gFromUnit, double gToUnit ) const



{

    double gFwd = forwardForwardTU( g, gFromUnit, gToUnit );

    return exp( -gFwd * ( gToUnit - gFromUnit ) * dtTU() );

}





//

//   a c c r u a l F a c t o r T U

//



double tHJMTermStruct::accrualFactorTU( double g,

    double gFromUnit, double gToUnit ) const



{

    return 1 / discountFactorTU( g, gFromUnit, gToUnit );

}





//

//   p r e s e n t V a l u e S I

//



double tHJMTermStruct::presentValueSI( double gFromSub, double gToSub ) const



{

    double gFwd = forwardSI( gFromSub, gToSub );

    return exp( -gFwd * ( gToSub - gFromSub ) * dtSI() );

}





//

//   f u t u r e V a l u e S I

//



double tHJMTermStruct::futureValueSI( double gFromSub, double gToSub ) const



{

    return 1 / presentValueSI( gFromSub, gToSub );

}





//

//   d i s c o u n t F a c t o r S I

//



double tHJMTermStruct::discountFactorSI( int n,

    double gFromSub, double gToSub ) const



{

    double gFwd = forwardForwardSI( n, gFromSub, gToSub );

    return exp( -gFwd * ( gToSub - gFromSub ) * dtSI() );

}





//

//   a c c r u a l F a c t o r S I

//



double tHJMTermStruct::accrualFactorSI( int n,

    double gFromSub, double gToSub ) const



{

    return 1 / discountFactorSI( n, gFromSub, gToSub );

}





//

//   d i s c o u n t F a c t o r S I

//



double tHJMTermStruct::discountFactorSI( double g,

    double gFromSub, double gToSub ) const



{

    double gFwd = forwardForwardSI( g, gFromSub, gToSub );

    return exp( -gFwd * ( gToSub - gFromSub ) * dtSI() );

}





//

//   a c c r u a l F a c t o r S I

//



double tHJMTermStruct::accrualFactorSI( double g,

    double gFromSub, double gToSub ) const



{

    return 1 / discountFactorSI( g, gFromSub, gToSub );

}





//

//   p r e s e n t V a l u e

//



double tHJMTermStruct::presentValue( tDate Start, tDate End ) const



{

    double gFwd = forward( Start, End );

    return exp( -gFwd * dayCount().fraction( Start, End ) );

}





//

//   f u t u r e V a l u e

//



double tHJMTermStruct::futureValue( tDate Start, tDate End ) const



{

    return 1 / presentValue( Start, End );

}





//

//   d i s c o u n t F a c t o r

//



double tHJMTermStruct::discountFactor( tDate Ref,

    tDate Start, tDate End ) const



{

    double gFwd = forwardForward( Ref, Start, End );

    return exp( -gFwd * dayCount().fraction( Start, End ) );

}





//

//   a c c r u a l F a c t o r

//



double tHJMTermStruct::accrualFactor( tDate Ref,

    tDate Start, tDate End ) const



{

    return 1 / discountFactor( Ref, Start, End );

}





//

//   d i s c o u n t F a c t o r 0

//



double tHJMTermStruct::discountFactor0( tDate Start, tDate End ) const



{

    double gFwd = forwardForward0( Start, End );

    return exp( -gFwd * dayCount().fraction( Start, End ) );

}





//

//   a c c r u a l F a c t o r 0

//



double tHJMTermStruct::accrualFactor0( tDate Start, tDate End ) const



{

    return 1 / discountFactor0( Start, End );

}



MTG_END_NAMESPACE

