// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#if ! defined(_MTG_HJMTERM_STRUCT_)

#define _MTG_HJMTERM_STRUCT_



#include "MtgTriangle.h"

#include "MtgDrift.h"



MTG_BEGIN_NAMESPACE



class tInterestSpline;





//

//   t H J M T e r m S t r u c t

//



class tHJMTermStruct : public tDateBase {



    typedef tDateBase super;



public:



    typedef float tSample;     // can be float or double, to save space



    class tEvolutionStub {

    public:

        virtual void init( const tHJMTermStruct& HJM,

            int nNumOfSamples, const tHeap<tSample>& Initial ) {

        }



        virtual void cleanup() {

        }



        virtual tRetCode progress( int nStep, int nNumOfSamples,

            const tHeap<tSample>& Initial, const tHeap<tSample>& Last,

            tHeap<tSample>& Future );

    };



private:



    int m_nNumOfSamples;

    int m_nMaxStart;



    tHeap<tSample> m_Initial;

    tHeap<tSample> m_Forward;

    tTriangle<tSample> m_Discount;



    bool m_bIsFinalized;



        // Access to the discount triangle. To be more logical,

        // the original alignment (upper left triangle) is

        // translated to a upper right triangle.



    tSample D( int n ) const {

        MTG_ASSERT( n >= 0 && ( m_nMaxStart < 0 || n <= m_nMaxStart ) );

        return m_Discount[n][0];

    }



    tSample D( int n, int N ) const {

        MTG_ASSERT( n >= 0 && n <= N && N < m_nNumOfSamples &&

                    ( m_nMaxStart < 0 || n <= m_nMaxStart ) );

        return m_Discount[n][N - n];

    }



    void init();



    void allocSpace( int nNumOfSamples );

    void allocSpace( const tDrift& Drift );

    tRetCode loadDrift( const tDrift& Drift, int nNumOfSamples );



protected:



    void copyFrom( const tDrift& Drift );

    void copyFrom( const tHJMTermStruct& HJM );



public:



    tHJMTermStruct();

    tHJMTermStruct( const tDrift& Drift );



        // Passed on to the tDateBase constructor:



    tHJMTermStruct( tScale nScale );

    tHJMTermStruct( tScale nScale, int nNumOfPeriods );

    tHJMTermStruct( tScale nScale, const tDayCount& DayCount );

    tHJMTermStruct( tScale nScale, int nNumOfPeriods,

        const tDayCount& DayCount );

    tHJMTermStruct( tScale nScale, const tDayCount& DayCount,

        tDate Base );

    tHJMTermStruct( tScale nScale, int nNumOfPeriods,

        const tDayCount& DayCount, tDate Base );



    tHJMTermStruct( const tHJMTermStruct& Drift );



    ~tHJMTermStruct();



    tHJMTermStruct& operator=( const tHJMTermStruct& HJM );

    tHJMTermStruct& operator=( const tDrift& Drift );



        // If one doesn't want to change base date etc, the following

        // functions alloc more control:



    tRetCode loadInitial( const tDrift& Drift, int nNumOfSamples = -1 );

    tRetCode loadInitial( tInterestSpline& Spline,

        int nNumOfSamples = -1 );



        // This version loads a constant initial rate; the number of

        // samples parameter must be greater than zero.



    tRetCode loadInitial( double gConstInitial, int nNumOfSamples );



    void limitEvolution( int nMaxStart = -1 );



    tRetCode finalize( tEvolutionStub& Evolution, int nMaxStart );

    tRetCode finalize( tEvolutionStub& Evolution );



    int numOfSamples() const {

        return m_Initial.numOfElems();

    }



        // The following functions test whether a subinterval,

        // time unit or date is covered by the term structure.



    bool isCoveredSI( int nSub ) const;

    bool isCoveredSI( double gSub ) const;



    bool isCoveredTU( int nUnit ) const;

    bool isCoveredTU( double gUnit ) const;



    bool isCovered( int nSub ) const {

        return isCoveredSI( nSub );

    }



    bool isCovered( double gSub ) const {

        return isCoveredSI( gSub );

    }



    bool isCovered( tDate Date ) const;



        // Retrieval based on time unit and scaling:



    double forwardTU( int nFromUnit, int nToUnit ) const;

    double forwardTU( double gFromUnit, double gToUnit ) const;



    double forwardForwardTU( int n ) const;

    double forwardForwardTU( int n, int nUnit ) const;

    double forwardForwardTU( int n,

        int nFromUnit, int nToUnit ) const;



    double forwardForwardTU( int n, double gUnit ) const;

    double forwardForwardTU( int n,

        double gFromUnit, double gToUnit ) const;



    double forwardForwardTU( double g ) const;

    double forwardForwardTU( double g, int nUnit ) const;

    double forwardForwardTU( double g,

        int nFromUnit, int nToUnit ) const;



    double forwardForwardTU( double g, double gUnit ) const;

    double forwardForwardTU( double g,

        double gFromUnit, double gToUnit ) const;



        // Retrieval independent of time unit and scaling:



    double forwardSI( int nFromSub, int nToSub ) const;

    double forwardSI( double gFromSub, double gToSub ) const;



    double forwardForwardSI( int n ) const;

    double forwardForwardSI( int n, int nSub ) const;

    double forwardForwardSI( int n,

        int nFromSub, int nToSub ) const;



    double forwardForwardSI( int n, double gSub ) const;

    double forwardForwardSI( int n,

        double gFromSub, double gToSub ) const;



    double forwardForwardSI( double g ) const;

    double forwardForwardSI( double g, int nSub ) const;

    double forwardForwardSI( double g,

        int nFromSub, int nToSub ) const;



    double forwardForwardSI( double g, double gSub ) const;

    double forwardForwardSI( double g,

        double gFromSub, double gToSub ) const;



        // By default, retrieval is based on subintervals:



    double forward( int nFromSub, int nToSub ) const {

        return forward( nFromSub, nToSub );

    }



    double forward( double gFromSub, double gToSub ) const {

        return forward( gFromSub, gToSub );

    }



    double forwardForward( int n ) const {

        return forwardForwardSI( n );

    }



    double forwardForward( int n, int nSub ) const {

        return forwardForwardSI( n, nSub );

    }



    double forwardForward( int n,

        int nFromSub, int nToSub ) const {

        return forwardForwardSI( n, nFromSub, nToSub );

    }



    double forwardForward( int n, double gSub ) const {

        return forwardForward( n, gSub );

    }



    double forwardForward( int n,

        double gFromSub, double gToSub ) const {

        return forwardForward( n, gFromSub, gToSub );

    }



    double forwardForward( double g ) const {

        return forwardForward( g );

    }



    double forwardForward( double g, int nSub ) const {

        return forwardForward( g, nSub );

    }



    double forwardForward( double g,

        int nFromSub, int nToSub ) const {

        return forwardForward( g, nFromSub, nToSub );

    }



    double forwardForward( double g, double gSub ) const {

        return forwardForward( g, gSub );

    }



    double forwardForward( double g,

        double gFromSub, double gToSub ) const {

        return forwardForward( g, gFromSub, gToSub );

    }



        // Date based retrieval:



    double forward( tDate Start, tDate End ) const;



    double forwardForward( tDate Ref ) const;

    double forwardForward( tDate Ref, tDate Date ) const;

    double forwardForward( tDate Ref, tDate Start, tDate End ) const;



    double forwardForward0( tDate Date ) const;

    double forwardForward0( tDate Start, tDate End ) const;



        // The forward rates can be used to compute present

        // and future values under continuous compounding.

        // The forward forward rates can be used to compute 

        // discount and accrual factors under continuous compounding.



    double presentValueTU( int nFromUnit, int nToUnit ) const;

    double futureValueTU( int nFromUnit, int nToUnit ) const;



    double discountFactorTU( int n, int nFromUnit, int nToUnit ) const;

    double accrualFactorTU( int n, int nFromUnit, int nToUnit ) const;



    double discountFactorTU( double g, int nFromUnit, int nToUnit ) const;

    double accrualFactorTU( double g, int nFromUnit, int nToUnit ) const;



    double presentValueSI( int nFromSub, int nToSub ) const;

    double futureValueSI( int nFromSub, int nToSub ) const;



    double discountFactorSI( int n, int nFromSub, int nToSub ) const;

    double accrualFactorSI( int n, int nFromSub, int nToSub ) const;



    double discountFactorSI( double g, int nFromSub, int nToSub ) const;

    double accrualFactorSI( double g, int nFromSub, int nToSub ) const;



        // The default is to retrieve by subinterval:



    double presentValue( int nFromSub, int nToSub ) const {

        return presentValueSI( nFromSub, nToSub );

    }



    double futureValue( int nFromSub, int nToSub ) const {

        return futureValueSI( nFromSub, nToSub );

    }



    double discountFactor( int n, int nFromSub, int nToSub ) const {

        return discountFactorSI( n, nFromSub, nToSub );

    }



    double accrualFactor( int n, int nFromSub, int nToSub ) const {

        return accrualFactorSI( n, nFromSub, nToSub );

    }



    double discountFactor( double g, int nFromSub, int nToSub ) const {

        return discountFactorSI( g, nFromSub, nToSub );

    }



    double accrualFactor( double g, int nFromSub, int nToSub ) const {

        return accrualFactorSI( g, nFromSub, nToSub );

    }



    double presentValueTU( double gFromUnit, double gToUnit ) const;

    double futureValueTU( double gFromUnit, double gToUnit ) const;



    double discountFactorTU( int n, double gFromUnit, double gToUnit ) const;

    double accrualFactorTU( int n, double gFromUnit, double gToUnit ) const;



    double discountFactorTU( double g, double gFromUnit, double gToUnit ) const;

    double accrualFactorTU( double g, double gFromUnit, double gToUnit ) const;



    double presentValueSI( double gFromSub, double gToSub ) const;

    double futureValueSI( double gFromSub, double gToSub ) const;



    double discountFactorSI( int n, double gFromSub, double gToSub ) const;

    double accrualFactorSI( int n, double gFromSub, double gToSub ) const;



    double discountFactorSI( double g, double gFromSub, double gToSub ) const;

    double accrualFactorSI( double g, double gFromSub, double gToSub ) const;



        // The default is to retrieve by subinterval:



    double presentValue( double gFromSub, double gToSub ) const {

        return presentValueSI( gFromSub, gToSub );

    }                         



    double futureValue( double gFromSub, double gToSub ) const {

        return futureValueSI( gFromSub, gToSub );

    }                         



    double discountFactor( int n, double gFromSub, double gToSub ) const {

        return discountFactorSI( n, gFromSub, gToSub );

    }



    double accrualFactor( int n, double gFromSub, double gToSub ) const {

        return accrualFactorSI( n, gFromSub, gToSub );

    }                         



    double discountFactor( double g, double gFromSub, double gToSub ) const {

        return discountFactorSI( g, gFromSub, gToSub );

    }



    double accrualFactor( double g, double gFromSub, double gToSub ) const {

        return accrualFactorSI( g, gFromSub, gToSub );

    }                         



        // The following two functions are consistently based

        // on the day count convention and work under all scales.

        // It is assumed the investment is made at the beginning

        // of date Start, and paid back at the beginning of

        // date End.



    double presentValue( tDate Start, tDate End ) const;

    double futureValue( tDate Start, tDate End ) const;



    double discountFactor( tDate Ref, tDate Start, tDate End ) const;

    double accrualFactor( tDate Ref, tDate Start, tDate End ) const;



    double discountFactor0( tDate Start, tDate End ) const;

    double accrualFactor0( tDate Start, tDate End ) const;

};



MTG_END_NAMESPACE



#endif

