// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgImage.h"

#include "MtgBootstrap.h"

#include "MtgCurve.h"

#include "MtgCurveContainer.h"

#include "MtgDrift.h"

#include "MtgImageInstance.h"

#include "MtgPortfolio.h"

#include "MtgPathSpace.h"



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tImage::init()



{

    m_sTitle = 0;

    m_nSizeX = 640;

    m_nSizeY = 480;

    m_bHasMinY = false;

    m_bHasMaxY = false;

    m_bDrawHorzGrid = true;

    m_bDrawVertGrid = true;

    m_bDrawPfGrid = true;

    m_gFactor = 100;

    m_pPortfolio = 0;

    m_pPathSpace = 0;

}





//

//   c l e a n u p

//



void tImage::cleanup()



{

    for( int i = 0; i < m_Plot.numOfElems(); ++i ) {

        MTG_ASSERT( m_Plot[i].m_pGraph != 0 );



        m_Plot[i].m_pGraph->downRef();

        if( m_Plot[i].m_pGraph->canDelete() )

            delete m_Plot[i].m_pGraph;

    }

    m_Plot.reset();



    if( m_sTitle != 0 ) {

        delete m_sTitle;

        m_sTitle = 0;

    }



    setObjRefToZeroOrSo( m_pPathSpace );

    setObjRefToZeroOrSo( m_pPortfolio );

}





//

//   c o p y F r o m

//



void tImage::copyFrom( const tImage& Image )



{

    if( &Image == this )

        return;



    cleanup();



    for( int i = 0; i < Image.m_Plot.numOfElems(); ++i ) {

        tPlot& P = Image.m_Plot[i];

        tPlot& Q = addPlot( P.m_pGraph );



        Q.m_nColor = P.m_nColor;

        Q.m_nStyle = P.m_nStyle;

    }



    m_sTitle = StrCopy( Image.m_sTitle );

    m_nSizeX = Image.m_nSizeX;

    m_nSizeY = Image.m_nSizeY;

    m_bHasMinY = Image.m_bHasMinY;

    m_gMinY = Image.m_gMinY;

    m_bHasMaxY = Image.m_bHasMaxY;

    m_gMaxY = Image.m_gMaxY;

    m_bDrawHorzGrid = Image.m_bDrawHorzGrid;

    m_bDrawVertGrid = Image.m_bDrawVertGrid;

    m_bDrawPfGrid = Image.m_bDrawPfGrid;

    m_gFactor = Image.m_gFactor;

    m_OutFile = Image.m_OutFile;



    setObjRefToZeroOrSo( m_pPathSpace, Image.m_pPathSpace );

    setObjRefToZeroOrSo( m_pPortfolio, Image.m_pPortfolio );



    super::copyFrom( Image );

}





//

//   a d d P l o t

//



tImage::tPlot& tImage::addPlot( tObject* pGraph )



{

    MTG_ASSERT( pGraph != 0 );



    ++m_Plot;



    tPlot& P = m_Plot.last();



    P.m_pGraph = pGraph;

    P.m_nColor = Black;

    P.m_nStyle = xLines;



    P.m_pGraph->upRef();



    return P;

}





//

//   p a r s e P l o t

//



tRetCode tImage::parsePlot( tParser& Parser )



{

    MTG_ASSERT( Parser.curToken() == xTokCurve ||

                Parser.curToken() == xTokDrift ||

                Parser.curToken() == xTokBootstrap ||

                Parser.curToken() == xTokId );



    tRetCode nRet;

    tObject* pGraph;

    char* sName;



    tToken nType = Parser.curToken();



    if( Parser.curToken() == xTokCurve ||

        Parser.curToken() == xTokDrift ||

        Parser.curToken() == xTokBootstrap ) {

        if( ( nRet = Parser.readToken() ) != OK )

            return nRet;

    }



    if( Parser.curToken() == xTokId ) {

        sName = StrCopy( Parser.curText() );

        if( ( nRet = Parser.readToken() ) != OK ) {

            delete sName;

            return nRet;

        }

    }

    else {

        sName = 0;

    }



    if( Parser.beginOfObj() ) {

         switch( nType ) {

            case xTokCurve :

                if( ( nRet = tCurve::parse( Parser,

                        system(), pGraph ) ) != OK ) {

                    if( sName != 0 )

                        delete sName;

                    return nRet;

                }

                break;



            case xTokDrift :

                if( ( nRet = tDrift::parse( Parser,

                        system(), pGraph ) ) != OK ) {

                    if( sName != 0 )

                        delete sName;

                    return nRet;

                }

                break;



            case xTokBootstrap :

                if( ( nRet = tBootstrap::parse( Parser,

                        system(), pGraph ) ) != OK ) {

                    if( sName != 0 )

                        delete sName;

                    return nRet;

                }

                break;



            default :

                return Parser.setError( OBJECT_MISMATCH );

        }



        if( sName != 0 ) {

            pGraph->setName( sName );

            delete sName;

        }

    } 

    else {

        if( sName == 0 )

            return Parser.setError( INVALID_KEYWORD );



        pGraph = Parser.findObject( sName );

        delete sName;



        if( pGraph == 0 )

            return Parser.setError( NOT_FOUND );



        if( dynamic_cast<tCurve*>( pGraph ) != 0 ) {

            if( nType != xTokCurve && nType != xTokId )

                return Parser.setError( OBJECT_MISMATCH );

        }

        else

        if( dynamic_cast<tDrift*>( pGraph ) != 0 ) {

            if( nType != xTokDrift && nType != xTokId )

                return Parser.setError( OBJECT_MISMATCH );

        }

        else

        if( dynamic_cast<tBootstrap*>( pGraph ) != 0 ) {

            if( nType != xTokBootstrap && nType != xTokId )

                return Parser.setError( OBJECT_MISMATCH );

        }

        else {

            return Parser.setError( OBJECT_MISMATCH );

        }

    }



    tPlot& P = addPlot( pGraph );



    bool bGo = true;

    bool bColor = false;

    bool bStyle = false;



    while( bGo ) { 

        switch( Parser.curToken() ) {

            case xTokRed :

            case xTokGreen :

            case xTokBlue :

                if( bColor )

                    return Parser.setError( ATTR_REDEFINITION );

                switch( Parser.curToken() ) {

                    case xTokRed :   P.m_nColor = Red;   break;

                    case xTokGreen : P.m_nColor = Green; break;

                    case xTokBlue :  P.m_nColor = Blue;  break;

                    default : throw tException( INTERNAL_ERROR );

                }

                bColor = true;

                break;



            case xTokSteps :

            case xTokLines :

                if( bStyle )

                    return Parser.setError( ATTR_REDEFINITION );

                switch( Parser.curToken() ) {

                    case xTokLines : P.m_nStyle = xLines; break;

                    case xTokSteps : P.m_nStyle = xSteps; break;

                    default : throw tException( INTERNAL_ERROR );

                }

                bStyle = true;

                break;



            default :

                bGo = false;

                break;

        }



        if( bGo && ( nRet = Parser.readToken() ) != OK )

            return nRet;

    }



    return OK;

}





//

//   p a r s e P a r a m

//



tRetCode tImage::parseParam( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;

    tObject* pObj;

    tPathSpace* pPathSpace;

    tPortfolio* pPortfolio;



    tParseInfo& I = static_cast<tParseInfo&>( Info );



    switch( Parser.curToken() ) {

        case xTokTitle :

            if( m_sTitle != 0 )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.scanString( m_sTitle ) ) != OK ) {

                return nRet;

            }

            break;



        case xTokPathSpace :

            if( m_pPathSpace != 0 )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() != xTokId )

                return Parser.setError( INVALID_ID );

            if( ( pObj = Parser.findObject() ) == 0 ) 

                return Parser.setError( NOT_FOUND );

            if( ( pPathSpace = dynamic_cast<tPathSpace*>( pObj ) ) == 0 )

                return Parser.setError( OBJECT_MISMATCH );

            setObjRefToZeroOrSo( m_pPathSpace, pPathSpace );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        case xTokPortfolio :

            if( m_pPortfolio != 0 )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() != xTokId )

                return Parser.setError( INVALID_ID );

            if( ( pObj = Parser.findObject() ) == 0 ) 

                return Parser.setError( NOT_FOUND );

            if( ( pPortfolio = dynamic_cast<tPortfolio*>( pObj ) ) == 0 )

                return Parser.setError( OBJECT_MISMATCH );

            setObjRefToZeroOrSo( m_pPortfolio, pPortfolio );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        case xTokSize :

            if( I.m_bSize )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.scanInteger( m_nSizeX, 32 ) ) != OK ||

                ( nRet = Parser.scanInteger( m_nSizeY, 32 ) ) != OK ) {

                return nRet;

            }

            I.m_bSize = true;

            break;



        case xTokMin :

            if( m_bHasMinY )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.scanDouble( m_gMinY ) ) != OK ) {

                return nRet;

            }

            m_bHasMinY = true;

            break;



        case xTokMax :

            if( m_bHasMaxY )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.scanDouble( m_gMaxY ) ) != OK ) {

                return nRet;

            }

            m_bHasMaxY = true;

            break;



        case xTokCurve :

        case xTokDrift :

        case xTokBootstrap :

            if( ( nRet = parsePlot( Parser ) ) != OK )

                return nRet;

            break;



        case xTokSave :

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() == xTokImage ) { 

                if( ( nRet = Parser.readToken() ) != OK )

                    return nRet;

            }

            if( m_OutFile.isValid() )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = m_OutFile.parseWrite( Parser ) ) != OK )

                return nRet;

            break;



        default :

            return super::parseParam( Parser, Info );

    }

    

    return OK;

}





//

//   p a r s e P o s t f i x

//



tRetCode tImage::parsePostfix( tParser& Parser, tParseInfoStub& Info )



{

    if( m_Plot.numOfElems() == 0 )

        return MISSING_CURVES;

    if( ! m_OutFile.isValid() )

        return MISSING_SAVE;



    return super::parsePostfix( Parser, Info );

}





//

//   t I m a g e

//



tImage::tImage()



{

    init();

}





//

//   t I m a g e

//



tImage::tImage( const tImage& Image )



{

    init();

    copyFrom( Image );

}





//

//   ~ t I m a g e

//



tImage::~tImage()



{

    cleanup();

}





//

//   c l o n e

//



tObject* tImage::clone() const



{

    return new tImage( *this );

}





//

//   o p e r a t o r =

//



tImage& tImage::operator=( const tImage& Image )



{

    if( &Image != this )

        copyFrom( Image );

    return *this;

}





//

//   f i n a l i z e

//



tRetCode tImage::finalize()



{

    return super::finalize();

}





//

//   c r e a t e I n s t a n c e

//



tRetCode tImage::createInstance( tImageInstance*& pImage )



{

    pImage = new tImageInstance( *this );

    return OK;

}





//

//   c r e a t e I n s t a n c e

//



tRetCode tImage::createInstance( tImageInstance*& pImage,

    tCurveContainer& Container )



{

    tRetCode nRet;

    

    if( ( nRet = createInstance( pImage ) ) != OK )

        return nRet;



    for( int i = 0; i < m_Plot.numOfElems(); ++i ) {

        tCurve* pCurve = dynamic_cast<tCurve*>( m_Plot[i].m_pGraph );



        if( pCurve != 0 )

            Container.addCurve( pCurve, pImage, i );

    }



    return OK;

}





//

//   p a r s e

//



tRetCode tImage::parse( tParser& Parser, tSystem& System, tObject*& pObj )



{

    tRetCode nRet;

    tImage* pImage;

    tParseInfo Info;



    Info.m_bSize = false;



    if( ( nRet = Parser.scanBeginOfObj() ) != OK )

        return nRet;



    pImage = new tImage;

    pImage->setSystem( System );



    if( ( nRet = pImage->super::parse( Parser, &Info ) ) != OK ) {

        delete pImage;

        return nRet;

    }



    pObj = pImage;

    return OK;

}



MTG_END_NAMESPACE