// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgImageContainer.h"

#include "MtgCurve.h"

#include "MtgCurveInstance.h"

#include "MtgImage.h"

#include "MtgImageInstance.h"

#include "MtgParser.h"



MTG_BEGIN_NAMESPACE





//

//   c l e a n u p

//



void tImageContainer::cleanup()



{

    for( int i = 0; i < m_Item.numOfElems(); ++i ) {

        tItem& I = *m_Item[i];



        I.m_pImage->downRef();

        if( I.m_pImage->canDelete() )

            delete I.m_pImage;



        if( I.m_pInstance != 0 )

            delete I.m_pInstance;



        delete m_Item[i];

    }

    m_Item.reset();

}





//

//   c o p y F r o m

//



void tImageContainer::copyFrom( const tImageContainer& Container )



{

    if( &Container == this )

        return;



    cleanup();



        // note that we're not copying the curve instances



    for( int i = 0; i < Container.m_Item.numOfElems(); ++i ) {

        tItem& I = *Container.m_Item[i];

        addImage( I.m_pImage, &I.m_OutFile );

    }

}





//

//   a d d I m a g e

//



void tImageContainer::addImage( tImage* pImage, const tFileName* pOutFile )



{

    MTG_ASSERT( pImage != 0 );



    tItem* p = new tItem;



    p->m_pImage = pImage;

    p->m_pImage->upRef();



    if( pOutFile != 0 )

        p->m_OutFile = *pOutFile;



    p->m_pInstance = 0;



    m_Item.append( p );

}





//

//   t I m a g e C o n t a i n e r

//



tImageContainer::tImageContainer()



{

}





//

//   t I m a g e C o n t a i n e r

//



tImageContainer::tImageContainer( const tImageContainer& Container )



{

    copyFrom( Container );

}





//

//   ~ t I m a g e C o n t a i n e r

//



tImageContainer::~tImageContainer()



{

    cleanup();

}





//

//   o p e r a t o r =

//



tImageContainer& tImageContainer::operator=(

    const tImageContainer& Container )



{

    if( &Container != this )

        copyFrom( Container );

    return *this;

}





//

//   m e r g e

//



void tImageContainer::merge( const tImageContainer& Container )



{

    if( &Container == this )

        return;



        // simply append, without checking for duplicates



    for( int i = 0; i < Container.m_Item.numOfElems(); ++i ) {

        tItem& I = *Container.m_Item[i];

        addImage( I.m_pImage, &I.m_OutFile );

    }

}





//

//   p r e p a r e

//



tRetCode tImageContainer::prepare( tCurveContainer& Curves )



{

    tRetCode nRet;



    for( int i = 0; i < m_Item.numOfElems(); ++i ) {

        tItem& I = *m_Item[i];



            // first create image instance



        if( I.m_pInstance != 0 ) {

            delete I.m_pInstance;

            I.m_pInstance = 0;

        }



        if( ( nRet = I.m_pImage->createInstance(

                        I.m_pInstance, Curves ) ) != OK ) {

            goto error;

        }

    }



    return OK;



error:



    for( MTG_FOR_INIT( int ) i = 0; i < m_Item.numOfElems(); ++i ) {

        if( m_Item[i]->m_pInstance != 0 ) {

            delete m_Item[i]->m_pInstance;

            m_Item[i]->m_pInstance = 0;

        }

    }



    return nRet;

}





//

//   s a v e

//



tRetCode tImageContainer::save() const



{

    tRetCode nRet;



    for( int i = 0; i < m_Item.numOfElems(); ++i ) {

        tItem& I = *m_Item[i];

        MTG_ASSERT( I.m_pInstance != 0 );



        if( ( nRet = I.m_pInstance->save( I.m_OutFile ) ) != OK )

            return nRet;

    }



    return OK;

}





//

//   p a r s e

//



tRetCode tImageContainer::parse( tParser& Parser )



{

    tRetCode nRet;

    tObject* pObj;

    tImage* pImage;

    tFileName OutFile;



    switch( Parser.curToken() ) {

        case xTokImage :

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            // fall through



        case xTokId :

            if( Parser.curToken() != xTokId )

                return Parser.setError( INVALID_KEYWORD );

            if( ( pObj = Parser.findObject() ) == 0 )

                return Parser.setError( NOT_FOUND );

            if( ( pImage = dynamic_cast<tImage*>( pObj ) ) == 0 ) 

                return Parser.setError( OBJECT_MISMATCH );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() == xTokString ) {

                if( ( nRet = OutFile.parseWrite( Parser ) ) != OK )

                    return nRet;

            }

            break;



        default :

            return GO;

    }



    addImage( pImage, &OutFile );



    return OK;

}



MTG_END_NAMESPACE



