// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgImageInstance.h"

#include "MtgBootstrap.h"

#include "MtgClaim.h"

#include "MtgCurve.h"

#include "MtgCurveInstance.h"

#include "MtgDrift.h"

#include "MtgImage.h"

#include "MtgPortfolio.h"



extern "C" {

    #include <gdfontt.h>

    #include <gdfonts.h>

    #include <gdfontmb.h>

    #include <gdfontl.h>

    #include <gdfontg.h>

}



MTG_BEGIN_NAMESPACE





//

//   i n i t G D

//



tRetCode tImageInstance::initGD( const tFileName& OutFile )



{

    MTG_ASSERT( m_pGD == 0 && m_pOutFile == 0 );



    tRetCode nRet;



    m_nPlotOrgX = 0;

    m_nPlotOrgY = 0;

    m_nPlotWidth = m_Image.m_nSizeX;

    m_nPlotHeight = m_Image.m_nSizeY;



    if( m_nPlotHeight % 8 != 0 )

        m_nPlotHeight += 8 - m_nPlotHeight % 8;



    if( ( m_pGD = gdImageCreate( m_nPlotWidth, m_nPlotHeight ) ) == 0 )

        return OUT_OF_MEMORY;



    if( ( nRet = OutFile.openWrite( m_pOutFile ) ) != OK ) {

        cleanupGD();

        return nRet;

    }

    

    m_pTitleFont = gdFontMediumBold;

    m_pAxisFont = gdFontSmall;

    

    m_nTopMargin = m_pAxisFont->h / 2;

    m_nBottomMargin = ( 3 * m_pAxisFont->h ) / 2;

    m_nLeftMargin = 7 * m_pAxisFont->w;

    m_nRightMargin = 4 * m_pAxisFont->w;



    m_nBgColor = White.allocate( m_pGD );

    m_nTextColor = Black.allocate( m_pGD );

    m_nAxisColor = Black.allocate( m_pGD );

    m_nGridColor = Gray.allocate( m_pGD );

    m_nHiliteColor = Black.allocate( m_pGD );



    return OK;

}





//

//   c l e a n u p G D

//



void tImageInstance::cleanupGD()



{

    if( m_pGD != 0 ) {

        gdImageDestroy( m_pGD );

        m_pGD = 0;

    }



    if( m_pBrush != 0 ) {

        gdImageDestroy( m_pBrush );

        m_pBrush = 0;

    }



    if( m_pOutFile != 0 ) {

        fclose( m_pOutFile );

        m_pOutFile = 0;

    }

}





//

//   f i l l B r u s h

//



void tImageInstance::fillBrush( tColor Color )



{

    if( m_pBrush == 0 ) {

        m_pBrush = gdImageCreate( 3, 3 );



        int w = White.allocate( m_pBrush );

        gdImageColorTransparent( m_pBrush, w );



        for( int i = 0; i < m_pBrush->sx; ++i ) {

            for( int j = 0; j < m_pBrush->sy; ++j )

                gdImageSetPixel( m_pBrush, i, j, w );

        }

    }



    int c = Color.allocate( m_pBrush );



        // draw a cross



    gdImageSetPixel( m_pBrush, 0, 1, c );

    gdImageSetPixel( m_pBrush, 1, 0, c );

    gdImageSetPixel( m_pBrush, 1, 1, c );

    gdImageSetPixel( m_pBrush, 1, 2, c );

    gdImageSetPixel( m_pBrush, 2, 1, c );



    gdImageSetBrush( m_pGD, m_pBrush );

}





//

//   a n a l y z e E x t e n t X

//



tRetCode tImageInstance::analyzeExtentX()



{

    bool bHasBoot = false;

    bool bHasNonBoot = false;



    tDate BootEnd;



    setScale( xYear, 1 );



    for( int k = 0; k < m_Image.m_Plot.numOfElems(); ++k ) {

        const tDateRange* pRange =

            dynamic_cast<const tDateRange*>( m_Image.m_Plot[k].m_pGraph );



        if( pRange == 0 ) {

            tBootstrap* pBoot =

                dynamic_cast<tBootstrap*>( m_Image.m_Plot[k].m_pGraph );



                // could be bootstrap - mark for later

            if( pBoot == 0 )

                throw tException( INTERNAL_ERROR );



            tDate End;

            double gYield;



            pBoot->spline().getFirstLeg( End, gYield );

            while( pBoot->spline().getNextLeg( End, gYield ) );



            if( ! bHasBoot || End > BootEnd )

                BootEnd = End;

            bHasBoot = true;



            if( k == 0 ) {

                setDayCount( pBoot->spline().interest().dayCount() );

            }

            else {

                if( ! ( dayCount() ==

                            pBoot->spline().interest().dayCount() ) ) {

                    return DATEBASE_MISMATCH;

                }

            }

        }

        else {

            if( k == 0 ) {

                setDayCount( pRange->dayCount() );

            }

            else {

                if( ! ( dayCount() == pRange->dayCount() ) ) 

                    return DATEBASE_MISMATCH;

            }

            if( ! bHasNonBoot ) {

                setBase( pRange->base() );

                bHasNonBoot = true;

            }

            else {

                if( pRange->base() < base() ) 

                    setBase( pRange->base() );

            }

        }

    }



        // find duration as day count fraction



    if( bHasBoot && ! bHasNonBoot ) {

        setBase( m_Image.constSystem().base() );

        m_gDuration = dayCount().fraction( base(), BootEnd );

    }

    else {

        m_gDuration = 0;

        for( MTG_FOR_INIT( int ) k = 0;

                k < m_Image.m_Plot.numOfElems(); ++k ) {

            const tDateRange* pRange =

                dynamic_cast<tDateRange*>( m_Image.m_Plot[k].m_pGraph );



            if( pRange != 0 ) {

                double g = pRange->duration( base() );



                if( g > m_gDuration )

                    m_gDuration = g;

            }

        }

    }



    return OK;

}





//

//   a n a l y z e E x t e n t Y

//



tRetCode tImageInstance::analyzeExtentY()



{

    bool bFirst = true;



    for( int k = 0; k < m_Image.m_Plot.numOfElems(); ++k ) {

        const tDrift* pDrift =

            dynamic_cast<const tDrift*>( m_Image.m_Plot[k].m_pGraph );

        tBootstrap* pBoot =

            dynamic_cast<tBootstrap*>( m_Image.m_Plot[k].m_pGraph );



        if( pDrift != 0 ) {

            MTG_ASSERT( m_Curve[k] == 0 );



            if( pDrift->numOfSamples() != 0 ) {

                double gMin, gMax;



                pDrift->getFwdRange( gMin, gMax );



                gMin *= m_Image.m_gFactor;

                gMax *= m_Image.m_gFactor;



                if( bFirst ) {

                    m_gDataMinY = gMin;

                    m_gDataMaxY = gMax;

                    bFirst = false;

                }

                else {

                    if( m_gDataMinY > gMin )

                        m_gDataMinY = gMin;

                    if( m_gDataMaxY < gMax )

                        m_gDataMaxY = gMax;

                }

            }

        }

        else

        if( pBoot != 0 ) {

            MTG_ASSERT( m_Curve[k] == 0 );



            tDate End;

            double gYield;



            pBoot->spline().getFirstLeg( End, gYield );



            do {

                gYield *= m_Image.m_gFactor;

                if( bFirst ) {

                    m_gDataMinY = gYield;

                    m_gDataMaxY = gYield;

                    bFirst = false;

                }

                else {

                    if( m_gDataMinY > gYield )

                         m_gDataMinY = gYield;

                    else

                    if( m_gDataMaxY < gYield )

                        m_gDataMaxY = gYield;

                }

            } while( pBoot->spline().getNextLeg( End, gYield ) );

        }

        else {

            MTG_ASSERT( m_Curve[k] != 0 );



            const tHeap<double>& Sample = m_Curve[k]->sample();



            for( int j = 0; j < Sample.numOfElems(); ++j ) {

                double g = Sample[j] * m_Image.m_gFactor;



                if( bFirst ) {

                    m_gDataMinY = m_gDataMaxY = g;

                    bFirst = false;

                }

                else {

                    if( g < m_gDataMinY )

                        m_gDataMinY = g;

                    else

                    if( g > m_gDataMaxY )

                        m_gDataMaxY = g;

                }

            }

        }

    }



    if( m_Image.m_bHasMinY )

        m_gDataMinY = m_Image.m_gMinY;

    if( m_Image.m_bHasMaxY )

        m_gDataMaxY = m_Image.m_gMaxY;



    if( m_gDataMinY >= m_gDataMaxY ) {

        double t = m_gDataMinY;

        m_gDataMinY = m_gDataMaxY;

        m_gDataMaxY = t;

    }



    MTG_ASSERT( ! bFirst );

    return OK;

}





//

//   a n a l y z e D a t a

//



tRetCode tImageInstance::analyzeData()



{

    MTG_ASSERT( m_Image.m_Plot.numOfElems() > 0 );



        // find base date (principle: draw entire curves);

        // do all calculations in day count fractions



    tRetCode nRet;



    if( ( nRet = analyzeExtentX() ) != OK ||

        ( nRet = analyzeExtentY() ) != OK ) {

        return nRet;

    }



    return OK;

}





//

//   w i d e n D a t a R a n g e

//



void tImageInstance::widenDataRange()



{

        // widen area if necessary



    if( m_gDataMinY == m_gDataMaxY ) {

        if( m_gDataMinY == 0 ) {

            m_gDataMinY = -0.1;

            m_gDataMaxY = 0.1;

        }

        else {

            if( m_gDataMinY > 0 )

                m_gDataMinY /= 1.1;

            else

                m_gDataMinY *= 1.1;

            if( m_gDataMaxY > 0 )

                m_gDataMaxY *= 1.1;

            else

                m_gDataMaxY /= 1.1;

        }

    }



        // include zero if it is close



    if( m_gDataMinY > 0 ) {

        if( m_gDataMaxY / m_gDataMinY > 3 )

            m_gDataMinY = 0;

    }

    else

    if( m_gDataMaxY < 0 ) {

        if( m_gDataMaxY / m_gDataMinY > 3 )

            m_gDataMaxY = 0;

    }

}





//

//   x l a t X

//



int tImageInstance::xlatX( double gDataX, bool bAdjust ) const



{

    double g = gDataX / m_gDuration;

    int x = (int) floor( g * m_nPlotWidth + 0.5 ) + m_nPlotOrgX;



    if( bAdjust ) {

        if( x < m_nPlotOrgX )

            return m_nPlotOrgX;

        if( x > m_nPlotOrgX + m_nPlotWidth )

            return m_nPlotOrgX + m_nPlotWidth;

    }

    return x;

}





//

//   x l a t Y

//



int tImageInstance::xlatY( double gDataY, bool bAdjust ) const



{

    double g = ( m_gDataMaxY - gDataY ) / m_gDataHeight;

    int y = (int) floor( g * m_nPlotHeight + 0.5 ) + m_nPlotOrgY;



    if( bAdjust ) {

        if( y < m_nPlotOrgY )

            return m_nPlotOrgY;

        if( y > m_nPlotOrgY + m_nPlotHeight )

            return m_nPlotOrgY + m_nPlotHeight;

    }

    return y;

}





//

//   p u t V e r t i c a l

//



void tImageInstance::putVertical( int x, int nColor )



{

    if( x < m_nPlotOrgX || x > m_nPlotOrgX + m_nPlotWidth )

        return;



    gdImageLine( m_pGD, x, m_nPlotOrgY + 1, x,

        m_nPlotOrgY + m_nPlotHeight - 1, nColor );

}





//

//   p u t V e r t i c a l

//



void tImageInstance::putVertical( double gFraction, int nColor )



{

    putVertical( xlatX( gFraction, false ), nColor );

}





//

//   p u t V e r t i c a l

//



void tImageInstance::putVertical( tDate Date, int nColor )



{

    putVertical( dayCount().fraction( base(), Date ), nColor );

}





//

//   p u t T i c k

//



void tImageInstance::putTick( const char* sText, int x, int nColor,

    int& nFrom, int& nTo )



{

    int l = m_pAxisFont->w * strlen( sText );

    int y = m_nPlotOrgY + m_nPlotHeight + m_pAxisFont->h / 2;



    nFrom = x - l / 2;

    nTo = nFrom + l;



    gdImageString( m_pGD, m_pAxisFont, nFrom,

        y, (unsigned char*) const_cast<char*>( sText ), nColor );



    y = m_nPlotOrgY + m_nPlotHeight;

    gdImageLine( m_pGD, x, y, x, y + m_pAxisFont->h / 4, m_nAxisColor );

}





//

//   p u t T i c k I f

//



void tImageInstance::putTickIf( const char* sText, int x, int nColor,

    int& nLeft, int nRight )



{

    int l = m_pAxisFont->w * strlen( sText );   

    int t = x - l / 2;



    if( t >= nLeft + 2 * m_pAxisFont->w ) {

        if( t + l <= nRight - 2 * m_pAxisFont->w )

            putTick( sText, x, nColor, t, nLeft );

    }

}





//

//   d o T i t l e

//



tRetCode tImageInstance::doTitle()



{

    if( m_Image.m_sTitle == 0 )

        return OK;



        // check boundaries



    int h = m_pTitleFont->h;

    if( m_nPlotHeight < h )

        return IMAGE_TOO_SMALL;



    time_t NowT = m_Image.constSystem().creation();

    struct tm Now = *localtime( &NowT );



    char* s = 0;

    int k = 0;



        // find special sequence "%t"



    while( m_Image.m_sTitle[k] ) {

        if( m_Image.m_sTitle[k] == '%' && m_Image.m_sTitle[k + 1] == 't' ) {

            s = new char[strlen( m_Image.m_sTitle ) + 4];

            memcpy( s, m_Image.m_sTitle, k );

            sprintf( s + k, "%02d:%02d%s",

                Now.tm_hour, Now.tm_min - Now.tm_min % 5,

                &m_Image.m_sTitle[k + 2] );

            break;

        }

        ++k;

    }



    tDateFormat Format( s == 0 ? m_Image.m_sTitle : s, tDate( Now ) );



    if( s != 0 )

        delete s;



    int w = m_pTitleFont->w * strlen( Format() );

    if( m_nPlotWidth < w )

        return IMAGE_TOO_SMALL;



    gdImageString( m_pGD, m_pTitleFont,

        m_nPlotOrgX + ( m_nPlotWidth - w ) / 2, m_nPlotOrgY,

        (unsigned char*) const_cast<char*>( Format()) , m_nTextColor );



    m_nPlotOrgY += h;

    m_nPlotHeight -= h;



    return OK;

}





//

//   d o B o x

//



tRetCode tImageInstance::doBox()



{

    m_nPlotOrgY += m_nTopMargin;

    m_nPlotHeight -= m_nTopMargin + m_nBottomMargin;

    m_nPlotOrgX += m_nLeftMargin;

    m_nPlotWidth -= m_nLeftMargin + m_nRightMargin;



    if( m_nPlotWidth < 32 || m_nPlotHeight < 32 )

        return IMAGE_TOO_SMALL;



    gdImageRectangle( m_pGD, m_nPlotOrgX, m_nPlotOrgY,

        m_nPlotOrgX + m_nPlotWidth,

        m_nPlotOrgY + m_nPlotHeight, m_nAxisColor );



    return OK;

}





//

//   d o T i c k s Y

//



tRetCode tImageInstance::doTicksY()



{

    static int Step[] = {

        1, 2, 5, 10, 20, 50, 100

    };



    widenDataRange();



        // count intervals, not ticks yet; limit the number

        // of steps



    int nNumOfSteps = m_nPlotHeight / ( 3 * m_pAxisFont->h / 2 );



    if( nNumOfSteps == 0 )

        return IMAGE_TOO_SMALL;

    if( nNumOfSteps > 16 )

        nNumOfSteps = 16;       



    double gMin = m_gDataMinY;

    double gMax = m_gDataMaxY;

    double gSpan = gMax - gMin;



        // normalize the data so that the data range

        // falls between between 10 and 100



    int nNorm = 0;



    if( gSpan > 100 ) {

        while( gSpan > 100 ) {

            gSpan /= 10; gMin /= 10; gMax /= 10;

            ++nNorm;

        }

    }

    else {

        while( gSpan < 10 ) {

            gSpan *= 10; gMin *= 10; gMax *= 10;

            --nNorm;

        }

    }



        // now try all the steps and choose the smallest one



    int nNewMin;

    int nNewMax;

    int k = 0;



    while( k < sizeof(Step) / sizeof(Step[0]) ) {

        nNewMin = (int) floor( gMin / Step[k] );

        nNewMax = (int) ceil( gMax / Step[k] );

        if( nNewMax - nNewMin <= nNumOfSteps ) {

            nNumOfSteps = nNewMax - nNewMin;

            nNewMin *= Step[k];

            nNewMax *= Step[k];

            break;

        }

        ++k;

    }



    if( k == sizeof(Step) / sizeof(Step[0]) ) {

        nNewMin = (int) floor( gMin / 100 ) * 100;

        nNewMax = (int) ceil( gMax / 100 ) * 100;

        nNumOfSteps = 1;            

    }



    m_gStepY = ( nNewMax - nNewMin ) / (double) nNumOfSteps;

    m_gDataMinY = nNewMin;

    m_gDataMaxY = nNewMax;



    if( nNorm != 0 ) {

        double p10 = pow( 10, nNorm );



        m_gStepY *= p10;

        m_gDataMinY *= p10;

        m_gDataMaxY *= p10;

    }



    m_nNumOfTicksY = nNumOfSteps + 1;

    m_gDataHeight = m_gDataMaxY - m_gDataMinY;



    return OK;

}





//

//   d o T i c k s X

//



tRetCode tImageInstance::doTicksX()



{

    static double Step[] = {

        1.0 / 12, 1.0 / 4, 1.0 / 2, 1, 2, 5 

    };



        // find step size



    int k = 0;

    double g = ( m_gDuration * 2 * m_pAxisFont->w ) / m_nPlotWidth;



    while( k < sizeof(Step) / sizeof(Step[0]) - 1 && Step[k] < g )

        ++k;

    m_gStepX = Step[k];



        // generate dates for ticks



    m_XTick.append( tXTick( m_nPlotOrgX, base() ) );



    double gFraction = m_gStepX;

    while( gFraction < m_gDuration / 1.01 ) {

        m_XTick.append(

            tXTick( xlatX( gFraction ), getClosestDateSI( gFraction ) ) );

        gFraction += m_gStepX;

    }

    m_XTick.append( tXTick( m_nPlotOrgX + m_nPlotWidth, 

        getClosestDateSI( m_gDuration ) ) );



        // find print format



    if( m_gStepX >= 1 ) {

        m_LeftFormat = tDateFormat( "%b %y" );

        m_MidFormat = tDateFormat( "%y" );

        m_RightFormat = tDateFormat( "%Y" );

    }

    else

    if( m_gStepX >= 1.0 / 4 ) {

        m_LeftFormat = tDateFormat( "%b %y" );

        m_MidFormat = tDateFormat( "%b %y" );

        m_RightFormat = tDateFormat( "%b %y" );

    }

    else {

        m_LeftFormat = tDateFormat( "%m/%d/%y" );

        m_MidFormat = tDateFormat( "%m/%d/%y" );

        m_RightFormat = tDateFormat( "%m/%d/%y" );

    }



    return OK;

}





//

//   d o G r i d

//



tRetCode tImageInstance::doGrid()



{

        // draw horizontal lines



    if( m_Image.m_bDrawHorzGrid ) {

        for( int i = 0; i < m_nNumOfTicksY; ++i ) {

            int y;

            double g;



            if( i == 0 ) {

                y = m_nPlotOrgY + m_nPlotHeight;

                g = m_gDataMinY;

            }

            else

            if( i == m_nNumOfTicksY - 1 ) {

                y = m_nPlotOrgY;

                g = m_gDataMaxY;

            }

            else {

                g = m_gDataMinY + i * m_gStepY;

                y = xlatY( g );

                gdImageLine( m_pGD, m_nPlotOrgX + 1, y,

                    m_nPlotOrgX + m_nPlotWidth - 1, y, m_nGridColor );

            }



                // make sure zero is zero

            if( fabs( g ) < m_gStepY / 100 )

                g = 0;



            char s[64];

            sprintf( s, "%g", g );



            gdImageString( m_pGD, m_pAxisFont,

                m_nPlotOrgX - m_pAxisFont->w * ( strlen( s ) + 1 ),

                y - m_pAxisFont->h / 2, (unsigned char*) s, m_nTextColor );

        }

    }



        // now draw vertical lines and ticks



    if( m_Image.m_bDrawVertGrid ) {

        const char* sl = m_LeftFormat( m_XTick[0].m_Date );

        const char* sr = m_RightFormat( m_XTick.last().m_Date );



        int ll, lr, t;



        putTick( sl, m_XTick[0].m_nX, m_nTextColor, t, ll );

        putTick( sr, m_XTick.last().m_nX, m_nTextColor, lr, t );



        for( int i = 1; i < m_XTick.numOfElems() - 1; ++i ) {

            const char* s = m_MidFormat( m_XTick[i].m_Date );



            putVertical( m_XTick[i].m_nX, m_nGridColor );

            putTickIf( s, m_XTick[i].m_nX, m_nTextColor, ll, lr );

        }

    }



        // now draw lines indicating portfolio maturities



    if( m_Image.m_bDrawPfGrid ) {

        int Style[2];



        Style[0] = m_nHiliteColor;

        Style[1] = gdTransparent;



        gdImageSetStyle( m_pGD, Style, sizeof(Style) / sizeof(Style[0]) );



        if( m_Image.m_pPortfolio != 0 ) {

            const tPortfolio& Portfolio = *m_Image.m_pPortfolio;



            for( int i = 0; i < Portfolio.numOfClaims(); ++i ) {

                putVertical( Portfolio.claim( i ).maturityDate(),

                    gdStyled );

            }

        }

    }



    return OK;

}





//

//   d o D a t a

//



tRetCode tImageInstance::doData()



{

    for( int k = 0; k < m_Image.m_Plot.numOfElems(); ++k ) {

        tImage::tStyle nStyle = m_Image.m_Plot[k].m_nStyle;



        fillBrush( m_Image.m_Plot[k].m_nColor );

        int nColor = gdBrushed;        



        const tDateRange* pRange =

            dynamic_cast<const tDateRange*>( m_Image.m_Plot[k].m_pGraph );



        if( pRange == 0 ) {

            tBootstrap* pBoot =

                dynamic_cast<tBootstrap*>( m_Image.m_Plot[k].m_pGraph );



            MTG_ASSERT( pBoot != 0 );



            tDate End;

            double gYield;



            bool bFirst = true;

            int nCurX = m_nPlotOrgX;

            int nLastY = 0;



            pBoot->spline().getFirstLeg( End, gYield );

            do {

                if( End >= base() ) {

                    int nNextX = xlatX( dayCount().fraction( base(), End ) );

                    int nCurY = xlatY( gYield * m_Image.m_gFactor );



                    if( ! bFirst ) {

                        gdImageLine( m_pGD,

                            nCurX, nLastY, nCurX, nCurY, nColor );

                    }

                    else {

                        bFirst = false;

                    }

                    gdImageLine( m_pGD, nCurX, nCurY, nNextX, nCurY, nColor );



                    if( nNextX >= m_nPlotOrgX + m_nPlotWidth )

                        break;



                    nCurX = nNextX;

                    nLastY = nCurY;

                }

            } while( pBoot->spline().getNextLeg( End, gYield ) );

        }

        else {

            int n = pRange->numOfSamples();

            if( n == 0 )

                continue;



            const tHeap<double>* pSample;

            const tDrift* pDrift =

                dynamic_cast<const tDrift*>( m_Image.m_Plot[k].m_pGraph );



            if( pDrift == 0 ) {

                pSample = &m_Curve[k]->sample();

                MTG_ASSERT( pSample->numOfElems() == n );

            }

            else {

                pSample = 0;

            }



            double gCurData = pSample ?

                (*pSample)[0] : pDrift->forwardSI( 0 );

            double gLastData = 0;

            double gNextData = 0;



            int nCurX = xlatX( pRange->getFractionSI( 0 ) );

            int gCurY = xlatY( gCurData * m_Image.m_gFactor );

            int nLastY = 0;

            int nNextY = 0;



            for( int i = 0; i < n; ++i ) {

                int nNextX = xlatX( pRange->getFractionSI( i + 1 ) );



                if( i < n - 1 ) {

                    gNextData = pSample ?

                        (*pSample)[i + 1] : pDrift->forwardSI( i + 1 );

                    nNextY = xlatY( gNextData * m_Image.m_gFactor );

                }

        

                switch( nStyle ) {

                    case tImage::xSteps : {

                            if( i > 0 ) {

                                gdImageLine( m_pGD,

                                    nCurX, nLastY, nCurX, gCurY, nColor );

                            }

                            gdImageLine( m_pGD,

                                nCurX, gCurY, nNextX, gCurY, nColor );

                        }

                        break;



                    case tImage::xLines :

                        if( i < n - 1 ) {

                            gdImageLine( m_pGD,

                                nCurX, gCurY, nNextX, nNextY, nColor );

                        }

                        else {

                            if( i == 0 ) {

                                gdImageLine( m_pGD,

                                    nCurX, gCurY, nNextX, gCurY, nColor );

                            }

                            else {

                                gdImageLine( m_pGD, nCurX, gCurY, nNextX,

                                    xlatY( m_Image.m_gFactor *

                                        ( 2 * gCurData - gLastData ) ),

                                    nColor );

                            }

                        }

                        break;



                    default :

                        throw tException( INTERNAL_ERROR );

                }



                nCurX = nNextX;

                gLastData = gCurData;

                gCurData = gNextData;

                nLastY = gCurY;

                gCurY = nNextY;

            }

        }

    }



    return OK;

}





//

//   t I m a g e I n s t a n c e

//



tImageInstance::tImageInstance( const tImage& Image )

    : m_Image( Image )



{

    m_pGD = 0;

    m_pBrush = 0;

    m_pOutFile = 0;



        // wait for the data to be registered here:



    m_Curve.numOfElems( m_Image.m_Plot.numOfElems() );

    m_Curve.fill( 0 );

}





//

//   ~ t I m a g e I n s t a n c e

//



tImageInstance::~tImageInstance()



{

    cleanupGD();

}





//

//   r e g i s t e r D a t a

//



void tImageInstance::registerData( const tCurveInstance* pCurve, int nTag )



{

    MTG_ASSERT( pCurve != 0 && m_Curve[nTag] == 0 );

    MTG_ASSERT( static_cast<const tObject*>( &pCurve->curve() ) ==

        m_Image.m_Plot[nTag].m_pGraph );



    m_Curve[nTag] = pCurve;

}





//

//   s a v e

//



tRetCode tImageInstance::save( const tFileName& OutFile )



{

    tRetCode nRet;



    if( m_Image.m_Plot.numOfElems() == 0 )

        return OK;



        // make sure all data is there

    for( int i = 0; i < m_Image.m_Plot.numOfElems(); ++i ) {

        if( dynamic_cast<const tCurve*>( m_Image.m_Plot[i].m_pGraph ) != 0 &&

                m_Curve[i] == 0 ) {

            return INTERNAL_ERROR;

        }

    }



    const tFileName& File =

        OutFile.isValid() ? OutFile : m_Image.m_OutFile;



    if( ! File.isValid() )        

        return OK;



    if( ( nRet = analyzeData() ) != OK )

        return nRet;



    if( m_gDuration <= 0 )

        return NOT_ENOUGH_DATA;



    if( ( nRet = initGD( File ) ) != OK )

        return nRet;



    if( ( nRet = doTitle() ) != OK ||

        ( nRet = doBox() ) != OK ||

        ( nRet = doTicksY() ) != OK ||

        ( nRet = doTicksX() ) != OK ||

        ( nRet = doGrid() ) != OK ||

        ( nRet = doData() ) != OK ) {

        cleanupGD();

        return nRet;

    }



        // now dump the whole thing

        

    gdImagePng( m_pGD, m_pOutFile );



    cleanupGD();

    return OK;

}





//

//   s a v e

//



tRetCode tImageInstance::save()



{

    return save( m_Image.m_OutFile );

}



MTG_END_NAMESPACE



