// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgInterest.h"



#include <float.h>



MTG_BEGIN_NAMESPACE





//

//   v a r i a b l e s

//



const int tInterest::m_nMaxIter = 50;





//

//   i n i t

//



void tInterest::init( tType nType, tQuote nQuote,

    const tDayCount& DayCount )



{

    m_nType = nType;

    m_nQuote = nQuote;

    m_DayCount = DayCount;

    m_pCompounder = 0;

}





//

//   c l e a n u p

//



void tInterest::cleanup()



{

    if( m_pCompounder != 0 ) {

        delete m_pCompounder;

        m_pCompounder = 0;

    }

}





//

//   c o p y F r o m

//



void tInterest::copyFrom( const tInterest& Interest )



{

    if( &Interest == this )

        return;



    cleanup();



    m_nType = Interest.m_nType;

    m_nQuote = Interest.m_nQuote;

    m_DayCount = Interest.m_DayCount;



    if( Interest.m_pCompounder != 0 ) 

        m_pCompounder = Interest.m_pCompounder->clone();

}





//

//   f i n d F r a c t i o n s

//



void tInterest::findFractions( tDate Start, tDate End,

    const tCompounder* pCompounder, tHeap<double>& Fraction,

    int& nPow0, int& nPeriods ) const



{

    nPow0 = 1;



    if( pCompounder == 0 || 

        ( nPeriods = pCompounder->numOfPeriodsPerYear() ) <= 0 ) {



            // Case 1: there is no compounder, or it indicates

            // continuous compounding. In this case, 

            // we always stick with continuous compounding.



        nPeriods = 0;

        Fraction.append( m_DayCount.fraction( Start, End ) );

        return;

    }



    if( dynamic_cast<const tPeriodCompounder*>( pCompounder ) != 0 ) {



            // Case 2: the compounder is restricted in that it

            // only supplies the number of periods per year, not

            // the actual compounding days.



            // The day count fraction tells us how many years are

            // covered by the interval.

        

        double f = m_DayCount.fraction( Start, End );



            // How often does compounding occur? Here we assume

            // that compounding occurs every 1/nPeriods years.



        nPow0 = (int) floor( f * nPeriods );



        if( nPow0 == 0 ) {

            nPow0 = 1;

            Fraction.append( f );

        }

        else {

            Fraction.append( 1.0 / nPeriods );



            double g = fmod( f, Fraction[0] );

            if( g > 0 )

                Fraction.append( g );

        }



        return;

    }



        // Case 3: the compounder supplies the individual

        // compounding days as well. So get them!



    double gFractionFactor;



    if( ! pCompounder->getYearFactor( m_DayCount, gFractionFactor ) )

        throw tException( INTERNAL_ERROR );



    tDate Date = Start;

    tDate PeriodStart, PeriodEnd;



    pCompounder->getCompoundingPeriod( Date, PeriodStart, PeriodEnd );



    while( true ) {

        tDate Mid = ( PeriodEnd < End ) ? PeriodEnd : End;



        double gFraction = m_DayCount.fraction( Date, Mid,

            PeriodStart, PeriodEnd );



            // gFraction is either relative to the period or

            // to the year. Normalize it:



        Fraction.append( gFraction * gFractionFactor );



        if( Mid >= End )

            break;



        pCompounder->getNextCompoundingPeriod( PeriodStart, PeriodEnd );

        Date = PeriodStart;

    }

}





//

//   f i n d F r a c t i o n s

//



void tInterest::findFractions( double gOverallFraction,

    const tCompounder* pCompounder, tHeap<double>& Fraction,

    int& nPow0, int& nPeriods ) const



{

    nPow0 = 1;



    if( pCompounder == 0 || 

        ( nPeriods = pCompounder->numOfPeriodsPerYear() ) <= 0 ) {



            // Case 1: there is no compounder, or it indicates

            // continuous compounding. In this case, 

            // we always stick with continuous compounding.



        nPeriods = 0;

        Fraction.append( gOverallFraction );

        return;

    }



    if( dynamic_cast<const tPeriodCompounder*>( pCompounder ) != 0 ) {



            // Case 2: the compounder is restricted in that it

            // only supplies the number of periods per year, not

            // the actual compounding days.



        nPow0 = (int) floor( gOverallFraction * nPeriods );



        if( nPow0 == 0 ) {

            nPow0 = 1;

            Fraction.append( gOverallFraction );

        }

        else {

            Fraction.append( 1.0 / nPeriods );



            double g = fmod( gOverallFraction, Fraction[0] );

            if( g > 0 )

                Fraction.append( g );

        }



        return;

    }



        // Case 3: the compounder supplies the individual

        // compounding days as well. This we cannot handle

        // if we only have the overall day count fraction.

        // Error!



    throw tException( NOT_AVAILABLE );

}





//

//   f i n d F r a c t i o n s

//



void tInterest::findFractions( tDate Start, tDate End,

    tHeap<double>& Fraction, int& nPow0 ) const



{

    int nPeriods;

    findFractions( Start, End, m_pCompounder, Fraction, nPow0, nPeriods );

}





//

//   a c c r u e E x p o n e n t i a l

//



double tInterest::accrueExponential( tHeap<double>& Fraction, int nPow0,

    double gYield, int nPeriods, double* pGradient ) const



{

    int i;



        // Type: standard. Sum over the fractions, as only the

        // sum matters.



    double f = Fraction[0] * nPow0;

    for( i = 1; i < Fraction.numOfElems(); ++i )

        f += Fraction[i];



    double y, v;



    if( m_nQuote == xDiscount && nPeriods > 0 ) {

        y = gYield / ( 1 - gYield / nPeriods );

    }

    else {

            // Under continuous compounding, the quoting

            // convention doesn't matter.

        y = gYield;

    }



    if( nPeriods > 0 ) {

            // discrete compounding

        v = pow( 1 + y / nPeriods, f * nPeriods );

        if( pGradient )

            *pGradient = f * v / ( 1 + y / nPeriods );

    }

    else {

            // continuous compounding,

        v = exp( y * f );

        if( pGradient )

            *pGradient = f * v;

    }



    if( m_nQuote == xDiscount && nPeriods > 0 && pGradient ) {

        double g = 1 - gYield / nPeriods;

        *pGradient /=  g * g;

    }



    return v;

}





//

//   a c c r u e L i n e a r

//



double tInterest::accrueLinear( tHeap<double>& Fraction, int nPow0,

    double gYield, int nPeriods, double* pGradient ) const



{

    if( nPeriods <= 0 ) {

            // continuous compounding is the same for both types

        return accrueExponential( Fraction, nPow0, gYield, nPeriods,

            pGradient );

    }



        // Type: Moosmueller. In this case, we have to look at

        // each fraction individually. Also, we need to distinguish

        // more carefully between yield and discount quoting.



    int i;

    double v;



    switch( m_nQuote ) {

        case xDiscount :

            v = 1 / pow( 1 - Fraction[0] * gYield, nPow0 ); 

            for( i = 1; i < Fraction.numOfElems(); ++i )

                v /= 1 - Fraction[i] * gYield;



            if( pGradient ) {

                *pGradient = nPow0 * Fraction[0] / ( 1 - Fraction[0] * gYield );

                for( i = 1; i < Fraction.numOfElems(); ++i )

                    *pGradient += Fraction[i] / ( 1 - Fraction[i] * gYield );

                *pGradient *= v;

            }

            break;



        case xYield:

            if( gYield > 10 ) {

                    // safeguard against overflow in pathetic cases

                double gLimit = pow( DBL_MAX, 1.0 / nPow0 ) / Fraction[0];



                if( gLimit <= gYield )

                    throw tException( OUT_OF_RANGE );

                v = pow( 1 + Fraction[0] * gYield, nPow0 ); 



                for( i = 1; i < Fraction.numOfElems(); ++i ) {

                    gLimit = ( ( DBL_MAX / v ) - 1 ) / Fraction[i];

                    if( gLimit <= gYield )

                        throw tException( OUT_OF_RANGE );

                    v *= 1 + Fraction[i] * gYield;

                }

            }

            else {

                v = pow( 1 + Fraction[0] * gYield, nPow0 ); 

                for( i = 1; i < Fraction.numOfElems(); ++i )

                    v *= 1 + Fraction[i] * gYield;

            }



            if( pGradient ) {

                *pGradient = nPow0 * Fraction[0] / ( 1 + Fraction[0] * gYield );

                for( i = 1; i < Fraction.numOfElems(); ++i )

                    *pGradient += Fraction[i] / ( 1 + Fraction[i] * gYield );

                if( DBL_MAX / v <= *pGradient )

                    throw tException( OUT_OF_RANGE );

                *pGradient *= v;

            }

            break;



        default:

            throw tException( INTERNAL_ERROR );

    }



    return v;

}





//

//   a c c r u e

//



double tInterest::accrue( tHeap<double>& Fraction, int nPow0,

    double gYield, int nPeriods, double* pGradient ) const



{

    switch( m_nType ) {

        case xExponential :    

            return accrueExponential( Fraction, nPow0, gYield,

                nPeriods, pGradient );



        case xLinear :

            return accrueLinear( Fraction, nPow0, gYield, 

                nPeriods, pGradient );



        default :

            throw tException( INTERNAL_ERROR );

    }

    return 0;

}





//

//   i n v e r s e E x p o n e n t i a l

//



bool tInterest::inverseExponential( tHeap<double>& Fraction, int nPow0,

    double gValue, int nPeriods, double& gYield ) const



{

    MTG_ASSERT( Fraction.numOfElems() > 0 );



        // In this case, everything can be done directly.



    double f = Fraction[0] * nPow0;

    for( int i = 1; i < Fraction.numOfElems(); ++i )

        f += Fraction[i];



    if( nPeriods > 0 ) {

            // discrete compounding

        gYield = nPeriods *

            ( pow( gValue, 1.0 / ( nPeriods * f ) ) - 1 );

    }

    else {

            // continuous conpounding,

        gYield = log( gValue ) / f;

    }



    if( m_nQuote == xDiscount && nPeriods > 0 )

        gYield /= 1 + gYield / nPeriods;



    return true;

}





//

//   i n v e r s e L i n e a r

//



bool tInterest::inverseLinear( tHeap<double>& Fraction, int nPow0,

    double gValue, int nPeriods, double& gYield ) const



{

    MTG_ASSERT( Fraction.numOfElems() > 0 );



    if( nPeriods <= 0 ) {

            // continuous compounding is the same for both types

        return inverseExponential( Fraction, nPow0, gValue, nPeriods,

            gYield );

    }



        // Some cases can be handled directly:



    if( Fraction.numOfElems() == 1 ) {

        if( nPow0 != 1 )

            gValue = pow( gValue, 1.0 / nPow0 );



        switch( m_nQuote ) {

            case xDiscount :

                    // gYield is to be interpreted as discount!

                gYield = ( gValue - 1 ) / ( gValue * Fraction[0] );

                break;

        

            case xYield :

                gYield = ( gValue - 1 ) / Fraction[0];

                break;



            default :

                throw tException( INTERNAL_ERROR );

        }

    }

    else

    if( Fraction.numOfElems() == 2 && nPow0 == 1 ) {

        double a = Fraction[0] * Fraction[1];

        double b = Fraction[0] + Fraction[1];

        double c = 1 - gValue;



        if( m_nQuote == xDiscount ) {

            b = -b;

            c = -c / gValue;

        }

        gYield = ( sqrt( b * b - 4 * a * c ) - b ) / ( 2 * a );

    }

    else {

            // All other cases:

            // First find upper bound guess: to do that, find

            // the smallest fraction if gValue >= 1, and the

            // largest fraction if not. This will lead to max

            // yield/discount.



        double f = Fraction[0];



        if( gValue >= 1 ) {

            for( int i = 1; i < Fraction.numOfElems(); ++i ) {

                if( Fraction[i] < f )

                    f = Fraction[i];

            }

        }

        else {

            for( int i = 1; i < Fraction.numOfElems(); ++i ) {

                if( Fraction[i] > f )

                    f = Fraction[i];

            }

        }



            // The first fraction may appear more than once.

        int n = Fraction.numOfElems() + nPow0 - 1;

            

        switch( m_nQuote ) {

            case xDiscount :

                gYield = ( 1 - pow( gValue, -1.0 / n ) ) / f;

                break;

        

            case xYield :

                gYield = ( pow( gValue, 1.0 / n ) - 1 ) / f;



                if( gYield > 100 ) {

                        // safeguard against overflow in

                        // pathetic cases

                    double f = Fraction[0] * nPow0;

                    for( int i = 1; i < Fraction.numOfElems(); ++i )

                        f += Fraction[i];

                    if( f * gYield >= log( DBL_MAX ) )

                        gYield = log( DBL_MAX ) / f;

                }

                break;



            default :

                throw tException( INTERNAL_ERROR );

        }



            // Now refine upper bound guess with Newton-Raphson.



        int nIter = 0;

        double gLastError = 0;



        while( true ) {

            double gGradient;

            double gNewValue =

                accrueLinear( Fraction, nPow0, gYield, nPeriods, &gGradient );



            double gNewYield = gYield - ( gNewValue - gValue ) / gGradient;

            double gError = fabs( gNewValue / gValue - 1 );



            ++nIter;



            if( gLastError > 0 &&

                gError >= gLastError &&

                nIter > m_nMaxIter / 2 ) {

                    // No improvement; be content with last guess.

                    // (This escape only after a considerable number

                    // of iterations.)

                break;

            }



            if( gError < 1e-10 ) {

                    // Be content with 10 digits accuracy.

                gYield = gNewYield;

                break;

            }

        

            if( nIter >= m_nMaxIter )

                return false;



            gLastError = gError;

            gYield = gNewYield;

        }

    }



    return true;

}





//

//   i n v e r s e

//



bool tInterest::inverse( tHeap<double>& Fraction, int nPow0,

    double gValue, int nPeriods, double& gYield ) const



{

    switch( m_nType ) {

        case xExponential :    

            return inverseExponential( Fraction, nPow0, gValue,

                nPeriods, gYield );



        case xLinear :

            return inverseLinear( Fraction, nPow0, gValue, 

                nPeriods, gYield );



        default :

            throw tException( INTERNAL_ERROR );

    }

    return false;

}





//

//   t I n t e r e s t

//



tInterest::tInterest()



{

    init( xExponential, xYield, DayCountACT_365 );

}





//

//   t I n t e r e s t

//



tInterest::tInterest( tType nType, tQuote nQuote,

    const tDayCount& DayCount )



{

    init( nType, nQuote, DayCount );

}





//

//   t I n t e r e s t

//



tInterest::tInterest( tType nType, tQuote nQuote,

    const tDayCount& DayCount, const tCompounder& Compounder )



{

    init( nType, nQuote, DayCount );

    m_pCompounder = Compounder.clone();

}





//

//   t I n t e r e s t

//



tInterest::tInterest( const tInterest& Interest )



{

    init( xExponential, xYield, DayCountACT_365 );

    copyFrom( Interest );

}





//

//   ~ t I n t e r e s t

//



tInterest::~tInterest()



{

    cleanup();

}





//

//   o p e r a t o r =

//



tInterest& tInterest::operator=( const tInterest& Interest )



{

    if( &Interest != this )

        copyFrom( Interest );

    return *this;

}





//

//   s e t

//



void tInterest::set( tType nType, tQuote nQuote, const tDayCount& DayCount )



{

    cleanup();



    m_nType = nType;

    m_nQuote = nQuote;

    m_DayCount = DayCount;

}





//

//   s e t

//



void tInterest::set( tType nType, tQuote nQuote, const tDayCount& DayCount,

    const tCompounder& Compounder )



{

    cleanup();



    m_nType = nType;

    m_nQuote = nQuote;

    m_DayCount = DayCount;

    m_pCompounder = Compounder.clone();

}





//

//   s e t

//



void tInterest::set( tType nType )



{

    m_nType = nType;

}





//

//   s e t

//



void tInterest::set( tQuote nQuote )



{

    m_nQuote = nQuote;

}





//

//   s e t

//



void tInterest::set( const tDayCount& DayCount )



{

    m_DayCount = DayCount;

}





//

//   s e t

//



void tInterest::set( const tCompounder& Compounder )



{

    cleanup();

    m_pCompounder = Compounder.clone();

}





//

//   h a s C o m p o u n d i n g D a t e s

//



bool tInterest::hasCompoundingDates() const



{

        // there are no explicit compounding dates if

        // it is continuous compounding or if it's a

        // period compounder

    if( m_pCompounder == 0 ||

        dynamic_cast<tPeriodCompounder*>( m_pCompounder ) != 0 ) {

        return false;

    }

    return true;

}





//

//   i s C o n t i n u o u s C o m p o u n d i n g

//



bool tInterest::isContinuousCompounding() const



{

    if( m_pCompounder == 0 || m_pCompounder->numOfPeriodsPerYear() <= 0 )

        return true;

    return false;

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( tDate Start, tDate End, double gYield ) const

    

{

    return 1 / futureValue( Start, End, gYield, m_pCompounder );

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( tDate Start, tDate End,

    double gYield, const tCompounder* pCompounder ) const



{

    return 1 / futureValue( Start, End, gYield, pCompounder );

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( tDate Start, tDate End,

    double gYield, int nNumOfPeriodsPerYear ) const



{

    return 1 / futureValue( Start, End, gYield, nNumOfPeriodsPerYear );

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( double gOverallFraction,

    double gYield ) const

    

{

    return 1 / futureValue( gOverallFraction, gYield, m_pCompounder );

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( double gOverallFraction,

    double gYield, const tCompounder* pCompounder ) const



{

    return 1 / futureValue( gOverallFraction, gYield, pCompounder );

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( double gOverallFraction,

    double gYield, int nNumOfPeriodsPerYear ) const



{

    return 1 / futureValue( gOverallFraction, gYield,

        nNumOfPeriodsPerYear );

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( tDate Start, tDate End,

    double gYield, double& gGradient ) const



{

    double gValue =

        futureValue( Start, End, gYield, gGradient, m_pCompounder );



    gGradient /= -gValue * gValue;

    return 1 / gValue;

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( tDate Start, tDate End, double gYield,

    double& gGradient, const tCompounder* pCompounder ) const



{

    double gValue =

        futureValue( Start, End, gYield, gGradient, pCompounder );



    gGradient /= -gValue * gValue;

    return 1 / gValue;

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( tDate Start, tDate End, double gYield,

    double& gGradient, int nNumOfPeriodsPerYear ) const



{

    double gValue =

        futureValue( Start, End, gYield, gGradient, nNumOfPeriodsPerYear );



    gGradient /= -gValue * gValue;

    return 1 / gValue;

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( double gOverallFraction, double gYield,

    double& gGradient ) const



{

    double gValue =

        futureValue( gOverallFraction, gYield, gGradient, m_pCompounder );



    gGradient /= -gValue * gValue;

    return 1 / gValue;

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( double gOverallFraction, double gYield,

    double& gGradient, const tCompounder* pCompounder ) const



{

    double gValue =

        futureValue( gOverallFraction, gYield, gGradient, pCompounder );



    gGradient /= -gValue * gValue;

    return 1 / gValue;

}





//

//   p r e s e n t V a l u e

//



double tInterest::presentValue( double gOverallFraction, double gYield,

    double& gGradient, int nNumOfPeriodsPerYear ) const



{

    double gValue = futureValue( gOverallFraction, gYield, gGradient,

                        nNumOfPeriodsPerYear );



    gGradient /= -gValue * gValue;

    return 1 / gValue;

}





//

//   i n v P r e s e n t V a l u e

//



double tInterest::invPresentValue( tDate Start, tDate End,

    double gValue ) const



{

    MTG_ASSERT( gValue > 0 );

    return invFutureValue( Start, End, 1 / gValue, m_pCompounder );

}





//

//   i n v P r e s e n t V a l u e

//



double tInterest::invPresentValue( tDate Start, tDate End,

    double gValue, const tCompounder* pCompounder ) const



{

    MTG_ASSERT( gValue > 0 );

    return invFutureValue( Start, End, 1 / gValue, pCompounder );

}





//

//   i n v P r e s e n t V a l u e

//



double tInterest::invPresentValue( tDate Start, tDate End,

    double gValue, int nNumOfPeriodsPerYear ) const



{

    MTG_ASSERT( gValue > 0 );

    return invFutureValue( Start, End, 1 / gValue, nNumOfPeriodsPerYear );

}





//

//   i n v P r e s e n t V a l u e

//



double tInterest::invPresentValue( double gOverallFraction,

    double gValue ) const



{

    MTG_ASSERT( gValue > 0 );

    return invFutureValue( gOverallFraction, 1 / gValue, m_pCompounder );

}





//

//   i n v P r e s e n t V a l u e

//



double tInterest::invPresentValue( double gOverallFraction,

    double gValue, const tCompounder* pCompounder ) const



{

    MTG_ASSERT( gValue > 0 );

    return invFutureValue( gOverallFraction, 1 / gValue, pCompounder );

}





//

//   i n v P r e s e n t V a l u e

//



double tInterest::invPresentValue( double gOverallFraction,

    double gValue, int nNumOfPeriodsPerYear ) const



{

    MTG_ASSERT( gValue > 0 );

    return invFutureValue( gOverallFraction,

                1 / gValue, nNumOfPeriodsPerYear );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( tDate Start, tDate End,

    double gYield ) const



{

    return futureValue( Start, End, gYield, m_pCompounder );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( tDate Start, tDate End,

    double gYield, const tCompounder* pCompounder ) const



{

    MTG_ASSERT( Start <= End );



    tHeap<double> Fraction;

    int nPow0, nPeriods;



    findFractions( Start, End, pCompounder, Fraction, nPow0, nPeriods );

    return accrue( Fraction, nPow0, gYield, nPeriods );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( tDate Start, tDate End,

    double gYield, int nNumOfPeriodsPerYear ) const



{

    tPeriodCompounder Compounder( nNumOfPeriodsPerYear );

    return futureValue( Start, End, gYield, &Compounder );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( double gOverallFraction,                                                           

    double gYield ) const



{

    return futureValue( gOverallFraction, gYield, m_pCompounder );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( double gOverallFraction, double gYield,

    const tCompounder* pCompounder ) const



{

    MTG_ASSERT( gOverallFraction >= 0 );



    tHeap<double> Fraction;

    int nPow0, nPeriods;



    findFractions( gOverallFraction, pCompounder,

        Fraction, nPow0, nPeriods );

    return accrue( Fraction, nPow0, gYield, nPeriods );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( double gOverallFraction, double gYield,

    int nNumOfPeriodsPerYear ) const



{

    tPeriodCompounder Compounder( nNumOfPeriodsPerYear );

    return futureValue( gOverallFraction, gYield, &Compounder );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( tDate Start, tDate End,

    double gYield, double& gGradient ) const



{

    return futureValue( Start, End, gYield, gGradient, m_pCompounder );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( tDate Start, tDate End,

    double gYield, double& gGradient, const tCompounder* pCompounder ) const



{

    MTG_ASSERT( Start <= End );



    tHeap<double> Fraction;

    int nPow0, nPeriods;



    findFractions( Start, End, pCompounder, Fraction, nPow0, nPeriods );

    return accrue( Fraction, nPow0, gYield, nPeriods, &gGradient );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( tDate Start, tDate End, double gYield,

    double& gGradient, int nNumOfPeriodsPerYear ) const



{

    tPeriodCompounder Compounder( nNumOfPeriodsPerYear );

    return futureValue( Start, End, gYield, gGradient, &Compounder );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( double gOverallFraction, double gYield,

    double& gGradient ) const



{

    return futureValue( gOverallFraction, gYield,

                gGradient, m_pCompounder );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( double gOverallFraction, double gYield,

    double& gGradient, const tCompounder* pCompounder ) const



{

    MTG_ASSERT( gOverallFraction >= 0 );



    tHeap<double> Fraction;

    int nPow0, nPeriods;



    findFractions( gOverallFraction,

        pCompounder, Fraction, nPow0, nPeriods );

    return accrue( Fraction, nPow0, gYield, nPeriods, &gGradient );

}





//

//   f u t u r e V a l u e

//



double tInterest::futureValue( double gOverallFraction, double gYield,

    double& gGradient, int nNumOfPeriodsPerYear ) const



{

    tPeriodCompounder Compounder( nNumOfPeriodsPerYear );

    return futureValue( gOverallFraction, gYield, gGradient, &Compounder );

}





//

//   i n v F u t u r e V a l u e

//



double tInterest::invFutureValue( tDate Start, tDate End,

    double gValue ) const



{

    return invFutureValue( Start, End, gValue, m_pCompounder );

}





//

//   i n v F u t u r e V a l u e

//



double tInterest::invFutureValue( tDate Start, tDate End,

    double gValue, const tCompounder* pCompounder ) const



{

    MTG_ASSERT( Start <= End && gValue > 0 );



    tHeap<double> Fraction;



    int nPow0, nPeriods;



    findFractions( Start, End, pCompounder, Fraction, nPow0, nPeriods );



    int j = 0;

    for( int i = 0; i < Fraction.numOfElems(); ++i ) {

        if( Fraction[i] != 0 )

            Fraction[j++] = Fraction[i];

    }



    if( j == 0 )

        throw tException( DIVISION_BY_ZERO );

    Fraction.numOfElems( j );



    double gYield;

    if( ! inverse( Fraction, nPow0, gValue, nPeriods, gYield ) )

        throw tException( NO_CONVERGENCE );



    return gYield;

}





//

//   i n v F u t u r e V a l u e

//



double tInterest::invFutureValue( tDate Start, tDate End,

    double gValue, int nNumOfPeriodsPerYear ) const



{

    tPeriodCompounder Compounder( nNumOfPeriodsPerYear );

    return invFutureValue( Start, End, gValue, &Compounder );

}





//

//   i n v F u t u r e V a l u e

//



double tInterest::invFutureValue( double gOverallFraction,

    double gValue ) const



{

    return invFutureValue( gOverallFraction, gValue, m_pCompounder );

}





//

//   i n v F u t u r e V a l u e

//



double tInterest::invFutureValue( double gOverallFraction,

    double gValue, const tCompounder* pCompounder ) const



{

    MTG_ASSERT( gOverallFraction >= 0 && gValue > 0 );



    tHeap<double> Fraction;



    int nPow0, nPeriods;



    findFractions( gOverallFraction,

        pCompounder, Fraction, nPow0, nPeriods );



    int j = 0;

    for( int i = 0; i < Fraction.numOfElems(); ++i ) {

        if( Fraction[i] != 0 )

            Fraction[j++] = Fraction[i];

    }



    if( j == 0 )

        throw tException( DIVISION_BY_ZERO );



    Fraction.numOfElems( j );



    double gYield;

    if( ! inverse( Fraction, nPow0, gValue, nPeriods, gYield ) )

        throw tException( NO_CONVERGENCE );



    return gYield;

}





//

//   i n v F u t u r e V a l u e

//



double tInterest::invFutureValue( double gOverallFraction, double gValue,

    int nNumOfPeriodsPerYear ) const



{

    tPeriodCompounder Compounder( nNumOfPeriodsPerYear );

    return invFutureValue( gOverallFraction, gValue, &Compounder );

}



MTG_END_NAMESPACE





//#define _TEST

#if defined(_TEST)



#if defined(_WIN32)

    #include <conio.h>

#else

    #define getche getchar

#endif



MTG_USING_NAMESPACE



//

//   m a i n

//



void main( int argc, char *argv[] )



{

    printf( "\nTest tInterest\n\n" );



    char sBuf[256];

    int cCmd, nPeriods;

    double gYield, gValue;



    tCouponCompounder Compounder;

    tDayCount DayCount = DayCountACT_365;

    tInterest::tType nType = tInterest::xExponential;

    tInterest::tQuote nQuote = tInterest::xYield;

    tInterest Interest( nType, nQuote, DayCount );



    tDate Maturity;

    tDate Start( 1, 1, 95 );

    tDate End( 1, 1, 96 );



    double gFraction = 1;



    bool bGo = true;

    bool bWithCompounder = false;



    while( bGo ) { 

        printf( "<D>ayct <C>ompd <T>ype <Q>uote P<e>eriod <F>V <P>V "

                "<I>nvFV I<n>vPV E<x>it: " );

        cCmd = getche();

        printf( "\n" );



        switch( cCmd ) {

            case 'd' :

            case 'D' :

                printf( "1=ACT/ACT     2=ACT/365   3=ACT/365NL\n"

                        "4=ACT/365ISDA 5=ACT/360   6=30/360ISDA\n"

                        "7=30/360PSA   8=30/360SIA 9=30E/360\n" 

                        "Choose: " );

                cCmd = getche();

                printf( "\n" );

                switch( cCmd ) {

                    case '1' : DayCount = DayCountACT_ACT;      break;

                    case '2' : DayCount = DayCountACT_365;      break;

                    case '3' : DayCount = DayCountACT_365_NL;   break;

                    case '4' : DayCount = DayCountACT_365_ISDA; break;

                    case '5' : DayCount = DayCountACT_360;      break;

                    case '6' : DayCount = DayCount30_360_ISDA;  break;

                    case '7' : DayCount = DayCount30_360_PSA;   break;

                    case '8' : DayCount = DayCount30_360_SIA;   break;

                    case '9' : DayCount = DayCount30E_360;      break;

                }

                break;



            case 'c' :

            case 'C' :

                printf( "Maturity: " );

                gets( sBuf );

                if( Maturity.set( sBuf ) == 0 ) {

                    printf( "Compounder off\n" );

                    bWithCompounder = false;

                }

                else {

                    printf( "Periods:  " );

                    gets( sBuf );

                    sscanf( sBuf, "%d", &nPeriods );

                    Compounder.set( Maturity, nPeriods );

                    bWithCompounder = true;

                }

                break;



            case 't' :

            case 'T' :

                if( nType == tInterest::xExponential ) {

                    nType = tInterest::xLinear;

                    printf( "Type is now linear\n" );

                }

                else {

                    nType = tInterest::xExponential;

                    printf( "Type is now exponential\n" );

                }

                break;



            case 'q' :

            case 'Q' :

                if( nQuote == tInterest::xDiscount ) {

                    nQuote = tInterest::xYield;

                    printf( "Quote is now yield\n" );

                }

                else {

                    nQuote = tInterest::xDiscount;

                    printf( "Quote is now discount\n" );

                }

                break;



            case 'e' :

            case 'E' :

                printf( "Start or fraction: " );

                gets( sBuf );

                if( Start.set( sBuf ) == 0 ) {

                    sscanf( sBuf, "%lg", &gFraction );

                }

                else {

                    printf( "End: " );

                    gets( sBuf );

                    if( sBuf[0] == '+' ) {

                        int n;



                        sscanf( sBuf, "%d", &n );

                        End = Start + n;

                    }

                    else {

                        if( End.set( sBuf ) == 0 )

                            printf( "Format error\n" );

                    }

                }

                break;



            case 'f' :

            case 'F' :

            case 'p' :

            case 'P' :

                if( bWithCompounder ) {

                    Interest.set( nType, nQuote, DayCount, Compounder );

                }

                else {

                    printf( "Periods per year: " );

                    gets( sBuf );

                    sscanf( sBuf, "%d", &nPeriods );

                    Interest.set( nType, nQuote, DayCount );

                }

                if( nQuote == tInterest::xYield )

                    printf( "Yield: " );

                else

                    printf( "Discount: " );

                gets( sBuf );

                sscanf( sBuf, "%lg", &gYield );

                if( cCmd == 'f' || cCmd == 'F' ) {

                    printf( "Future value of $1 is %.8lg\n",

                        bWithCompounder ?

                            Interest.futureValue( Start, End, gYield ) :

                            Interest.futureValue( Start, End, gYield, nPeriods ) );

                }

                else {

                    printf( "Present value of $1 is %.8lg\n",

                        bWithCompounder ?

                            Interest.presentValue( Start, End, gYield ) :

                            Interest.presentValue( Start, End, gYield, nPeriods ) );

                }

                break;



            case 'i' :

            case 'I' :

            case 'n' :

            case 'N' :

                if( bWithCompounder ) {

                    Interest.set( nType, nQuote, DayCount, Compounder );

                }

                else {

                    printf( "Periods per year: " );

                    gets( sBuf );

                    sscanf( sBuf, "%d", &nPeriods );

                    Interest.set( nType, nQuote, DayCount );

                }

                printf( "Value: " );

                gets( sBuf );

                sscanf( sBuf, "%lg", &gValue );

                if( cCmd == 'i' || cCmd == 'I' ) {

                    if( bWithCompounder )

                        gYield = Interest.invFutureValue( Start, End, gValue );

                    else

                        gYield = Interest.invFutureValue( Start, End, gValue, nPeriods );

                }

                else {

                    if( bWithCompounder )

                        gYield = Interest.invPresentValue( Start, End, gValue );

                    else

                        gYield = Interest.invPresentValue( Start, End, gValue, nPeriods );

                }

                if( nQuote == tInterest::xYield )

                    printf( "Yield is %lg\n", gYield );

                else

                    printf( "Discount is %lg\n", gYield );

                break;



            case 'x' :

            case 'X' :

                bGo = false;

                break;

        }

    }



#if ! defined(_WIN32)

    printf( "\n" );

#endif

}



#endif