// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgJobPool.h"

#include "MtgLog.h"



#if defined(_WIN32)

    #include <process.h>

#endif



MTG_BEGIN_NAMESPACE





//

//   c o n s t a n t s

//



const int tJobPool::m_nStdNumOfJobs = 4;





//

//   s t i l l A c t i v e

//



bool tJobPool::tJob::stillActive()



{

    if( ! m_bActive )

        return false;



#if defined(_WIN32)



    DWORD nCode;



    if( ! GetExitCodeProcess( m_Info.hProcess, &nCode ) ||

        nCode != STILL_ACTIVE ) {

        CloseHandle( m_Info.hProcess );

        m_bActive = false;

        return false;

    }

    return true;



#else

    return false;

#endif

}





//

//   t e r m i n a t e

//



void tJobPool::tJob::terminate( tLog* pLog )



{

    if( m_sGroup != 0 ) {

        delete m_sGroup;

        m_sGroup = 0;

    }



    if( ! stillActive() )

        return;



    m_bActive = false;



    if( pLog != 0 )

        pLog->putNormal( "[Jobpool] Preemptively teminated process" );



#if defined(_WIN32)

    TerminateProcess( m_Info.hProcess, 0 );

#endif

}





//

//   l o c k

//



void tJobPool::lock()



{

#if defined(_WIN32)

    EnterCriticalSection( &m_Lock );

#endif

}





//

//   u n l o c k

//



void tJobPool::unlock()



{

#if defined(_WIN32)

    LeaveCriticalSection( &m_Lock );

#endif

}





//

//   c l e a n u p

//



void tJobPool::cleanup()



{

    lock();



    for( int i = 0; i < m_Job.numOfElems(); ++i )

        m_Job[i].terminate();



    unlock();

}





//

//   g e t S l o t B y G r o u p

//



int tJobPool::getSlotByGroup( const char* sGroup ) const



{

    if( sGroup == 0 )

        return -1;



    for( int i = 0; i < m_Job.numOfElems(); ++i ) {

        if( m_Job[i].m_sGroup != 0 &&

            strcmp( m_Job[i].m_sGroup, sGroup ) == 0 ) {

            return i;

        }

    }



    return -1;

}





//

//   g e t N o n A c t i v e S l o t

//



int tJobPool::getNonActiveSlot( bool bCheck ) const



{

    for( int i = 0; i < m_Job.numOfElems(); ++i ) {

        if( ! m_Job[i].m_bActive )

            return i;

        if( bCheck && ! m_Job[i].stillActive() )

            return i;

    }

    return -1;

}





//

//   ~ t J o b P o o l

//



tJobPool::~tJobPool()



{

    cleanup();

}





//

//   t J o b P o o l

//



tJobPool::tJobPool( int nNumOfJobs )



{

    m_nRefCnt = 1;



    if( nNumOfJobs < 0 )

        nNumOfJobs = m_nStdNumOfJobs;



    m_Job.numOfElems( nNumOfJobs );

    for( int i = 0; i < m_Job.numOfElems(); ++i ) {

        m_Job[i].m_bActive = false;

        m_Job[i].m_sGroup = 0;

    }



#if defined(_WIN32)

    InitializeCriticalSection( &m_Lock );

#endif

}





//

//   u p R e f

//



void tJobPool::upRef()



{

    lock();

    ++m_nRefCnt;

    unlock();

}





//

//   d o w n R e f

//



void tJobPool::downRef()



{

    lock();

    int n = --m_nRefCnt;

    unlock();



    MTG_ASSERT( n >= 0 );

    if( n == 0 )

        delete this;

}





//

//   d i e

//



void tJobPool::die()



{

    downRef();

}





//

//   c r e a t e J o b

//



tRetCode tJobPool::createJob( const char* sGroup, const char* sDir,

    const char* sProgram, const char* sCmdLine, tLog* pLog )



{

    lock();



    int nIndex = getSlotByGroup( sGroup );



    if( nIndex == -1 )

        nIndex = getNonActiveSlot( false );

    if( nIndex == -1 )

        nIndex = getNonActiveSlot( true );



    if( nIndex == -1 ) {

            // no slot available

        unlock();

        return FORK_ERROR;

    }



    tJob& J = m_Job[nIndex];



    J.terminate( pLog );



#if defined(_WIN32)



    STARTUPINFO SInfo;



    memset( &SInfo, 0, sizeof(SInfo) );

    SInfo.cb = sizeof(SInfo);



    char* s;

    

    if( sProgram == 0 ) {

        s = StrCopy( sCmdLine );

    }

    if( sCmdLine == 0 ) {

        s = 0;

    }

    else {

        s = new char[strlen( sProgram ) + strlen( sCmdLine ) + 2];

        sprintf( s, "%s %s", sProgram, sCmdLine );

    }



    if( ! CreateProcess( sProgram, s, NULL, NULL, FALSE,

            DETACHED_PROCESS | CREATE_NEW_PROCESS_GROUP,

            NULL, sDir, &SInfo, &J.m_Info ) ) {

        delete s;

        unlock();

        return FORK_ERROR;

    }

    delete s;



    J.m_bActive = true;

    J.m_sGroup = StrCopy( sGroup );



    unlock();

    return OK;



#else

    return FORK_ERROR;

#endif

}



MTG_END_NAMESPACE



