// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgJumpBag.h"



MTG_BEGIN_NAMESPACE





//

//   i n i t

//

    

void tJumpBag::init()



{

    m_pRefSig = 0;

    m_nCoord = -1;

    m_nLeft = 0;

    m_nRight = 0;

    m_nSkip = 0;

    m_bUseControl = false;

    m_Node = 0;

    m_nNumOfNodes = 0;

    m_nUsedNodes = 0;

}





//

//   c l e a n u p

//



void tJumpBag::cleanup()



{

    if( m_nNumOfNodes > 0 ) {

        delete[] m_Node;

        m_nNumOfNodes = 0;

        m_Node = 0;

    }

}





//

//   t J u m p B a g

//



tJumpBag::tJumpBag()



{

    init();

}





//

//   t J u m p B a g

//



tJumpBag::tJumpBag( const tJumpBag& Bag )



{

    init();

    copyFrom( Bag );

}





//

//   ~ t J u m p B a g

//



tJumpBag::~tJumpBag()



{

    cleanup();

}





//

//   o p e r a t o r =

//



tJumpBag& tJumpBag::operator=( const tJumpBag& Bag )



{

    if( &Bag != this )

        copyFrom( Bag );

    return *this;

}





//

//   c o p y F r o m

//



void tJumpBag::copyFrom( const tJumpBag& Bag )



{

    if( &Bag == this )

        return;



    m_pRefSig = Bag.m_pRefSig;



    m_nCoord = Bag.m_nCoord;

    m_nLeft = Bag.m_nLeft;

    m_nRight = Bag.m_nRight;

    m_nSkip = Bag.m_nSkip;

    m_bUseControl = Bag.m_bUseControl;

    m_nUsedNodes = Bag.m_nUsedNodes;



    if( m_nNumOfNodes != Bag.m_nNumOfNodes ) {

        if( m_nNumOfNodes > 0 )

            delete[] m_Node;

        m_nNumOfNodes = Bag.m_nNumOfNodes;

        if( m_nNumOfNodes > 0 )

            m_Node = new tNode[m_nNumOfNodes];

        else

            m_Node = 0;

    }



    for( int i = 0; i < m_nNumOfNodes; ++i )

        m_Node[i] = Bag.m_Node[i];

}





//

//   s e t C o o r d

//



void tJumpBag::setCoord( int nCoord, int nLeft, int nRight,

    bool bUseControl )



{

    m_nCoord = nCoord;

    m_nLeft = nLeft;

    m_nRight = nRight;

    m_nSkip = 0;

    m_bUseControl = bUseControl;

    m_nUsedNodes = m_nRight - m_nLeft + 1;



    if( m_nNumOfNodes < m_nUsedNodes ) {

        const tSignature* pRefSig = m_pRefSig;

        m_pRefSig = 0;

        if( m_nNumOfNodes > 0 )

            delete[] m_Node;

        m_nNumOfNodes = m_nUsedNodes;

        m_Node = new tNode[m_nNumOfNodes];

        setRefSignature( pRefSig );

    }



    if( m_bUseControl ) {

        for( int i = 0; i < m_nUsedNodes; ++i ) {

            m_Node[i].m_Approx.reset();

            m_Node[i].m_Control.reset();

        }

    }

    else {

        for( int i = 0; i < m_nUsedNodes; ++i )

            m_Node[i].m_Approx.reset();

    }

}





//

//   r e s t a r t

//



void tJumpBag::restart()



{

    if( m_bUseControl ) {

        for( int i = m_nSkip; i < m_nUsedNodes; ++i ) {

            m_Node[i].m_Approx.reset();

            m_Node[i].m_Control.reset();

        }

    }

    else {

        for( int i = m_nSkip; i < m_nUsedNodes; ++i )

            m_Node[i].m_Approx.reset();

    }

}





//

//   s h r i n k

//



void tJumpBag::shrink( int nLeft, int nRight )



{

    MTG_ASSERT( nLeft <= nRight );

    MTG_ASSERT( m_nLeft <= nLeft && nRight <= m_nRight );



    m_nUsedNodes = nRight - m_nLeft + 1;

    m_nSkip = nLeft - m_nLeft;

}





//

//   h i d e M a y E x A l t e r n a t i v e s

//



void tJumpBag::hideMayExAlternatives( bool bUnderControl )



{

    if( m_bUseControl ) {

        for( int i = m_nSkip; i < m_nUsedNodes; ++i ) {

            m_Node[i].m_Approx.hideMayExAlternatives( bUnderControl );

            m_Node[i].m_Control.hideMayExAlternatives( bUnderControl );

        }

    }

    else {

        for( int i = m_nSkip; i < m_nUsedNodes; ++i )

            m_Node[i].m_Approx.hideMayExAlternatives( bUnderControl );

    }

}





//

//   s h o w M a y E x A l t e r n a t i v e s

//



void tJumpBag::showMayExAlternatives( bool bUnderControl )



{

    if( m_bUseControl ) {

        for( int i = m_nSkip; i < m_nUsedNodes; ++i ) {

            m_Node[i].m_Approx.showMayExAlternatives( bUnderControl );

            m_Node[i].m_Control.showMayExAlternatives( bUnderControl );

        }

    }

    else {

        for( int i = m_nSkip; i < m_nUsedNodes; ++i )

            m_Node[i].m_Approx.showMayExAlternatives( bUnderControl );

    }

}





//

//   s e t R e f S i g n a t u r e

//



void tJumpBag::setRefSignature( const tSignature* pSig )



{

    if( pSig == m_pRefSig )

        return;



    m_pRefSig = pSig;

    for( int i = 0; i < m_nUsedNodes; ++i ) {

        m_Node[i].m_Approx.setRefSignature( m_pRefSig );

        m_Node[i].m_Control.setRefSignature( m_pRefSig );

    }

}   





//

//   b e g i n R e f i n e

//



bool tJumpBag::beginRefine( bool bPreRefined )



{

    for( int i = m_nSkip; i < m_nUsedNodes; ++i ) {

        bool bRefined = ! m_Node[i].m_Control.hasAlternatives();



        m_Node[i].m_bRefined = bRefined || bPreRefined;

        m_Node[i].m_bMarked = false;



        if( ! bRefined ) {

                // work to do!

            if( bPreRefined ) {

                for( ++i; i < m_nUsedNodes; ++i ) {

                    m_Node[i].m_bRefined = true;

                    m_Node[i].m_bMarked = false;

                }

            }

            else {

                for( ++i; i < m_nUsedNodes; ++i ) {

                    m_Node[i].m_bRefined =

                        ! m_Node[i].m_Control.hasAlternatives();

                    m_Node[i].m_bMarked = false;

                }

            }



            m_Pos.numOfElems( 1 );

            m_nLeftJumpSig = -1;

            m_nRightJumpSig = -1;

            m_bForward = true;

            m_nBacktrack = 0;

            m_nCurNode = m_nSkip;



            return true;

        }

    }

    return false;

}





//

//   e n d R e f i n e

//



void tJumpBag::endRefine( tRefine& Ref, bool bPreRefined )



{

    if( bPreRefined )

        return;



    const tSignature* pAltSig;



    for( int i = m_nSkip; i < m_nUsedNodes; ++i ) {

        if( ! m_Node[i].m_bRefined && m_Node[i].m_Approx.hasAlternatives() ) {

            m_Pos[m_nCoord] = m_nLeft + i;

            Ref.refine( m_Node[i].m_Approx, m_Pos, pAltSig, false );

        }

    }

}





//

//   f i n d J u m p s

//



bool tJumpBag::findJumps()



{

    if( m_nLeftJumpSig < 0 ) {

        if( m_nRightJumpSig >= 0 )

            m_nLeftJumpSig = 1 - m_nRightJumpSig;

        else

            m_nLeftJumpSig = 0;



        if( m_nCurNode == 0 ) {

                // This is a heuristic: at the (original) left edge,

                // perturbate the exercise decision only

                // if we can assume that there is currently

                // no break throughout in that decision.

            m_bLeftJump =

                m_Node[0].m_Approx.jumpForceExAnd( 

                    m_Node[m_nUsedNodes - 1].m_Approx,

                    m_JumpSig[m_nLeftJumpSig] );

        }

        else {

            m_bLeftJump =

                m_Node[m_nCurNode].m_Approx.jumpForceExXor( 

                    m_Node[m_nCurNode - 1].m_Approx,

                    m_JumpSig[m_nLeftJumpSig] );

        }

    }



    if( m_nRightJumpSig < 0 ) {

        m_nRightJumpSig = 1 - m_nLeftJumpSig;



        if( m_nCurNode == m_nRight - m_nLeft ) {

                // This is the same heuristic. Note that we don't

                // compare with m_nUsedNodes here, since this

                // quantity might have been reduced in a call

                // to shrink(), and we are only interested in

                // the original right edge here.

            m_bRightJump =

                m_Node[m_nCurNode].m_Approx.jumpForceExAnd( 

                    m_Node[0].m_Approx,

                    m_JumpSig[m_nRightJumpSig] );

        }

        else {

            m_bRightJump =

                m_Node[m_nCurNode].m_Approx.jumpForceExXor( 

                    m_Node[m_nCurNode + 1].m_Approx,

                    m_JumpSig[m_nRightJumpSig] );

        }

    }



    return m_bLeftJump || m_bRightJump;

}





//

//   c o m p a r e

//



bool tJumpBag::compare( const tSignature* pAltSig,

    const tSignature& CurSig, tRefine& Ref )



{

    m_Node[m_nCurNode].m_Approx.reset();

    m_Node[m_nCurNode].m_Approx.extendForceExNeg( pAltSig,

        Ref.m_UnderControl );



    if( pAltSig == 0 ) {

        if( CurSig.isEmpty() ) {

            m_bForward = true;

            return true;

        }

    }

    else

        if( *pAltSig == CurSig ) {

            m_bForward = true;

            return true;

        }    



    bool bMark = false;



    if( m_bForward ) {

        if( m_bRightJump && m_nCurNode > m_nSkip ) {

            m_bForward = false;

            bMark = true;

        }

    }

    else {

        if( m_nCurNode == m_nSkip )

            m_bForward = true;

        else

            if( m_bLeftJump ) 

                bMark = true;

    }

    

    if( bMark ) {

        if( m_nCurNode + 1 < m_nUsedNodes ) {

            if( ! m_Node[m_nCurNode + 1].m_bMarked ) {

                m_Node[m_nCurNode + 1].m_bMarked = true;

                ++m_nBacktrack;

            }

        }

        else {

            ++m_nBacktrack;

        }

    }



    m_nRightJumpSig = -1;

    m_nLeftJumpSig = -1;



    return false;

}





//

//   a d v a n c e

//



void tJumpBag::advance( bool bNoAction )



{

    if( m_bForward ) {

        m_nLeftJumpSig = m_nRightJumpSig;

        m_bLeftJump = m_bRightJump;

        m_nRightJumpSig = -1;

        ++m_nCurNode;



        if( bNoAction && m_nBacktrack > 0 ) {

            while( m_nCurNode < m_nUsedNodes ) {

                if( m_Node[m_nCurNode].m_bMarked ) {

                    m_Node[m_nCurNode].m_bMarked = false;

                    --m_nBacktrack;

                    break;

                }

                m_nLeftJumpSig = -1;

                if( ++m_nCurNode == m_nUsedNodes )

                    m_nBacktrack = 0;

            }

        }

    }

    else {

        MTG_ASSERT( m_nCurNode > m_nSkip && ! bNoAction );



        m_nRightJumpSig = m_nLeftJumpSig;

        m_bRightJump = m_bLeftJump;

        m_nLeftJumpSig = -1;

        --m_nCurNode;

    }

}





//

//   r e f i n e

//



void tJumpBag::refine( tRefine& Ref, bool bPreRefined )



{

    if( beginRefine( bPreRefined ) ) {

        tSignature MaySig, CurSig;

        const tSignature* pAltSig;



        while( m_nCurNode < m_nUsedNodes ) {

            tNode& N = m_Node[m_nCurNode];

            bool bNoAction = true;



            if( N.m_bMarked ) {

                MTG_ASSERT( m_bForward && m_nBacktrack > 0 );

                --m_nBacktrack;

                N.m_bMarked = false;

            }



            if( N.m_Control.hasAlternatives() ) {

                if( findJumps() ) {

                    if( m_bLeftJump ) {

                        MaySig = m_JumpSig[m_nLeftJumpSig];

                        if( m_bRightJump )

                            MaySig |= m_JumpSig[m_nRightJumpSig];

                    }

                    else {

                        MaySig = m_JumpSig[m_nRightJumpSig];

                    }

                    N.m_Control.reduceMayEx( MaySig );

                    N.m_Approx.getForceExNeg( CurSig );

                    N.m_Approx.extendMayEx( MaySig, Ref.m_UnderControl );



                    if( N.m_Approx.hasAlternatives() ) {

                        m_Pos[m_nCoord] = m_nLeft + m_nCurNode;

                        Ref.refine( N.m_Approx, m_Pos, pAltSig,

                            ! N.m_bRefined );

                        bNoAction = compare( pAltSig, CurSig, Ref );

                    }

                    else {

                        m_bForward = true;

                    }

                    N.m_bRefined = true;

                }

                else {

                    m_bForward = true;

                }

            }

            else {

                m_bForward = true;

            }



            advance( bNoAction );

        }



        MTG_ASSERT( m_nBacktrack == 0 );



        endRefine( Ref, bPreRefined );

    }

}



MTG_END_NAMESPACE



//#define _TEST

#if defined(_TEST)



#include <conio.h>



MTG_USING_NAMESPACE



//

//   m a i n

//



void main( int argc, char *argv[] )



{

    int cCmd;

    tSignature Sig;

    tJumpBag JumpBag, Backup;

    tHeap<bool> UnderControl;



    printf( "\nTest tJumpBag\n\n" );



    Sig.reset( 8 );

    for( int k = 0; k < Sig.numOfFlags(); ++k )

        Sig.on( k );



    UnderControl.numOfElems( Sig.numOfFlags() );



    JumpBag.setRefSignature( &Sig );

    JumpBag.setCoord( 0, -5, 5 );



    for( MTG_FOR_INIT( int ) k = -5; k <= 5; ++k ) {

        JumpBag.approxSigBag( k ).reset();

        JumpBag.controlSigBag( k ).reset();

    }



    for( MTG_FOR_INIT( int ) k = 0; k < Sig.numOfFlags(); ++k ) {

        printf( "Option %d: <L>ong <S>hort Skip<A>ll: ", k );

        do{ 

            cCmd = getche();

        } while( strchr( "LlSsAa", cCmd ) == NULL );

        printf( "\n" );



        if( cCmd == 'a' || cCmd == 'A' )

            break;



        UnderControl[k] = ( cCmd == 'l' || cCmd == 'L' );



        do{

            printf( "        : <F>orce <M>ay/force M<a>y/dont <S>kip: " );

            do{ 

                cCmd = getche();

            } while( strchr( "FfMmAaSs", cCmd ) == NULL );

            printf( "\n" );



            if( cCmd != 's' && cCmd != 'S' ) {

                int nFrom, nTo;



                printf( "        : from to = " );

                scanf( "%d %d", &nFrom, &nTo );



                for( int i = nFrom; i <= nTo; ++i ) {

                    switch( cCmd ) {

                        case 'f' : case 'F' :

                            JumpBag.approxSigBag( i ).

                                forceExercise( UnderControl[k], k );

                            JumpBag.controlSigBag( i ).

                                forceExercise( UnderControl[k], k );

                            break;



                        case 'm' : case 'M' :

                            JumpBag.approxSigBag( i ).

                                forceExercise( UnderControl[k], k );

                            JumpBag.controlSigBag( i ).

                                mayExercise( UnderControl[k], k );

                            break;



                        case 'a' : case 'A' :

                            JumpBag.controlSigBag( i ).

                                mayExercise( UnderControl[k], k );

                            break;

                    }

                }

            }

        } while( cCmd != 's' && cCmd != 'S' );

    }



    struct tRefine : public tJumpBag::tRefine {

        tSignature m_Alt;

        tJumpBag& m_JumpBag;



        tRefine( const tHeap<bool>& UnderControl, tJumpBag& JumpBag ) 

            : tJumpBag::tRefine( UnderControl ), m_JumpBag( JumpBag ) {

        }



        void refine( tSigBag& Approx, const tHeap<int>& Pos,

                const tSignature*& pAltSig, bool bFirst ) {

            tSigBag& Control = m_JumpBag.controlSigBag( Pos[0] );



            printf( "Refine %d: Control\n", Pos[0] );

            print( Control );

            printf( "          Approx\n" );

            print( Approx );



            if( Approx.numOfAlternatives() == 1 ) {

                Approx.considerAlternative( -1, -1 );

            }

            else {

                int n1, n2;



                printf( "          Alt1, Alt2 = " );

                scanf( "%d %d", &n1, &n2 );

                Approx.considerAlternative( n1, n2 );

            }



            m_Alt = Approx.altSignature();

            pAltSig = &m_Alt;

        }



        void print( tSigBag& SB ) const {

            if( ! SB.hasAlternatives() ) {

                printf( "          No alternatives\n" );

                return;

            }

            for( int j = -1; j < SB.numOfAlternatives( 0 ); ++j ) {

                for( int k = -1; k < SB.numOfAlternatives( 1 ); ++k ) {

                    SB.considerAlternative( j, k );

                    const tSignature& S = SB.altSignature();

                    printf( "          Alt(%d,%d) = ", j, k );

                    for( int i = 0; i < S.numOfFlags(); ++ i )

                        printf( "%d", S[i] );

                    printf( "," );

                    for( i = 0; i < SB.altNumOfExIndexes(); ++i )

                        printf( " %d", SB.altExIndex( i ) );

                    printf( "\n" );

                }

            }

        }

    };



    Backup = JumpBag;

    do{

        printf( "\n" );



        JumpBag = Backup;

        tRefine R( UnderControl, JumpBag );

        JumpBag.refine( R );



        printf( "\n<R>epeat E<x>it: " );

        do{ 

            cCmd = getche();

        } while( strchr( "rRxX", cCmd ) == NULL );

        printf( "\n" );

    } while( cCmd == 'r' || cCmd == 'R' );



    printf( "\n" );

}



#endif

