// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgLog.h"

#include "MtgJobPipe.h"

#include "MtgSocket.h"



#if defined(_WIN32)

#include <share.h>

#include <windows.h>

#endif



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tLog::init()



{

    m_sFileName = 0;

    m_fFile = 0;



#if defined(_WIN32)

    m_nLock = 0;

#endif

}





//

//   l o c k

//



void tLog::lock()



{

#if defined(_WIN32)

    while( InterlockedExchange( &m_nLock, 1 ) == 1 )

        Sleep( 0 );

#endif

}





//

//   u n l o c k

//



void tLog::unlock()



{

#if defined(_WIN32)

    InterlockedExchange( &m_nLock, 0 );

#endif

}





//

//   b e g i n P u t

//



bool tLog::beginPut( bool bNormal )



{

    lock();



    if( m_fFile == 0 ) {

        if( open() != OK ) {

            unlock();

            return false;

        }

    }



    time_t T1;

    struct tm *pT2;



    time( &T1 );

    pT2 = localtime( &T1 );



    fprintf( m_fFile, "%02d-%02d-%02d %02d:%02d:%02d %s ",

        pT2->tm_year, pT2->tm_mon + 1, pT2->tm_mday,

        pT2->tm_hour, pT2->tm_min, pT2->tm_sec,

        bNormal ? "   " : "***" );



    return true;

}





//

//   p u t S o c k e t

//



void tLog::putSocket( const tSocket& Sock )



{

    MTG_ASSERT( m_fFile != 0 );

    fprintf( m_fFile, "[%s] ", Sock.getPeerHostName() );

}





//

//   p u t J o b P i p e

//



void tLog::putJobPipe( const tJobPipe& JobPipe )



{

    MTG_ASSERT( m_fFile != 0 );

    fprintf( m_fFile, "[%s] ", JobPipe.name() );

}





//

//   e n d P u t

//



void tLog::endPut()



{

    MTG_ASSERT( m_fFile != 0 );

    fprintf( m_fFile, "\n" );

    fflush( m_fFile );

    unlock();

}





//

//   t L o g

//



tLog::tLog()



{

    init();

}





//

//   t L o g

//



tLog::tLog( const char* sFileName )



{

    init();

    setFileName( sFileName );

}





//

//   ~ t L o g

//



tLog::~tLog()



{

    close();

    if( m_sFileName != 0 )

        delete m_sFileName;

}





//

//   u s e A r g s

//



tRetCode tLog::useArgs( int& argc, const char* argv[] )



{

    if( argc > 0 ) {

        int k = strlen( argv[0] );

        char* s = new char[k + 5];



        strcpy( s, argv[0] );

        while( k > 0 && s[k - 1] != '.' )

            --k;

        if( k > 0 )

            --k;

        else

            k = strlen( argv[0] );

        strcpy( &s[k], ".log" );

        setFileName( s );

        delete s;

    }



    for( int k = 1; k < argc - 1; ++k ) {

        if( strcmp( argv[k], "-log" ) == 0 ) {

            argv[k] = 0;

            setFileName( argv[++k] );

            argv[k] = 0;

        }

    }



        // Remove used arguments.



    int j = 1;

    for( MTG_FOR_INIT( int ) k = 1; k < argc; ++k ) {

        if( argv[k] != 0 )

            argv[j++] = argv[k];

    }

    argc = j;



    return OK;

}





//

//   s e t F i l e N a m e

//



void tLog::setFileName( const char* sFileName )



{

    close();

    if( m_sFileName != 0 )

        delete m_sFileName;

    if( sFileName != 0 )

        m_sFileName = StrCopy( sFileName );

}





//

//   o p e n

//



tRetCode tLog::open()



{

    close();



    if( m_sFileName == 0 )

        return OPEN_ERROR;



#if defined(_WIN32)

    m_fFile = _fsopen( m_sFileName, "at", _SH_DENYNO );

#else

    m_fFile = fopen( m_sFileName, "at" );

#endif



    if( m_fFile == 0 )

        return OPEN_ERROR;



    return OK;

}





//

//   o p e n

//



tRetCode tLog::open( const char* sFileName )



{

    setFileName( sFileName );

    return open();

}





//

//   c l o s e

//



void tLog::close()



{

    if( m_fFile != 0 ) {

        fclose( m_fFile );

        m_fFile = 0;

    }

}





//

//   p u t N o r m a l

//



void tLog::putNormal( const char* sFmt, ... )



{

    va_list Marker;

    va_start( Marker, sFmt );

    putNormal1( sFmt, Marker );

}





//

//   p u t N o r m a l 1

//



void tLog::putNormal1( const char* sFmt, va_list Marker )



{

    if( ! beginPut( true ) )

        return;

    vfprintf( m_fFile, sFmt, Marker );

    endPut();

}





//

//   p u t E r r o r

//



void tLog::putError( const char* sFmt, ... )



{

    va_list Marker;

    va_start( Marker, sFmt );

    putError1( sFmt, Marker );

}





//

//   p u t E r r o r 1

//



void tLog::putError1( const char* sFmt, va_list Marker )



{

    if( ! beginPut( false ) )

        return;

    vfprintf( m_fFile, sFmt, Marker );

    endPut();

}





//

//   p u t N o r m a l

//



void tLog::putNormal( const tSocket& Sock, const char* sFmt, ... )



{

    va_list Marker;

    va_start( Marker, sFmt );

    putNormal1( Sock, sFmt, Marker );

}





//

//   p u t N o r m a l 1 

//



void tLog::putNormal1( const tSocket& Sock, const char* sFmt, va_list Marker )



{

    if( ! beginPut( true ) )

        return;

    putSocket( Sock );

    vfprintf( m_fFile, sFmt, Marker );

    endPut();

}





//

//   p u t E r r o r

//



void tLog::putError( const tSocket& Sock, const char* sFmt, ... )



{

    va_list Marker;

    va_start( Marker, sFmt );

    putError1( Sock, sFmt, Marker );

}





//

//   p u t E r r o r 1

//



void tLog::putError1( const tSocket& Sock, const char* sFmt, va_list Marker )



{

    if( ! beginPut( false ) )

        return;

    putSocket( Sock );

    vfprintf( m_fFile, sFmt, Marker );

    endPut();

}





//

//   p u t N o r m a l

//



void tLog::putNormal( const tJobPipe& JobPipe, const char* sFmt, ... )



{

    va_list Marker;

    va_start( Marker, sFmt );

    putNormal1( JobPipe, sFmt, Marker );

}





//

//   p u t N o r m a l 1 

//



void tLog::putNormal1( const tJobPipe& JobPipe, 

    const char* sFmt, va_list Marker )



{

    if( ! beginPut( true ) )

        return;

    putJobPipe( JobPipe );

    vfprintf( m_fFile, sFmt, Marker );

    endPut();

}





//

//   p u t E r r o r

//



void tLog::putError( const tJobPipe& JobPipe, const char* sFmt, ... )



{

    va_list Marker;

    va_start( Marker, sFmt );

    putError1( JobPipe, sFmt, Marker );

}





//

//   p u t E r r o r 1

//



void tLog::putError1( const tJobPipe& JobPipe,

    const char* sFmt, va_list Marker )



{

    if( ! beginPut( false ) )

        return;

    putJobPipe( JobPipe );

    vfprintf( m_fFile, sFmt, Marker );

    endPut();

}



MTG_END_NAMESPACE

