// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgModel.h"

#include "MtgBSModel.h"

#include "MtgCalendar.h"

#include "MtgDataScanner.h"

#include "MtgFactor.h"

#include "MtgHJMGaussianModel.h"

#include "MtgParser.h"

#include "MtgVasicekModel.h"



#include <float.h>



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tModel::init()



{

    m_pCalendar = 0;

    m_bHasWeights = false;

    m_gGlobalWeight = 0;

}





//

//   l o a d W e i g h t s

//



tRetCode tModel::loadWeights()



{

    MTG_ASSERT( m_bHasWeights && m_WeightFile.isValid() );



    tRetCode nRet;

    tDataScanner Scanner;



    m_Weight.reset();

    if( ( nRet = Scanner.setSource( m_WeightFile ) ) != OK ||

        ( nRet = Scanner.append( m_Weight, 0 ) ) != OK ) {

        if( nRet == GO ) {

                // file doesn't exist; never mind

            m_bHasWeights = false;

        }

        else {

                // some other error

            return nRet;

        }

    }

    else {

        if( ! Scanner.atEOF() )

            return INVALID_CONSTANT;



        if( m_Weight.numOfElems() == 0 ) {

            if( m_WeightFile.isTry() )

                m_bHasWeights = false;

            else 

                return MISSING_WEIGHTS;

        }

        else {

            if( m_Weight.numOfElems() != numOfInstantiations() )

                return INCONSISTENT_OBJECT;

        }

    }



    return OK;

}





//

//   n o r m a l i z e W e i g h t s

//



void tModel::normalizeWeights()



{

    MTG_ASSERT( m_bHasWeights &&

                m_Weight.numOfElems() == numOfInstantiations() );



    double gSum = 0;



    for( int i = 0; i < m_Weight.numOfElems(); ++i )

        gSum += m_Weight[i];



    if( gSum == 0 )

        return;



    for( MTG_FOR_INIT( int ) i = 0; i < m_Weight.numOfElems(); ++i )

        m_Weight[i] /= gSum;

}





//

//   c o p y F r o m

//



void tModel::copyFrom( const tModel& Model )



{

    if( &Model == this )

        return;



    for( int i = 0; i < m_Factor.numOfElems(); ++i )

        setObjRefToZeroOrSo( m_Factor[i] );

    m_Factor.reset();



    for( MTG_FOR_INIT( int ) i = 0; i < Model.m_Factor.numOfElems(); ++i )

        appendFactor( Model.m_Factor[i] );



    setObjRefToZeroOrSo( m_pCalendar, Model.m_pCalendar );



    m_bHasWeights = Model.m_bHasWeights;

    m_WeightFile = Model.m_WeightFile;

    m_Weight = Model.m_Weight;

    m_gGlobalWeight = Model.m_gGlobalWeight;



    super::copyFrom( Model );

}





//

//   a p p e n d F a c t o r

//



tRetCode tModel::appendFactor( tFactor* pFactor )



{

    for( int i = 0; i < m_Factor.numOfElems(); ++i ) {

        if( m_Factor[i] == pFactor )

            return ATTR_REDEFINITION;

    }

    m_Factor.append( pFactor );

    pFactor->upRef();

    return OK;

}





//

//   p a r s e P r e f i x

//



tRetCode tModel::parsePrefix( tParser& Parser, tParseInfoStub& Info )



{

    m_bHasWeights = false;

    return super::parsePrefix( Parser, Info );

}





//

//   p a r s e P a r a m

//



tRetCode tModel::parseParam( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;

    double gDaysPerYear;

    tCalendar* pCal;

    tObject* pObj;



    switch( Parser.curToken() ) {

        case xTokCalendar :

            if( m_pCalendar != 0 )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() == xTokId ) {

                if( ( pObj = Parser.findObject() ) == 0 ) 

                    return Parser.setError( NOT_FOUND );

                if( ( pCal = dynamic_cast<tCalendar*>( pObj ) ) == 0 )

                    return Parser.setError( "calendar expected" );

                setObjRefToZeroOrSo( m_pCalendar, pCal );

                if( ( nRet = Parser.readToken() ) != OK )

                    return nRet;

            }

            else {

                if( ( nRet = tCalendar::parse( Parser,

                                system(), pObj ) ) != OK ) {

                    return nRet;

                }

                pCal = static_cast<tCalendar*>( pObj );

                setObjRefToZeroOrSo( m_pCalendar, pCal );

            }

            break;



        case xTokDaysPerYear :

            if( m_pCalendar != 0 )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.scanPosDouble( gDaysPerYear ) ) != OK ) {

                return nRet;

            }

            pCal = new tCalendar;

            pCal->setDaysPerYear( gDaysPerYear );

            setObjRefToZeroOrSo( m_pCalendar, pCal );

            break;



        case xTokWeight :

            if( m_bHasWeights )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.beginOfObj() ) {

                if( ( nRet = Parser.scanNonNegTable( m_Weight ) ) != OK )

                    return nRet;

                if( m_Weight.numOfElems() == 0 )

                    return Parser.setError( MISSING_WEIGHTS );

            }

            else {

                if( ( nRet = m_WeightFile.parseRead( Parser ) ) != OK )

                    return nRet;

            }

            m_bHasWeights = true;

            break;



        default :

            return super::parseParam( Parser, Info );

    }



    return OK;

}





//

//   p a r s e P o s t f i x

//



tRetCode tModel::parsePostfix( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;



    if( m_Factor.numOfElems() == 0 )

        return Parser.setError( MISSING_FACTOR );



    if( m_pCalendar == 0 )

        setObjRefToZeroOrSo( m_pCalendar, new tCalendar );



    int n = numOfInstantiations();



    if( n < 0 )

        return Parser.setError( OUT_OF_RANGE );



    m_gGlobalWeight = 1.0 / n;



    if( m_bHasWeights ) {

        if( m_WeightFile.isValid() ) {

            if( ( nRet = loadWeights() ) != OK )

                return Parser.setError( nRet );

        }

        else {

            if( n != m_Weight.numOfElems() )

                return Parser.setError( INCONSISTENT_OBJECT );

        }



        if( m_bHasWeights )

            normalizeWeights();

    }



    return super::parsePostfix( Parser, Info );

}





//

//   t M o d e l

//



tModel::tModel()



{

    init();

}





//

//   t M o d e l

//



tModel::tModel( const tModel& Model )



{

    init();

    copyFrom( Model );

}





//

//   ~ t M o d e l

//



tModel::~tModel()



{

    for( int i = 0; i < m_Factor.numOfElems(); ++i ) {

        m_Factor[i]->downRef();

        if( m_Factor[i]->canDelete() )

            delete m_Factor[i];

    }

    setObjRefToZeroOrSo( m_pCalendar );

}





//

//   f i n a l i z e

//



tRetCode tModel::finalize()



{

    return super::finalize();

}





//

//   m a t c h F a c t o r s

//



tRetCode tModel::matchFactors( const tHeap<tFactor*>& Mirror,

    tHeap<int>& Xlat ) const



{

    Xlat.numOfElems( Mirror.numOfElems() );



    for( int k = 0; k < Mirror.numOfElems(); ++k ) {

        int j = 0;

        

        while( j < m_Factor.numOfElems() && m_Factor[j] != Mirror[k] )

            ++j;



        if( j >= m_Factor.numOfElems() )

            return NOT_FOUND;

        Xlat[k] = j;

    }



    return OK;

}





//

//   d a y U n i t

//



double tModel::dayUnit() const



{

    if( m_pCalendar != 0 )

        return m_pCalendar->dayUnit();

    return tCalendar::stdDayUnit();

}





//

//   s e t I n s t a n t i a t i o n

//



void tModel::setInstantiation( tHeap<int>& Select,

    int nInstantiation ) const



{

    if( nInstantiation != 0 )

        throw tException( INTERNAL_ERROR );

}





//

//   s a v e W e i g h t s H e a d e r

//



tRetCode tModel::saveWeightsHeader( ostream& Out ) const



{

    return Out.good() ? OK : WRITE_ERROR;

}





//

//   s a v e W e i g h t s D a t a

//



tRetCode tModel::saveWeightsData( ostream& Out, tHeap<int> Select ) const



{

    for( int i = 0; i < Select.numOfElems(); ++i ) {

        if( i > 0 )

            Out << " ";

        Out << Select[i];

    }



    return Out.good() ? OK : WRITE_ERROR;

}





//

//   s a v e W e i g h t s

//



tRetCode tModel::saveWeights( ostream& Out,

    const tHeap<double>* pWeight ) const



{

    if( pWeight == 0 )

        return saveWeights( Out );

    return saveWeights( Out, *pWeight );

}





//

//   s a v e W e i g h t s

//



tRetCode tModel::saveWeights( ostream& Out,

    const tHeap<double>& Weight ) const



{

    int n = numOfInstantiations();

    MTG_ASSERT( Weight.numOfElems() == n );



    tRetCode nRet;



    if( ( nRet = saveWeightsHeader( Out ) ) != OK )

        return nRet;



    tHeap<int> Order, Select;



        // sort



    Order.numOfElems( n );



    for( int i = 0; i < n; ++i ) {

        double w = Weight[i];



        int i1 = i;

        while( i1 > 0 && Weight[Order[i1 - 1]] < w ) {

            Order[i1] = Order[i1 - 1];

            --i1;

        }



        Order[i1] = i;

    }



        // output



    Out << setprecision( DBL_DIG );

    for( MTG_FOR_INIT( int ) i = 0; i < n; ++i ) {

        setInstantiation( Select, Order[i] );



        Out << Weight[Order[i]] << " ";

        if( ( nRet = saveWeightsData( Out, Select ) ) != OK )

            return nRet;

        Out << endl;

    }



    return OK;

}





//

//   s a v e W e i g h t s

//



tRetCode tModel::saveWeights( ostream& Out ) const



{

    if( ! m_bHasWeights ) {

        tHeap<double> Weight;



        Weight.numOfElems( numOfInstantiations() );

        Weight.fill( m_gGlobalWeight );

        return saveWeights( Out, Weight );

    }

    return saveWeights( Out, m_Weight );

}





//

//   s a v e W e i g h t s

//



tRetCode tModel::saveWeights( const tFileName& Out,

    const tHeap<double>* pWeight ) const



{

    if( ! Out.isValid() )

        return OK;



    ofstream S;

    tRetCode nRet;



    if( ( nRet = Out.openWrite( S, "weights" ) ) != OK )

        return nRet;



    return saveWeights( S, pWeight );

}





//

//   s a v e W e i g h t s

//



tRetCode tModel::saveWeights( const tFileName& Out,

    const tHeap<double>& Weight ) const



{

    return saveWeights( Out, &Weight );

}





//

//   s a v e W e i g h t s

//



tRetCode tModel::saveWeights( const tFileName& Out ) const



{

    return saveWeights( Out, 0 );

}





//

//   p a r s e

//



tRetCode tModel::parse( tParser& Parser, tSystem& System, tObject*& pObj )



{

    tRetCode nRet;

    tModel* pModel;

    tToken nType;

    tParseInfo Info;



    Info.m_bParam1 = false;

    Info.m_bParam2 = false;



    if( ( nRet = parseType( Parser, nType, xTokBlackScholes ) ) != OK )

        return nRet;



    switch( nType ) {

        case xTokBlackScholes :

            pModel = new tBSModel;

            break;



        case xTokHJMGaussian :

            pModel = new tHJMGaussianModel;

            break;



        case xTokVasicek :

            pModel = new tVasicekModel;

            break;



        default :

            return Parser.setError( INVALID_KEYWORD );

    }



    pModel->setSystem( System );



    if( ( nRet = pModel->super::parse( Parser, &Info ) ) != OK ) {

        delete pModel;

        return nRet;

    }

    pObj = pModel;



    return OK;

}



MTG_END_NAMESPACE