// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgNetServer.h"

#include "MtgEvaluate.h"

#include "MtgLog.h"

#include "MtgNetSource.h"

#include "MtgParser.h"

#include "MtgRepository.h"

#include "MtgSocket.h"

#include "MtgVersion.h"



#if defined(_WIN32)

    #include <process.h>

#endif



MTG_BEGIN_NAMESPACE





//

//   c o n s t a n t s

//



const int tNetServer::m_nStdPort = 12233;





//

//   b a n n e r

//



tRetCode tNetServer::banner( tSocket& Sock )



{

    tRetCode nRet;

    char sBuffer[512];



    sprintf( sBuffer,

        "#Welcome to MtgSvr %d.%d (build %d)\n",

        VersionMajor, VersionMinor, VersionBuild );



    if( ( nRet = Sock.write( sBuffer ) ) != OK )

        return nRet;



    return OK;

}





//

//   p r o t E r r o r

//



tRetCode tNetServer::protError( tSocket& Sock, const char* sFmt, ... )



{

    tRetCode nRet;

    va_list Marker;

    char sBuffer[2048];



    va_start( Marker, sFmt );

    vsprintf( sBuffer, sFmt, Marker );



    if( m_pLog != 0 )

        m_pLog->putError( Sock, "%s", sBuffer );



    if( ( nRet = Sock.write( "#" ) ) != OK ||

        ( nRet = Sock.write( sBuffer ) ) != OK ||

        ( nRet = Sock.write( "\n" ) ) != OK ) {

        return commError( Sock, nRet );

    }



    return OK;

}





//

//   c o m m E r r o r

//



tRetCode tNetServer::commError( tSocket& Sock, tRetCode nRet )



{

    if( m_pLog != 0 ) 

        m_pLog->putError( Sock, "Communication error %d", nRet );

    return nRet;

}





//

//   r e p o r t

//



tRetCode tNetServer::report( tSocket& Sock, tEvaluate* pEvaluate, bool& bSayBye )



{

    tRetCode nRet;

    tHeap2<double> Front;

    char sBuffer[1024];



    sprintf( sBuffer, "%g %g %g",

        pEvaluate->total(), pEvaluate->delta(), pEvaluate->gamma() );



    if( ( nRet = Sock.write( sBuffer ) ) != OK )

        goto end;



    pEvaluate->front( Front );



    if( Front.numOfRows() > 0 ) {

        for( int i = 0; i < Front.numOfRows(); ++i ) {

            sprintf( sBuffer, " %g %g", Front[i][0], Front[i][1] );

            if( ( nRet = Sock.write( sBuffer ) ) != OK )

                goto end;

        }

    }



    if( ( nRet = Sock.write( " end\n" ) ) != OK )

        goto end;



    return OK;



end:



    bSayBye = false;

    return commError( Sock, nRet );

}





//

//   t S e r v e r

//



tNetServer::tNetServer()



{

    m_nPort = m_nStdPort;

    m_pLog = 0;

}





//

//   ~ t S e r v e r

//



tNetServer::~tNetServer()



{

}





//

//   s e t L o g

//



void tNetServer::setLog( tLog* pLog )



{

    m_pLog = pLog;

}





//

//   r u n

//



void tNetServer::run( tSocket& Sock )



{

    tRetCode nRet;



    tRepository Rep;

    tNetSource Source( &Sock );

    tParser Parser;

    tEvaluate* pEvaluate;



    time_t nDur = time( NULL );

    bool bSayBye = true;



    if( m_pLog != 0 )

        m_pLog->putNormal( Sock, "Socket connection opened" );



    Parser.setRepository( Rep );



    if( ( nRet = banner( Sock ) ) != OK ) {

        commError( Sock, nRet );

        bSayBye = false;

        goto end;

    }



    if( ( nRet = Parser.setSource( Source ) ) != OK ) {

        protError( Sock, "Error %d : %s", nRet, Parser.getErrorMsg() );

        goto end;

    }



    while( true ) {

        if( ( nRet = Parser.parseObjects( pEvaluate ) ) != OK ) {

            protError( Sock, "Error %d : %s", nRet, Parser.getErrorMsg() );

            goto end;

        }

                

        if( pEvaluate == 0 )

            break;



        if( ( nRet = pEvaluate->run() ) != OK ) {

            protError( Sock, "Runtime error %d", nRet );

            delete pEvaluate;

            goto end;

        }



        if( ( nRet = report( Sock, pEvaluate, bSayBye ) ) != OK ) {

            delete pEvaluate;

            goto end;

        }



        delete pEvaluate;

    }



end:



    if( bSayBye ) {

        if( ( nRet = Sock.write( "bye\n" ) ) != OK ) {

            commError( Sock, nRet );

        }

        else {

#if defined(_WIN32)

                // Necessary to make sure everything arrives

                // and is processed on the other side.

            Sleep( 1000 );

#else

            sleep( 1 );

#endif

        }

    }



    nDur = time( NULL ) - nDur;

    if( m_pLog != 0 ) {

        m_pLog->putNormal( Sock,

            "Socket connection closed after %02ld:%02ld:%02ld",

            nDur / 3600, nDur / 60 % 60, nDur % 60 );

    }



    Sock.close();

}





//

//   u s e A r g s

//



tRetCode tNetServer::useArgs( int& argc, const char* argv[] )



{

    for( int k = 1; k < argc - 1; ++k ) {

        if( strcmp( argv[k], "-port" ) == 0 ) {

            argv[k] = 0;

            m_nPort = atoi( argv[++k] );

            argv[k] = 0;

        }

    }



        // Remove used arguments.



    int j = 1;

    for( MTG_FOR_INIT( int ) k = 1; k < argc; ++k ) {

        if( argv[k] != 0 )

            argv[j++] = argv[k];

    }

    argc = j;



    return OK;

}





//

//   m a i n

//



tRetCode tNetServer::main()



{

    tSocket Sock;

    tRetCode nRet;



    class tT : public tSocket::tTask { 

    public:

        tNetServer* m_pServer;

        void (tNetServer::*m_func)( tSocket& Sock );

        void run( tSocket& Sock ) {

            (m_pServer->*m_func)( Sock );

        }

    } Task;



    Task.m_pServer = this;



    if( ( nRet = Sock.listen( m_nPort ) ) != OK ) {

        printf( "*** Unable to listen to port %d\n\n", m_nPort );

        return nRet;

    }



    printf( "Accepting requests at port %d\n\n", m_nPort );



    if( m_pLog != 0 ) {

        m_pLog->putNormal( "--- Accepting requests at port %d --- ", 

            m_nPort );

    }



    while( true ) {

        tRetCode nRet;



        Task.m_func = &tNetServer::run;

        if( ( nRet = Sock.multiServerAccept( Task ) ) != OK && m_pLog != 0 )

            m_pLog->putError( "Error %d during multi server accept", nRet );

    }



    return OK;

}



MTG_END_NAMESPACE

