// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgNumericalExpr.h"

#include "MtgDateExpr.h"

#include "MtgEvGenerator.h"

#include "MtgFactor.h"

#include "MtgFDEngine.h"

#include "MtgPortfolio.h"

#include "MtgParser.h"



MTG_BEGIN_NAMESPACE





//

//   v a r i a b l e s

//



const int tNumericalExpr::ALLOW_UNITVALUE = 0x04;

const int tNumericalExpr::ALLOW_FACTOR    = 0x08;





//

//   i n i t

//



void tNumericalExpr::init()



{

    m_nType = xNumber;

    m_sId = 0;

    m_nId = -1;

    m_gValue = 0;

    m_pFactor = 0;



    for( int k = 0; k < sizeof(m_Child) / sizeof(m_Child[0]); ++k )

        m_Child[k] = 0;

}





//

//   c l e a n u p

//



void tNumericalExpr::cleanup()



{

    for( int k = 0; k < sizeof(m_Child) / sizeof(m_Child[0]); ++k ) {

        if( m_Child[k] != 0 ) {

            delete m_Child[k];

            m_Child[k] = 0;

        }

    }



    if( m_sId != 0 ) {

        delete m_sId;

        m_sId = 0;

    }



    if( m_pFactor != 0 ) {

        m_pFactor->downRef();

        if( m_pFactor->canDelete() )

            delete m_pFactor;

        m_pFactor = 0;

    }



    m_nType = xNumber;

    m_nId = -1;

    m_gValue = 0;

}





//

//   c o p y F r o m

//



void tNumericalExpr::copyFrom( tNumericalExpr &Expr )



{

    if( &Expr == this )

        return;



    cleanup();



    m_nType = Expr.m_nType;

    m_nId = Expr.m_nId;

    m_gValue = Expr.m_gValue;



    if( Expr.m_sId != 0 )

        m_sId = StrCopy( Expr.m_sId );



    if( Expr.m_pFactor != 0 ) {

        m_pFactor = Expr.m_pFactor;

        m_pFactor->upRef();

    }



    for( int k = 0; k < 3; ++k ) {

        if( Expr.m_Child[k] != 0 )

            m_Child[k] = Expr.m_Child[k]->clone();

    }

}





//

//   s e t T y p e

//



void tNumericalExpr::setType( tType nType )



{

    MTG_ASSERT( nType == xUnitValue ||

                nType == xTime ||

                nType == xDay ||

                nType == xFractionOfDay );

                

    cleanup();

    m_nType = nType;

}





//

//   s e t T y p e

//



void tNumericalExpr::setType( tType nType, char *sId )



{

    MTG_ASSERT( nType == xUnitValue );



    cleanup();



    m_nType = nType;

    m_sId = sId;

}





//

//   s e t T y p e

//



void tNumericalExpr::setType( tType nType, int nId )



{

    MTG_ASSERT( nType == xUnitValue );



    cleanup();



    m_nType = nType;

    m_nId = nId;

}





//

//   s e t T y p e

//



void tNumericalExpr::setType( tType nType, double gValue )



{

    MTG_ASSERT( nType == xNumber );



    cleanup();



    m_nType = nType;

    m_gValue = gValue;

}





//

//   s e t T y p e

//



void tNumericalExpr::setType( tType nType, tFactor* pFactor )



{

    MTG_ASSERT( nType == xFactor );

    MTG_ASSERT( pFactor != 0 );



    cleanup();



    m_nType = nType;

    m_pFactor = pFactor;

    m_pFactor->upRef();

}





//

//   s e t T y p e

//



void tNumericalExpr::setType( tType nType, tNumericalExpr *pChild )



{

    MTG_ASSERT( nType == xNeg ||

                nType == xFloor ||

                nType == xCeil ||

                nType == xNot );



    cleanup();



    m_nType = nType;

    m_Child[0] = pChild;

}





//

//   s e t T y p e

//



void tNumericalExpr::setType( tType nType, tNumericalExpr *pChild1,

    tNumericalExpr *pChild2 )



{

    MTG_ASSERT( nType == xPlus ||

                nType == xMinus ||

                nType == xTimes ||

                nType == xOver ||

                nType == xMin ||

                nType == xMax ||

                nType == xEq ||

                nType == xNe ||

                nType == xGt ||

                nType == xLt ||

                nType == xGe ||

                nType == xLe ||

                nType == xAnd ||

                nType == xOr ||

                nType == xIfThen );



    cleanup();



    m_nType = nType;

    m_Child[0] = pChild1;

    m_Child[1] = pChild2;

}





//

//   s e t T y p e

//



void tNumericalExpr::setType( tType nType, tNumericalExpr *pChild1,

    tNumericalExpr *pChild2, tNumericalExpr *pChild3 )



{

    MTG_ASSERT( nType == xIfThenElse ||

                nType == xMin ||

                nType == xMax );



    cleanup();



    m_nType = nType;

    m_Child[0] = pChild1;

    m_Child[1] = pChild2;

    m_Child[2] = pChild3;

}





//

//   s e t T y p e

//



void tNumericalExpr::setType( tType nType, tDateExpr *pChild1,

    tDateExpr *pChild2 )



{

    MTG_ASSERT( nType == xFutureValue ||

                nType == xPresentValue );



    cleanup();



    m_nType = nType;

    m_Child[0] = pChild1;

    m_Child[1] = pChild2;

}





//

//   s e t T y p e

//



void tNumericalExpr::setType( tType nType, tDateExpr *pChild1,

    tDateExpr *pChild2, tDateExpr *pChild3 )



{

    MTG_ASSERT( nType == xAccrualFactor ||

                nType == xDiscountFactor );



    cleanup();



    m_nType = nType;

    m_Child[0] = pChild1;

    m_Child[1] = pChild2;

    m_Child[2] = pChild3;

}





//

//   p a r s e E n v

//



tRetCode tNumericalExpr::parseEnv( tParser& Parser, int nFlags, bool& bGo )



{

    tRetCode nRet;

    tType nType;



    bGo = false;

    switch( Parser.curToken() ) {

        case xTokUnitValue :

            if( ! ( nFlags & ALLOW_UNITVALUE ) )

                return Parser.setError( INVALID_KEYWORD );

            if( ! ( nFlags & ALLOW_FD ) )

                return Parser.setError( INVALID_KEYWORD );

            nType = xUnitValue;

            break;



        case xTokTime : 

            if( ! ( nFlags & ALLOW_FD ) )

                return Parser.setError( INVALID_KEYWORD );

            nType = xTime;

            break;



        case xTokDay :

            if( ! ( nFlags & ALLOW_FD ) )

                return Parser.setError( INVALID_KEYWORD );

            nType = xDay;

            break;



        case xTokFractionOfDay :

            if( ! ( nFlags & ALLOW_FD ) ) {

                return Parser.setError( INVALID_KEYWORD );

            }

            nType = xFractionOfDay;

            break;



        default :

            bGo = true;

            break;

    }



    if( ! bGo ) {

        if( ( nRet = Parser.readToken() ) != OK )

            return nRet;

        setType( nType );

    }



    return OK;

}





//

//   p a r s e I d

//



tRetCode tNumericalExpr::parseId( tParser& Parser, int nFlags )



{

    MTG_ASSERT( Parser.curToken() == xTokId );



    tRetCode nRet;

    tType nType;

    char* sId;

    tObject* pObj;

    tFactor* pFactor;



        // Check first if it's a factor.



    if( ( pObj = Parser.findObject() ) != 0 &&

        ( pFactor = dynamic_cast<tFactor*>( pObj ) ) != 0 ) {

        if( ! ( nFlags & ALLOW_FACTOR ) )

            return Parser.setError( INVALID_KEYWORD );

        if( ! ( nFlags & ALLOW_FD ) )

            return Parser.setError( INVALID_KEYWORD );

        if( ( nRet = Parser.readToken() ) != OK )

            return nRet;

        setType( xFactor, pFactor );

        return OK;

    }  



    sId = StrCopy( Parser.curText() );



    if( ( nRet = Parser.readToken() ) != OK ||

        ( nRet = Parser.swallowToken( xTokReference, "." ) ) != OK ) {

        delete sId;

        return nRet;

    }



    switch( Parser.curToken() ) {

         case xTokUnitValue :

                // This is always allowed for references to other

                // instruments; only the current instrument may

                // be restricted.

            nType = xUnitValue;

            break;



         default :

            delete sId;

            return Parser.setError( "token 'unitvalue' or "

                                        "'multiplier' expected" );

    }



    if( ( nRet = Parser.readToken() ) != OK ) {

         delete sId;

         return nRet;

    }



    setType( nType, sId );

    return OK;

}





//

//   p a r s e M i n M a x

//



tRetCode tNumericalExpr::parseMinMax( tParser& Parser, int nFlags )



{

    MTG_ASSERT( Parser.curToken() == xTokMin ||

                Parser.curToken() == xTokMax );



    tRetCode nRet;

    tType nType;



    if( Parser.curToken() == xTokMin )

        nType = xMin;

    else

        nType = xMax;



    tNumericalExpr* pArg1 = new tNumericalExpr;



    if( ( nRet = Parser.readToken() ) != OK ||

        ( nRet = Parser.swallowToken( xTokLeftParen, "(" ) ) != OK ||

        ( nRet = pArg1->parseTerm( Parser, nFlags ) ) != OK ) {

        delete pArg1;

        return nRet;

    }



    tNumericalExpr* pArg2 = new tNumericalExpr;



    if( ( nRet = Parser.swallowToken( xTokComma, "," ) ) != OK ||                    

        ( nRet = pArg2->parseTerm( Parser, nFlags ) ) != OK ) {

        delete pArg1;

        delete pArg2;

        return nRet;

    }



    tNumericalExpr* pArg3 = 0;



    if( Parser.curToken() == xTokComma ) {

        pArg3 = new tNumericalExpr;



        if( ( nRet = Parser.readToken() ) != OK ||

            ( nRet = pArg3->parseTerm( Parser, nFlags ) ) != OK ) {

            delete pArg1; 

            delete pArg2;

            delete pArg3;

            return nRet;

        }

    }



    if( ( nRet = Parser.swallowToken( xTokRightParen, ")" ) ) != OK ) {

        delete pArg1;

        delete pArg2;

        if( pArg3 != 0 )

            delete pArg3;

        return nRet;

    }



    if( pArg3 != 0 )

        setType( nType, pArg1, pArg2, pArg3 );

    else

        setType( nType, pArg1, pArg2 );



    return OK;

}





//

//   p a r s e I f T h e n E l s e

//



tRetCode tNumericalExpr::parseIfThenElse( tParser& Parser, int nFlags )



{

    MTG_ASSERT( Parser.curToken() == xTokIf );



    tRetCode nRet;



    if( ( nRet = Parser.readToken() ) != OK )

        return OK;



    tNumericalExpr* pCond = new tNumericalExpr;

    tNumericalExpr* pBr1 = new tNumericalExpr;

    tNumericalExpr* pBr2 = 0;



    if( ( nRet = pCond->parseTerm( Parser, nFlags ) ) != OK ||

        ( nRet = Parser.swallowToken( xTokThen, "then" ) ) != OK ||

        ( nRet = pBr1->parseTerm( Parser, nFlags ) ) != OK ) {

        goto error;

    }



    switch( Parser.curToken() ) {

        case xTokElse :

            pBr2 = new tNumericalExpr;



            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = pBr2->parseTerm( Parser, nFlags ) ) != OK ||

                ( nRet = Parser.swallowToken( xTokEndif,

                            "endif" ) ) != OK ) {

                goto error;

            }

            setType( xIfThenElse, pCond, pBr1, pBr2 );

            break;



        case xTokEndif :

            if( ( nRet = Parser.readToken() ) != OK )

                goto error;

            setType( xIfThen, pCond, pBr1 );

            break;



        default :

            nRet = Parser.setError( "token 'else' or 'endif' expected" );

            goto error;

    }



    return OK;



error:



    delete pCond;

    delete pBr1;

    if( pBr2 != 0 )

        delete pBr2;



    return nRet;

}





//

//   p a r s e R a t e E x p r

//



tRetCode tNumericalExpr::parseRateExpr( tParser& Parser, int nFlags )



{

    MTG_ASSERT( Parser.curToken() == xTokAccrualFactor ||

                Parser.curToken() == xTokDiscountFactor ||

                Parser.curToken() == xTokFutureValue ||

                Parser.curToken() == xTokPresentValue );



    tRetCode nRet;

    tType nType;



    if( ! ( nFlags & ALLOW_MC ) )

        return Parser.setError( INVALID_KEYWORD );



    switch( Parser.curToken() ) {

        case xTokAccrualFactor :

            nType = xAccrualFactor;  break;

        case xTokDiscountFactor :

            nType = xDiscountFactor; break;

        case xTokFutureValue :

            nType = xFutureValue;    break;

        case xTokPresentValue :

            nType = xPresentValue;   break;

        default :

            throw tException( INTERNAL_ERROR );

    }



    tDateExpr* pArg1 = new tDateExpr;



    if( ( nRet = Parser.readToken() ) != OK ||

        ( nRet = Parser.swallowToken( xTokLeftParen, "(" ) ) != OK ||

        ( nRet = pArg1->parse1( Parser, nFlags ) ) != OK ) {

        delete pArg1;

        return nRet;

    }



    tDateExpr* pArg2 = new tDateExpr;



    if( ( nRet = Parser.swallowToken( xTokComma, "," ) ) != OK ||                    

        ( nRet = pArg2->parse1( Parser, nFlags ) ) != OK ) {

        delete pArg1;

        delete pArg2;

        return nRet;

    }



    tDateExpr* pArg3 = 0;



    if( Parser.curToken() == xTokComma ) {

        if( nType == xFutureValue || nType == xPresentValue ) {

            delete pArg1;

            delete pArg2;

            return Parser.setError( WRONG_ARG_COUNT );

        }



        pArg3 = new tDateExpr;



        if( ( nRet = Parser.readToken() ) != OK ||

            ( nRet = pArg3->parse1( Parser, nFlags ) ) != OK ) {

            delete pArg1; 

            delete pArg2;

            delete pArg3;

            return nRet;

        }

    }

    else {

        if( nType == xAccrualFactor || nType == xDiscountFactor ) {

            delete pArg1;

            delete pArg2;

            return Parser.setError( WRONG_ARG_COUNT );

        }

    }



    if( ( nRet = Parser.swallowToken( xTokRightParen, ")" ) ) != OK ) {

        delete pArg1;

        delete pArg2;

        if( pArg3 != 0 )

            delete pArg3;

        return nRet;

    }



    if( pArg3 != 0 )

        setType( nType, pArg1, pArg2, pArg3 );

    else

        setType( nType, pArg1, pArg2 );



    return OK;

}





//

//   p a r s e F a c t o r

//



tRetCode tNumericalExpr::parseFactor( tParser& Parser, int nFlags )



{

    bool bGo;

    tRetCode nRet;

    tNumericalExpr *pArg1;

    tType nType;

    double gValue;

    tDate Date;



        // Eat plusses.



    while( Parser.curToken() == xTokPlus ) {

        if( ( nRet = Parser.readToken() ) != OK )

            return nRet;

    }

        

    if( ( nRet = parseEnv( Parser, nFlags, bGo ) ) != OK || ! bGo )

        return nRet;



    switch( Parser.curToken() ) {

        case xTokNumber :

            if( ( nRet = Parser.scanDouble( gValue ) ) != OK )

                return nRet;

            setType( xNumber, gValue );

            break;



        case xTokString :

        case xTokDate :

            if( ( nRet = Parser.scanDate( Date ) ) != OK )

                return nRet;

            return Parser.setError( NOT_AVAILABLE );

            //setType( xNumber, System.dateIndex( Date ) );

            break;



        case xTokId :

            if( ( nRet = parseId( Parser, nFlags ) ) != OK )

                return nRet;

            break;



        case xTokLeftParen :

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = parseTerm( Parser, nFlags ) ) != OK ||

                ( nRet = Parser.swallowToken( xTokRightParen,

                            ")" ) ) != OK ) {                    

                return nRet;

            }

            break;



        case xTokMinus :

            pArg1 = new tNumericalExpr;

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = pArg1->parseFactor( Parser,

                            nFlags ) ) != OK ) {

                delete pArg1;

                return nRet;

            }

            if( pArg1->m_nType == xNumber ) {

                setType( xNumber, -pArg1->m_gValue );

                delete pArg1;

            }

            else {

                setType( xNeg, pArg1 );

            }

            break;



        case xTokNot :

            pArg1 = new tNumericalExpr;

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = pArg1->parseFactor( Parser,

                            nFlags ) ) != OK ) {

                delete pArg1;

                return nRet;

            }

            setType( xNot, pArg1 );

            break;



        case xTokFloor :

        case xTokCeil :

            if( Parser.curToken() == xTokFloor )

                nType = tNumericalExpr::xFloor;

            else

                nType = tNumericalExpr::xCeil;



            pArg1 = new tNumericalExpr;

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.swallowToken( xTokLeftParen,

                            "(" ) ) != OK ||

                ( nRet = pArg1->parseTerm( Parser,

                            nFlags ) ) != OK ||

                ( nRet = Parser.swallowToken( xTokRightParen,

                            ")" ) ) != OK ) {                 

                delete pArg1;

                return nRet;

            }



            setType( nType, pArg1 );

            break;



        case xTokMin :

        case xTokMax :

            if( ( nRet = parseMinMax( Parser, nFlags ) ) != OK )

                return nRet;

            break;



        case xTokIf :

            if( ( nRet = parseIfThenElse( Parser, nFlags ) ) != OK )

                return nRet;

            break;



        case xTokAccrualFactor :

        case xTokDiscountFactor :

        case xTokFutureValue :

        case xTokPresentValue :

            if( ( nRet = parseRateExpr( Parser, nFlags ) ) != OK )

                return nRet;

            break;



        default :

            return Parser.setError( MALFORMED_EXPRESSION );

    }



    return OK;

}





//

//   p a r s e T e r m

//



tRetCode tNumericalExpr::parseTerm( tParser& Parser, int nFlags, int nLevel )



{

    tRetCode nRet;

    tType nType;



    static struct tTermLevel {

        int m_nNumOfAssoc;

        struct {

            tToken m_nToken;

            tType m_nExprType;

        } m_aAssoc[6];

    } aLevel[] = {

        { 2, { { xTokTimes, xTimes },

               { xTokOver, xOver } } },

        { 2, { { xTokPlus, xPlus },

               { xTokMinus, xMinus } } },

        { 6, { { xTokEq, xEq },

               { xTokNe, xNe },

               { xTokGt, xGt },

               { xTokLt, xLt },

               { xTokGe, xGe },

               { xTokLe, xLe } } },

        { 1, { { xTokAnd, xAnd } } },

        { 1, { { xTokOr, xOr } } },

        { 1, { { xTokQuestion, xIfThenElse } } },

    };



    if( nLevel == 0 ) {

        nRet = parseFactor( Parser, nFlags );

    }

    else {

        if( nLevel >= (int) ( sizeof(aLevel) / sizeof(tTermLevel) ) )

            nLevel = (int) ( sizeof(aLevel) / sizeof(tTermLevel) ) - 1;

        nRet = parseTerm( Parser, nFlags, nLevel - 1 );

    }

  

    if( nRet != OK )

        return nRet;



    while( true ) {

        int k;



        for( k = 0; k < aLevel[nLevel].m_nNumOfAssoc; ++k ) {

            if( aLevel[nLevel].m_aAssoc[k].m_nToken == Parser.curToken() ) {

                nType = aLevel[nLevel].m_aAssoc[k].m_nExprType;

                break;

            }

        }



        if( k >= aLevel[nLevel].m_nNumOfAssoc )

            break;



        if( ( nRet = Parser.readToken() ) != OK )

            return nRet;



        tNumericalExpr* pArg1 = new tNumericalExpr;

        tNumericalExpr* pArg2 = new tNumericalExpr;

        tNumericalExpr* pArg3 = 0;



        if( nLevel == 0 )

            nRet = pArg2->parseFactor( Parser, nFlags );

        else

            nRet = pArg2->parseTerm( Parser, nFlags, nLevel - 1 );



        if( nType == xIfThenElse && nRet == OK ) {

            if( Parser.curToken() != xTokColon )

                nRet = Parser.setError( "':' in ternary operator expected" );

            else

                nRet = Parser.readToken();

            if( nRet == OK ) {

                pArg3 = new tNumericalExpr;

                nRet = pArg3->parseTerm( Parser, nFlags, nLevel - 1 );

            }

        }

        

        if( nRet != OK ) {

            delete pArg1;

            delete pArg2;

            if( pArg3 != 0 )

                delete pArg3;

            return nRet;

        }



            // Need to rotate (left-associate).



        memcpy( pArg1, this, sizeof(*this) );

        init();



        if( nType == xIfThenElse )

            setType( nType, pArg1, pArg2, pArg3 );

        else

            setType( nType, pArg1, pArg2 );

    }



    return OK;

}





//

//   p a r s e 1

//



tRetCode tNumericalExpr::parse1( tParser& Parser, int nFlags )



{

    return parseTerm( Parser, nFlags );

}





//

//   p a r s e 1

//



tRetCode tNumericalExpr::parse1( tParser& Parser )



{

    return parse1( Parser, ALLOW_ALL );

}





//

//   t N u m e r i c a l E x p r

//



tNumericalExpr::tNumericalExpr()



{

    init();

}





//

//   t N u m e r i c a l E x p r

//



tNumericalExpr::tNumericalExpr( tNumericalExpr& Expr )



{

    init();

    copyFrom( Expr );

}





//

//   ~ t N u m e r i c a l E x p r

//



tNumericalExpr::~tNumericalExpr()



{

    cleanup();

}





//

//   c l o n e

//



tExpression* tNumericalExpr::clone()



{

    return new tNumericalExpr( *this );

}





//

//   o p e r a t o r =

//



tNumericalExpr& tNumericalExpr::operator=( tNumericalExpr& Expr )



{

    if( &Expr != this )

        copyFrom( Expr );

    return *this;

}





//

//   r e s o l v e

//



tRetCode tNumericalExpr::resolve( int nThisId, tLookup<int>& Claim,

    tLookup<int>& Factor, tHeap<tFactor*>& FactorPtr, tPortfolio& Pf,

    int nFlags )



{

    int k, nId;

    tRetCode nRet;



    if( m_nType == xUnitValue ) {

        if( m_sId != 0 ) {

            if( ( nRet = Claim.retrieve( m_sId, nId ) ) != OK )

                return nRet;

            if( nId != nThisId ) {

                Pf.regDependency( nThisId, nId );

                m_nId = nId;

            }

            else {

                    // If the claim doesn't have access to its

                    // own unit value, report an error, because

                    // that's what it's trying to do.

                if( ! ( nFlags & ALLOW_UNITVALUE ) )

                    return ILLEGAL_DEPENDENCY;

                if( ! ( nFlags & ALLOW_FD ) )

                    return ILLEGAL_DEPENDENCY;

            }

        }

    }

    else

    if( m_nType == xFactor ) {

        if( ( nRet = Factor.retrieve( m_pFactor->name(), nId ) ) != OK ) {

            if( nRet != NOT_FOUND )

                return nRet;



            MTG_ASSERT( FactorPtr.numOfElems() == Factor.numOfEntries() );



                // Don't increase reference count at this point!



            nId = FactorPtr.numOfElems();

            if( ( nRet = Factor.insert( m_pFactor->name(), nId ) ) != OK )

                return nRet;



            ++FactorPtr;

            FactorPtr.last() = m_pFactor;

        }

        else {

                // Check if there is an ambiguity (can happen if a

                // factor is redefined with the same name).

            if( FactorPtr[nId] != m_pFactor )

                return AMBIGUITY;

        }

        m_nId = nId;

    }



    for( k = 0; k < 3 && m_Child[k] != 0; ++k ) {

        if( ( nRet = m_Child[k]->resolve( nThisId, Claim,

                        Factor, FactorPtr, Pf, nFlags ) ) != OK ) {

            return nRet;

        }

    }  



    return OK;

}





//

//   r e s o l v e

//



tRetCode tNumericalExpr::resolve( int nThisId, tLookup<int>& Claim,

    tLookup<int>& Factor, tHeap<tFactor*>& FactorPtr, tPortfolio& Pf )



{

    return resolve( nThisId, Claim, Factor, FactorPtr, Pf, 

        ALLOW_UNITVALUE | ALLOW_FACTOR | ALLOW_FD | ALLOW_MC );

}





//

//   g e t E v e n t s

//



void tNumericalExpr::getEvents( tSeqEvGenerator& EvGen ) const



{

    if( m_nType == xEq || m_nType == xNe || m_nType == xGt ||

        m_nType == xLt || m_nType == xGe || m_nType == xLe ) {



        for( int k = 0; k <= 1; ++k ) {

            tNumericalExpr* p1 = static_cast<tNumericalExpr*>( m_Child[k] );



            if( p1->m_nType == xDay || p1->m_nType == xTime ) {

                tNumericalExpr* p2 =

                    static_cast<tNumericalExpr*>( m_Child[1 - k] );



                if( p2->m_nType == xNumber ) {

                    int nDay = (int) p2->m_gValue;



                    if( nDay > 0 && nDay == p2->m_gValue ) {

                        EvGen += nDay;

                        return;

                    }

                }

            }

        }

    }



    for( int k = 0; k < 3 && m_Child[k] != 0; ++k )

        m_Child[k]->getEvents( EvGen );

}





//

//   g e t B a r r i e r s

//



void tNumericalExpr::getBarriers( const tFactor* pFactor,

    tHeap<double>& Barrier ) const



{

    if( m_nType == xEq || m_nType == xNe || m_nType == xGt ||

        m_nType == xLt || m_nType == xGe || m_nType == xLe ) {



        for( int k = 0; k <= 1; ++k ) {

            tNumericalExpr* p1 = static_cast<tNumericalExpr*>( m_Child[k] );



            if( p1->m_nType == xFactor && p1->m_pFactor == pFactor ) {

                tNumericalExpr* p2 =

                    static_cast<tNumericalExpr*>( m_Child[1 - k] );



                if( p2->m_nType == xNumber && p2->m_gValue > 0 ) {

                    Barrier.append( p2->m_gValue );

                    return;

                }

            }

        }

    }



    for( int k = 0; k < 3 && m_Child[k] != 0; ++k )

        m_Child[k]->getBarriers( pFactor, Barrier );

}





//

//   g e t C o n s t a n t s

//



void tNumericalExpr::getConstants( tHeap<double>& Constant ) const



{

        // Tries to find the most likely barriers.



    switch( m_nType ) {

        case xNumber :

            Constant.append( m_gValue );

            break;



        case xIfThen :

        case xIfThenElse :

            { for( int k = 1; k < 3 && m_Child[k] != 0; ++k ) {

                tNumericalExpr* p =

                    static_cast<tNumericalExpr*>( m_Child[k] );



                if( p->m_nType == xNumber ||

                    p->m_nType == xIfThen ||

                    p->m_nType == xIfThenElse ) {

                    p->getConstants( Constant );

                }

            } }

            break;



        default :

            break;

    }

}





//

//   i s N u m b e r

//



bool tNumericalExpr::isNumber( double& gValue ) const



{

    if( m_nType != xNumber )

        return false;

    gValue = m_gValue;

    return true;

}





//

//   a p p l y

//



double tNumericalExpr::apply( tEngine& Engine, double gUnitValue,

    bool& bValid )



{

    double gResult, gArg1, gArg2;

    tFDEngine* pFD;

    bool bValid1;



        // bValid is set to false only if the missing branch of an

        // if..then..endif expression is triggered.



    bValid = true;



    switch( m_nType ) {

        case xNumber :

            return m_gValue;



        case xTime :          

            if( ( pFD = dynamic_cast<tFDEngine*>( &Engine ) ) == 0 )

                throw tException( NOT_AVAILABLE );

            return pFD->day() + pFD->fractionOfDay();



        case xDay :           

            if( ( pFD = dynamic_cast<tFDEngine*>( &Engine ) ) == 0 )

                throw tException( NOT_AVAILABLE );

            return pFD->day();



        case xFractionOfDay : 

            if( ( pFD = dynamic_cast<tFDEngine*>( &Engine ) ) == 0 )

                throw tException( NOT_AVAILABLE );

            return pFD->fractionOfDay();



        case xFactor :        

            if( ( pFD = dynamic_cast<tFDEngine*>( &Engine ) ) == 0 )

                throw tException( NOT_AVAILABLE );

            return pFD->factor( m_nId );



        case xUnitValue :

            if( m_nId < 0 )

                return gUnitValue;

            if( ( pFD = dynamic_cast<tFDEngine*>( &Engine ) ) == 0 )

                throw tException( NOT_AVAILABLE );

            pFD->getAuxClaim( m_nId, gResult );

            return gResult;



        default :

            break;

    }



        // now check for interest rate expressions



    if( m_nType == xAccrualFactor || m_nType == xDiscountFactor ||

        m_nType == xFutureValue || m_nType == xPresentValue ) {



        tDate Arg1, Arg2, Arg3;



        MTG_ASSERT( m_Child[0] != 0 );

        Arg1 = static_cast<tDateExpr*>( m_Child[0] )->

                apply( Engine, gUnitValue, bValid1 );



        MTG_ASSERT( m_Child[1] != 0 );

        Arg2 = static_cast<tDateExpr*>( m_Child[1] )->

                apply( Engine, gUnitValue, bValid1 );



        if( m_Child[2] != 0 ) {

            MTG_ASSERT( m_nType == xAccrualFactor ||

                        m_nType == xDiscountFactor );



            Arg3 = static_cast<tDateExpr*>( m_Child[2] )->

                    apply( Engine, gUnitValue, bValid1 );

        }



        switch( m_nType ) {

            case xAccrualFactor :

                return Engine.accrualFactor( Arg1, Arg2, Arg3 );



            case xDiscountFactor :

                return Engine.discountFactor( Arg1, Arg2, Arg3 );



            case xFutureValue :

                return Engine.futureValue( Arg1, Arg2 );



            case xPresentValue :

                return Engine.presentValue( Arg1, Arg2 );

   

            default :

                throw tException( INTERNAL_ERROR );

        }

            // we'll never get here

    }



        // so from now on, only numerical expressions have

        // to be considered



    MTG_ASSERT( m_Child[0] != 0 );

    gArg1 = static_cast<tNumericalExpr*>( m_Child[0] )->

                apply( Engine, gUnitValue, bValid1 );



    switch( m_nType ) {

        case xNeg :   return -gArg1; 

        case xFloor : return floor( gArg1 );

        case xCeil :  return ceil( gArg1 );

        case xNot :   return ( gArg1 == 0 ) ? 1 : 0;

    

        case xAnd :

            if( gArg1 == 0 )

                return 0;

            break;



        case xOr :

            if( gArg1 != 0 )

                return 1;

            break;



        case xIfThen :

            if( gArg1 != 0 ) {

                return

                    static_cast<tNumericalExpr*>( m_Child[1] )->

                        apply( Engine, gUnitValue, bValid );

                                // inherit bValid

            }

            bValid = false;     // forced to use default value of 0

            return 0;



        case xIfThenElse :

            if( gArg1 != 0 ) {

                return

                    static_cast<tNumericalExpr*>( m_Child[1] )->

                        apply( Engine, gUnitValue, bValid );                            

            }

            return

                static_cast<tNumericalExpr*>( m_Child[2] )->

                    apply( Engine, gUnitValue, bValid );                        



        default :

            break;

    }



    MTG_ASSERT( m_Child[1] != 0 );

    gArg2 = static_cast<tNumericalExpr*>( m_Child[1] )->

                apply( Engine, gUnitValue, bValid1 );



    switch( m_nType ) {

        case xPlus :  return gArg1 + gArg2;

        case xMinus : return gArg1 - gArg2;

        case xTimes : return gArg1 * gArg2;



        case xOver :

            if( gArg2 == 0 )

                throw tException( DIVISION_BY_ZERO );

            return gArg1 / gArg2;



        case xMin :

            gResult = ( gArg1 < gArg2 ) ? gArg1 : gArg2;

            if( m_Child[2] != 0 ) {

                gArg2 =

                    static_cast<tNumericalExpr*>( m_Child[2] )->

                        apply( Engine, gUnitValue, bValid1 );                            

                if( gArg2 < gResult )

                    gResult = gArg2;

            }

            return gResult;



        case xMax :

            gResult = ( gArg1 > gArg2 ) ? gArg1 : gArg2;

            if( m_Child[2] != 0 ) {

                gArg2 =

                    static_cast<tNumericalExpr*>( m_Child[2] )->

                        apply( Engine, gUnitValue, bValid1 );                            

                if( gArg2 > gResult )

                    gResult = gArg2;

            }

            return gResult;



        case xEq :  return ( gArg1 == gArg2 ) ? 1 : 0;  

        case xNe :  return ( gArg1 != gArg2 ) ? 1 : 0;

        case xGt :  return ( gArg1 > gArg2 ) ? 1 : 0;

        case xLt :  return ( gArg1 < gArg2 ) ? 1 : 0;

        case xGe :  return ( gArg1 >= gArg2 ) ? 1 : 0;

        case xLe :  return ( gArg1 <= gArg2 ) ? 1 : 0;

        case xAnd :     // fall through

        case xOr :  return ( gArg2 != 0 ) ? 1 : 0;



        default :

            throw tException( NOT_IMPLEMENTED );

    }



    return 0;

}





//

//   p a r s e

//



tRetCode tNumericalExpr::parse( tParser& Parser, int nFlags )



{

    tRetCode nRet;



        // First try the most basic expression: a positive number.

        // Note that in this case, subsequent arithmetic operators

        // are ignored.



    if( Parser.curToken() == xTokNumber ) {

        double gValue;



        if( ( nRet = Parser.scanDouble( gValue ) ) != OK )

            return nRet;

        setType( xNumber, gValue );

        return OK;

    }



        // Now try obligatory '{', '}'.



    if( ( nRet = Parser.scanBeginOfObj() ) != OK ||

        ( nRet = parse1( Parser, nFlags ) ) != OK ||

        ( nRet = Parser.scanEndOfObj() ) != OK ) {

        return nRet;

    }



    return OK;

}





//

//   p a r s e

//



tRetCode tNumericalExpr::parse( tParser& Parser )



{

    return parse( Parser, ALLOW_ALL );

}



MTG_END_NAMESPACE

