// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#if ! defined(_MTG_OFSOLVER_)

#define _MTG_OFSOLVER_



#include "MtgHeap.h"

#include "MtgSlotLayer.h"



MTG_BEGIN_NAMESPACE



class tOFEngine;





//

//   t O F S o l v e r

//



class tOFSolver {



public:



        // Note that for explicit schemes, "xGivenValue" is used

        // if the lattice is "widening."

        

    enum tBoundary {

        xGivenValue,

        xZeroGamma

    };



    struct tProcessParamsStub {

        // Process parameters like vol, drift and discount.

        // To be extended.

    };



        // Rollback weights. Not all parameters need to be

        // used in all schemes.



    struct tRBWeights {     

        double m_gCurU;

        double m_gCurM;

        double m_gCurD;

        double m_gLastU;

        double m_gLastM;

        double m_gLastD;

    };



        // Extrapolation weights for the

        // zero gamma condition:



    struct tZGWeights {

        double m_gBoundary;         // weight of boundary level

        double m_gInner;            // weight of level inside boundary

    };



        // Description for each sub-problem (one sub-problem per

        // instrument):



    struct tSlot {

        int m_nIndex;               // index of instrument

        double m_gMultiplier;

        int m_nNumOfUpLevels;

        int m_nNumOfDownLevels;

        tBoundary m_nUpBoundary;

        tBoundary m_nDownBoundary;  

        tRBWeights m_UpRBWeights;   // if boundary is zero gamma,

        tRBWeights m_DownRBWeights; // we need all this

        tZGWeights m_UpZGWeights;   

        tZGWeights m_DownZGWeights;

    };



        // Interface for implicit methods and American options:



    class tIncrement {            

        public:

            tIncrement() {}

            virtual void beginIncrement( int nAdjDown, int nAdjUp ) = 0;

            virtual void doIncrement( const tHeap<int>& Pos ) = 0;

            virtual void endIncrement( int nAdjDown, int nAdjUp ) = 0;

    };



private:



    int m_nRootLevel;



    tSlotLayer* m_pLayer;           // referenced, not owned

    tOFEngine* m_pEngine;           // referenced, not owned



    double m_gAbsDuration;

    double m_gSqrtAbsDuration;

    double m_gRelDuration;



    tHeap<tRBWeights> m_RBWeights;

    tHeap<tSlot> m_Slot;



    void init();



protected:



    tOFSolver();



    void copyFrom( const tOFSolver& Solver );



        // Data access:



    tSlotLayer& layer()            { return *m_pLayer; }

    tOFEngine& engine()            { return *m_pEngine; }



    int rootLevel() const          { return m_nRootLevel; }

    double absDuration() const     { return m_gAbsDuration; }

    double sqrtAbsDuration() const { return m_gSqrtAbsDuration; }

    double relDuration() const     { return m_gRelDuration; }



    tRBWeights& RBWeights( int nLevel ) {

        return m_RBWeights[m_nRootLevel + nLevel];

    }



    int numOfSlots() const {

        return m_Slot.numOfElems();

    }



    tSlot& slot( int nPos ) {

        return m_Slot[nPos];

    }



        // Required services:



    virtual void calcWeights( int nFromLevel, int nToLevel,

        const tProcessParamsStub& Params ) = 0;



    virtual void adjustZeroGammaUpBoundary( int nLevel,

        tRBWeights& RBWeights, tZGWeights& ZGWeights ) = 0;



    virtual void adjustZeroGammaDownBoundary( int nLevel,

        tRBWeights& RBWeights, tZGWeights& ZGWeights ) = 0;



        // Direct solver per instrument:



    virtual void solve( int nSlot ) = 0;



        // Services offered to subclasses:



    void copyCurToPrep( int nSlot, bool bInitial );

    const tRBWeights* getRBWeights( int nLevel, int nSlot );



public:



    tOFSolver( int nRootLevel, int nNumOfUpLevels,

        int nNumOfDownLevels );



    tOFSolver( const tOFSolver& Solver );



    virtual ~tOFSolver();



        // First prepare solver:



    virtual void prepare( tOFEngine& Engine, tSlotLayer& Layer,

        int nNumOfUpLevels, int nNumOfDownLevels,

        int nLastNumOfUpLevels, int nLastNumOfDownLevels );



        // Then prepare for each instrument individually:



    virtual void prepare( int nIndex, int nNumOfUpLevels,

        tBoundary nUpBoundary, int nNumOfDownLevels,

        tBoundary nDownBoundary );



        // There are two solvers, direct and iterative.

        // The iterative solver is used to refine the

        // result incrementally.



    virtual void solve();

    virtual tRetCode refine( tIncrement& Incr );

};



MTG_END_NAMESPACE



#endif

