// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#if ! defined(_MTG_OBJECT_)

#define _MTG_OBJECT_



#include "MtgSource.h"

#include "MtgWithSystem.h"



// Notice that MtgSystem.h is included at the end of the file.



MTG_BEGIN_NAMESPACE



class tParser;





//

//   t O b j e c t

//

    

class tObject : public tWithSystem {



    typedef tWithSystem super;



public:



    struct tParseInfoStub {

    };



private:



        // An object keeps a reference counter (some references

        // might not register here, if they are temporary).



    int m_nRefCnt;



        // An object also may have a name.



    char m_sName[nSourceMaxTokenLength];



        // The verbose flag is parsed in parseParam().



    bool m_bIsVerbose;



        // Finalization is voluntary.



    bool m_bIsFinalized;



protected:



    void touch() {

        m_bIsFinalized = false;

    }



    void copyFrom( const tObject& Obj );



        // Helper functions for parsing:



    static tRetCode parseType( tParser& Parser,

        tToken& nToken, tToken nDefault );



        // parse() implements the parsing control loop; parseParam()

        // needs to be overridden and is called for each parameter.

        // parsePostfix() is called immediately before finalize()

        // is called; parsePrefix() is called before parseParam()

        // is called the first time.



    virtual tRetCode parsePrefix( tParser& Parser, tParseInfoStub& Info );

    virtual tRetCode parseParam( tParser& Parser, tParseInfoStub& Info );

    virtual tRetCode parsePostfix( tParser& Parser, tParseInfoStub& Info );



    virtual tRetCode parse( tParser& Parser, tParseInfoStub* pInfo = 0 );



        // finalize() is called after parsing:



    virtual tRetCode finalize();



#if defined(_WIN32)



    template <class T>

    static void setObjRefToZeroOrSo( T*& pRef, T* pNew = 0 ) {

        if( pRef == pNew )

            return;

        if( pRef != 0 ) {

            pRef->downRef();

            if( pRef->canDelete() )

                delete pRef;

        }

        pRef = pNew;

        if( pRef != 0 )

            pRef->upRef();

    }



#endif



public:



    tObject();

    tObject( const tObject& Obj );



    virtual ~tObject();



    virtual tObject* clone() const = 0;



        // Read attributes:



    bool isVerbose() const {

        return m_bIsVerbose;

    }



    bool isFinalized() const {

        return m_bIsFinalized;

    }



        // Changing the name does not affect m_bIsFinalized.



    void setName( const char* sName );



    bool isAnonymous() const {

        return m_sName[0] ? false : true;

    }



    const char* name() const {

        return m_sName;

    }



        // Handle memory:



    void upRef();

    void downRef();



    bool canDelete() const;



        // Access to verbose reporting through this object:

    

    ostream& verbose() const;

    void verbose( const char* sFmt, ... ) const;



        // Some objects may do something on their own, like

        // triggerung saves to files etc.



    virtual tRetCode autoActivate() {

        return OK;

    }



    friend class tObjectStub;

};



#if ! defined(_WIN32)



    // If static template member functions are not supported,

    // we make setObjRefZeroOrSo a regular template function. This

    // is only the second best solution, of course, since it

    // puts setObjRefZeroOrSo into global scope. But I don't want to

    // change the signature of this function. (I made the name

    // look ridiculous in order to avoid conflicts.)



template <class T>

void setObjRefToZeroOrSo( T*& pRef, T* pNew = 0 )



{

    if( pRef == pNew )

        return;

    if( pRef != 0 ) {

        pRef->downRef();

        if( pRef->canDelete() )

            delete pRef;

    }

    pRef = pNew;

    if( pRef != 0 )

        pRef->upRef();

}



#endif



MTG_END_NAMESPACE



#include "MtgSystem.h"



#endif

