// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgParser.h"

#include "MtgBootstrap.h"

#include "MtgCalendar.h"

#include "MtgClaim.h"

#include "MtgCurve.h"

#include "MtgDrift.h"

#include "MtgEvaluate.h"

#include "MtgFactor.h"

#include "MtgImage.h"

#include "MtgInterestConvention.h"

#include "MtgLattice.h"

#include "MtgModel.h"

#include "MtgOptimizer.h"

#include "MtgPathSpace.h"

#include "MtgPortfolio.h"

#include "MtgRepository.h"

#include "MtgScenario.h"

#include "MtgShell.h"

#include "MtgSystem.h"

#include "MtgVol.h"



MTG_BEGIN_NAMESPACE





//

//   p a r s e P r e f i x

//



tRetCode tParser::parsePrefix( tObject& Obj )



{

    tRetCode nRet;



    if( ( nRet = readToken() ) != OK )

        return nRet;



    if( curToken() == xTokId ) {

        Obj.setName( curText() );

        if( ( nRet = readToken() ) != OK )

             return nRet;

    }

    else {

        Obj.setName( "" );

    }



    return OK;

}





//

//   m o v e P r e f i x

//



void tParser::movePrefix( const tObject& From, tObject* pTo )



{

    pTo->setName( From.name() );

}





//

//   s c a n T a b l e

//



tRetCode tParser::scanTable( tHeap<double>& Value,

    tRetCode (super::*scanDouble)( double& ) )



{

    double gValue;

    tRetCode nRet;

    bool bGo;



    Value.reset();



    if( beginOfObj() ) {

        if( ( nRet = scanBeginOfObj() ) != OK )

            return nRet;



        while( ! endOfObj() ) {

            if( ( nRet = (this->*scanDouble)( gValue ) ) != OK ||

                ( nRet = swallowComma( bGo ) ) != OK ) {

                return nRet;

            }

                // ignore whether comma is there or not

            Value.append( gValue );

        }



        if( ( nRet = scanEndOfObj() ) != OK )

            return nRet;

    }

    else {

        if( ( nRet = (this->*scanDouble)( gValue ) ) != OK )

            return nRet;

        Value.append( gValue );

    }



    return OK;

}





//

//   s c a n R a n g e

//



tRetCode tParser::scanRange( tHeap<double>& Value,

    tRetCode (tParser::*scanTable)( tHeap<double>& ), 

    tRetCode (super::*scanRange)( double&, double& ) )



{

    tRetCode nRet;



    Value.reset();



    if( beginOfObj() ) {

        tHeap<double> V;



        if( ( nRet = (this->*scanTable)( V ) ) != OK )

            return nRet;

        for( int i = 0; i < V.numOfElems(); ++i )

            Value.sortInAsc( V[i], true );

        return OK;

    }



    double gMin, gMax;

    int nGranularity = 1;



    if( ( nRet = (this->*scanRange)( gMin, gMax ) ) != OK )

        return nRet;



    if( curToken() == xTokColon ) {

        if( ( nRet = readToken() ) != OK ||

            ( nRet = scanInteger( nGranularity, 1 ) ) != OK ) {

            return nRet;

        }

    }



    if( gMin != gMax ) {

        double d = ( gMax - gMin ) / nGranularity;



        for( int k = 0; k < nGranularity; ++k, gMin += d )

            Value.append( gMin );

    }



    Value.append( gMax );

    return OK;

}





//

//   t P a r s e r

//



tParser::tParser()



{

    m_pRep = 0;

    m_pShell = 0;

}





//

//   ~ t P a r s e r

//



tParser::~tParser()



{

    if( m_pShell != 0 ) {

        m_pShell->downRef();

        if( m_pShell->canDelete() )

            delete m_pShell;

    }

}





//

//   s e t R e p o s i t o r y

//



void tParser::setRepository( tRepository* pRep )



{

    m_pRep = pRep;

}





//

//   s e t S h e l l

//



void tParser::setShell( tShell* pShell )



{

    if( pShell == m_pShell )

        return;



    if( m_pShell != 0 ) {

        m_pShell->downRef();

        if( m_pShell->canDelete() )

            delete m_pShell;

    }

    m_pShell = pShell;

    if( m_pShell != 0 )

        m_pShell->upRef();

}





//

//   b e g i n O f O b j

//



bool tParser::beginOfObj()



{

    if( curToken() == xTokLeftBrace )

        return true;

    return false;

}





//

//   e n d O f O b j

//



bool tParser::endOfObj()



{

    if( curToken() == xTokRightBrace ||

        curToken() == xTokEOF ) {

        return true;

    }

    return false;

}





//

//   s c a n B e g i n O f O b j

//



tRetCode tParser::scanBeginOfObj()



{

    return swallowToken( xTokLeftBrace, "{" );

}





//

//   s c a n E n d O f O b j

//



tRetCode tParser::scanEndOfObj()



{

    return swallowToken( xTokRightBrace, "}" );

}





//

//   s c a n T a b l e

//



tRetCode tParser::scanTable( tHeap<double>& Value )



{

    return scanTable( Value, super::scanDouble );

}





//

//   s c a n P o s T a b l e

//



tRetCode tParser::scanPosTable( tHeap<double>& Value )



{

    return scanTable( Value, super::scanPosDouble );

}





//

//   s c a n N o n N e g T a b l e

//



tRetCode tParser::scanNonNegTable( tHeap<double>& Value )



{

    return scanTable( Value, super::scanNonNegDouble );

}





//

//   s c a n P e r c e n t a g e T a b l e

//



tRetCode tParser::scanPercentageTable( tHeap<double>& Value )



{

    return scanTable( Value, super::scanPercentage );

}





//

//   s c a n P o s P e r c e n t a g e T a b l e

//



tRetCode tParser::scanPosPercentageTable( tHeap<double>& Value )



{

    return scanTable( Value, super::scanPosPercentage );

}





//

//   s c a n N o n N e g P e r c e n t a g e T a b l e

//



tRetCode tParser::scanNonNegPercentageTable( tHeap<double>& Value )



{

    return scanTable( Value, super::scanNonNegPercentage );

}





//

//   s c a n R a n g e

//



tRetCode tParser::scanRange( tHeap<double>& Value )



{

    return scanRange( Value, scanTable, super::scanRange );

}





//

//   s c a n P o s R a n g e

//



tRetCode tParser::scanPosRange( tHeap<double>& Value )



{

    return scanRange( Value, scanPosTable, super::scanPosRange );

}





//

//   s c a n N o n N e g R a n g e

//



tRetCode tParser::scanNonNegRange( tHeap<double>& Value )



{

    return scanRange( Value, scanNonNegTable, super::scanNonNegRange );

}





//

//   s c a n P e r c e n t a g e R a n g e

//



tRetCode tParser::scanPercentageRange( tHeap<double>& Value )



{

    return scanRange( Value, scanPercentageTable,

        super::scanPercentageRange );

}





//

//   s c a n P o s P e r c e n t a g e R a n g e

//



tRetCode tParser::scanPosPercentageRange( tHeap<double>& Value )



{

    return scanRange( Value, scanPosPercentageTable,

        super::scanPosPercentageRange );

}





//

//   s c a n N o n N e g P e r c e n t a g e R a n g e

//



tRetCode tParser::scanNonNegPercentageRange( tHeap<double>& Value )



{

    return scanRange( Value, scanNonNegPercentageTable,

        super::scanNonNegPercentageRange );

}





//

//   f i n d O b j e c t

//



tObject* tParser::findObject( const char* sName ) const



{

    MTG_ASSERT( m_pRep != 0 );



    if( sName == 0 ) {

        MTG_ASSERT( curToken() == xTokId );

        sName = curText();

    }



    return m_pRep->find( sName );

}







//

//   p a r s e O b j e c t

//



tRetCode tParser::parseObject( tObject*& pObj )



{

    tRetCode nRet;



    class tObjectStub : public tObject {

        tObject* clone() const { return 0; }

    } ObjStub;



    static struct tDispatch {

        tToken m_nToken;

        tRetCode (*m_parse)( tParser&, tSystem&, tObject*& );

    } Dispatch[] = {

        { xTokBootstrap, tBootstrap::parse },

        { xTokCalendar, tCalendar::parse },

        { xTokClaim, tClaim::parse },

        { xTokCurve, tCurve::parse },

        { xTokDrift, tDrift::parse },

        { xTokEvaluate, tEvaluate::parse },

        { xTokFactor, tFactor::parse },

        { xTokImage, tImage::parse },

        { xTokInterestConvention, tInterestConvention::parse },

        { xTokLattice, tLattice::parse },

        { xTokModel, tModel::parse },

        { xTokOptimizer, tOptimizer::parse },

        { xTokPathSpace, tPathSpace::parse },

        { xTokPortfolio, tPortfolio::parse },

        { xTokScenario, tScenario::parse },

        { xTokVol, tVol::parse },

    };



    pObj = 0;



        // System specifications and shell objects are a

        // little different.

 

    while( curToken() == xTokSystem || curToken() == xTokShell ) {

        if( curToken() == xTokSystem ) {

            MTG_ASSERT( m_pRep != 0 );



            if( ( nRet = readToken() ) != OK ||

                ( nRet = tSystem::parse( *this, pObj ) ) != OK ) {

                return nRet;

            }

    

            m_pRep->setSystem( static_cast<tSystem*>( pObj ) );

        }

        else {

            if( m_pShell == 0 ) 

                setShell( new tShell );     // create own shell



                // call may never return:

            if( ( nRet = readToken() ) != OK ||

                ( nRet = m_pShell->parse( *this ) ) != OK ||

                ( nRet = m_pShell->autoActivate() ) != OK ) {

                return nRet;

            }

        }

    }



    if( curToken() == xTokEOF ) 

        return OK;



    for( int k = 0; k < (int) ( sizeof(Dispatch) / sizeof(tDispatch) ); ++k ) {

        if( Dispatch[k].m_nToken == curToken() ) {

            if( ( nRet = parsePrefix( ObjStub ) ) != OK ||

                ( nRet = Dispatch[k].m_parse(

                            *this, m_pRep->system(), pObj ) ) != OK ) {

                if( m_pShell != 0 && m_pShell->hasOnError() )

                    m_pShell->onError();

                return nRet;

            }

            movePrefix( ObjStub, pObj );



            if( ( nRet = pObj->autoActivate() ) != OK ) {

                delete pObj;

                pObj = 0;

                setError( nRet );

                if( m_pShell != 0 && m_pShell->hasOnError() )

                    m_pShell->onError();

                return nRet;

            }

            return OK;

        }

    }



    return setError( INVALID_KEYWORD );

}





//

//   p a r s e O b j e c t s

//



tRetCode tParser::parseObjects()



{

    tRetCode nRet;

    tEvaluate* pEvaluate;



    while( ( nRet = parseObjects( pEvaluate ) ) == OK &&

            pEvaluate != 0 ) {

        nRet = pEvaluate->run();

        delete pEvaluate;



        if( nRet != OK ) {

            setError( nRet );

            if( m_pShell != 0 && m_pShell->hasOnError() )

                m_pShell->onError();

            return nRet;

        }

    }



    return nRet;

}





//

//   p a r s e O b j e c t s

//



tRetCode tParser::parseObjects( tEvaluate*& pEvaluate )



{

    MTG_ASSERT( m_pRep != 0 );



    tObject* pObj;

    tRetCode nRet;



    pEvaluate = 0;

    while( ( nRet = parseObject( pObj ) ) == OK && pObj != 0 ) {

        if( ( pEvaluate = dynamic_cast<tEvaluate*>( pObj ) ) != 0 )

            return OK;

        m_pRep->insert( pObj );

    }



    return nRet;

}





//

//   p a r s e O b j e c t s

//



tRetCode tParser::parseObjects( tEvaluate*& pEvaluate, tRepository& Rep )



{

    tRetCode nRet;



    tRepository* pOldRep = m_pRep;

    m_pRep = &Rep;

    nRet = parseObjects( pEvaluate );

    m_pRep = pOldRep;



    return nRet;

}



MTG_END_NAMESPACE

