// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgRandom.h"

#include "MtgPathSpace.h"

#include "MtgDataScanner.h"

#include "MtgDateBase.h"

#include "MtgModel.h"

#include "MtgPortfolio.h"

#include "MtgSamplePath.h"



#include <float.h>



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tPathSpace::init()



{

    m_nNumOfFactors = 0;

    m_nSeed1 = 1234567890;

    m_nSeed2 = 123456789;

    m_nSize = 0;

    m_bAntithetic = true;

    m_bHasWeights = false;

    m_gGlobalWeight = 0;

    m_pModel = 0;

    m_pPortfolio = 0;

}





//

//   c o p y F r o m

//



void tPathSpace::copyFrom( const tPathSpace& Space )



{

    if( &Space == this )

        return;



    m_Maturity = Space.m_Maturity;

    m_nNumOfFactors = Space.m_nNumOfFactors;

    m_nSeed1 = Space.m_nSeed1;

    m_nSeed2 = Space.m_nSeed2;

    m_nSize = Space.m_nSize;

    m_bAntithetic = Space.m_bAntithetic;

    m_bHasWeights = Space.m_bHasWeights;

    m_WeightFile = Space.m_WeightFile;

    m_Weight = Space.m_Weight;

    m_gGlobalWeight = Space.m_gGlobalWeight;

    m_OutFile = Space.m_OutFile;

    

    setObjRefToZeroOrSo( m_pModel, Space.m_pModel );

    setObjRefToZeroOrSo( m_pPortfolio, Space.m_pPortfolio );



    super::copyFrom( Space );

    tDateBase::copyFrom( Space );

}





//

//   l o a d W e i g h t s

//



tRetCode tPathSpace::loadWeights()



{

    MTG_ASSERT( m_bHasWeights && m_WeightFile.isValid() );



    tRetCode nRet;



    tDataScanner Scanner;



    m_Weight.reset();

    if( ( nRet = Scanner.setSource( m_WeightFile ) ) != OK ||

        ( nRet = Scanner.append( m_Weight, 0 ) ) != OK ) {

        if( nRet == GO ) {

                // file doesn't exist; never mind

            m_bHasWeights = false;

        }

        else {

                // some other error

            return nRet;

        }

    }

    else {

        if( ! Scanner.atEOF() )

            return INVALID_CONSTANT;



        if( m_Weight.numOfElems() == 0 ) {

            if( m_WeightFile.isTry() )

                m_bHasWeights = false;

            else 

                return MISSING_WEIGHTS;

        }

        else {

                // override size

            m_nSize = m_Weight.numOfElems();

        }

    }



    return OK;

}





//

//   n o r m a l i z e W e i g h t s

//



void tPathSpace::normalizeWeights()



{

    MTG_ASSERT( m_bHasWeights && m_Weight.numOfElems() == m_nSize );



    double gSum = 0;



    for( int i = 0; i < m_nSize; ++i )

        gSum += m_Weight[i];



    if( gSum == 0 )

        return;



    for( MTG_FOR_INIT( int ) i = 0; i < m_nSize; ++i )

        m_Weight[i] /= gSum;

}





//

//   p a r s e P r e f i x

//



tRetCode tPathSpace::parsePrefix( tParser& Parser, tParseInfoStub& Info )



{

    setBase( system().base() );

    setDayCount( system().dayCount() );



    m_bHasWeights = false;



    return super::parsePrefix( Parser, Info );

}





//

//   p a r s e P a r a m

//



tRetCode tPathSpace::parseParam( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;

    tObject* pObj;

    tModel* pModel;

    tPortfolio* pPortfolio;

    tDayCount DayCount;



    tParseInfo& I = static_cast<tParseInfo&>( Info );



    nRet = DayCount.parse( Parser );



    if( nRet == OK ) {

        if( I.m_bDayCount ) 

            return Parser.setError( ATTR_REDEFINITION );

        setDayCount( DayCount );

        I.m_bDayCount = true;

        return nRet;

    }



    if( nRet != GO )

        return nRet;



    switch( Parser.curToken() ) {

        case xTokBase :

            if( I.m_bBase )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = tDateBase::parse( Parser ) ) != OK )

                return nRet;

            I.m_bBase = true;

            break;



        case xTokScale :

        case xTokYear :

        case xTokDay :

            if( I.m_bScale )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = tDateBase::parse( Parser ) ) != OK )

                return nRet;

            I.m_bScale = true;            

            break;



        case xTokModel :

            if( m_pModel != 0 )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() != xTokId )

                return Parser.setError( "model expected" );

            if( ( pObj = Parser.findObject() ) == 0 )

                return Parser.setError( NOT_FOUND );

            if( ( pModel = dynamic_cast<tModel*>( pObj ) ) == 0 ) 

                return Parser.setError( "model expected" );

            if( m_nNumOfFactors > 0 &&

                m_pModel->numOfFactors() != m_nNumOfFactors ) {

                return Parser.setError( INVALID_WIDTH );

            }

            setObjRefToZeroOrSo( m_pModel, pModel );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        case xTokPortfolio :

            if( m_pPortfolio != 0 )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() != xTokId )

                return Parser.setError( "portfolio expected" );

            if( ( pObj = Parser.findObject() ) == 0 )

                return Parser.setError( NOT_FOUND );

            if( ( pPortfolio = dynamic_cast<tPortfolio*>( pObj ) ) == 0 ) 

                return Parser.setError( "portfolio expected" );

            setObjRefToZeroOrSo( m_pPortfolio, pPortfolio );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        case xTokSeed :

            if( I.m_bSeed )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.scanInteger( m_nSeed1, 1 ) ) != OK ) {

                return nRet;

            }

            if( Parser.beginOfNumber() ) {

                if( ( nRet = Parser.scanInteger( m_nSeed2, 1 ) ) != OK )

                    return nRet;

            }

            else {

                m_nSeed2 = 123456789;

            }

            I.m_bSeed = true;

            break;



        case xTokSize :

            if( m_nSize > 0 )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.scanInteger( m_nSize, 1 ) ) != OK ) {

                return nRet;

            }

            break;



        case xTokWidth :

            if( m_nNumOfFactors > 0 )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.scanInteger( m_nNumOfFactors, 1 ) ) != OK ) {

                return nRet;

            }

            if( m_pModel != 0 &&

                m_pModel->numOfFactors() != m_nNumOfFactors ) {

                return Parser.setError( INVALID_WIDTH );

            }

            break;



        case xTokMaturity :

            if( I.m_bSamples )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.scanDate( m_Maturity ) ) != OK ) {

                return nRet;

            }

            I.m_bSamples = true;

            break;



        case xTokWeight :

            if( m_bHasWeights )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.beginOfObj() ) {

                if( ( nRet = Parser.scanNonNegTable( m_Weight ) ) != OK )

                    return nRet;

                if( m_Weight.numOfElems() == 0 )

                    return Parser.setError( MISSING_WEIGHTS );

            }

            else {

                if( ( nRet = m_WeightFile.parseRead( Parser ) ) != OK )

                    return nRet;

            }

            m_bHasWeights = true;

            break;



        case xTokSave :

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;



            //switch( Parser.curToken() ) {

            //    default :

                    if( m_OutFile.isValid() )

                        return Parser.setError( ATTR_REDEFINITION );

                    if( Parser.curToken() == xTokPathSpace ) {

                        if( ( nRet = Parser.readToken() ) != OK )

                            return nRet;

                    }

                    if( ( nRet = m_OutFile.parseWrite( Parser ) ) != OK )

                        return nRet;

            //        break;

            //}

            break;



        default :

            if( ( nRet = super::parseParam( Parser, Info ) ) != OK )

                return nRet;

    }



    return OK;

}





//

//   p a r s e P o s t f i x

//



tRetCode tPathSpace::parsePostfix( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;

    tParseInfo& I = static_cast<tParseInfo&>( Info );



    if( m_pModel == 0 && m_nNumOfFactors <= 0 )

        return Parser.setError( MISSING_MODEL );

    if( m_pPortfolio == 0 && ! I.m_bSamples )

        return Parser.setError( MISSING_PORTFOLIO );



    if( ! I.m_bBase && m_pPortfolio != 0 )

        setBase( m_pPortfolio->settlement() );



    if( m_nNumOfFactors <= 0 ) 

        m_nNumOfFactors = m_pModel->numOfFactors();



    if( m_bHasWeights ) {

        if( m_WeightFile.isValid() ) {

            if( ( nRet = loadWeights() ) != OK )

                return Parser.setError( nRet );

        }

        else {

            if( m_nSize > 0 && m_nSize != m_Weight.numOfElems() )

                return Parser.setError( INVALID_SIZE );

            m_nSize = m_Weight.numOfElems();

        }

        if( m_bHasWeights )

            normalizeWeights();

    }



    if( m_nSize <= 0 )

        return Parser.setError( MISSING_SIZE );



    if( m_pPortfolio != 0 ) {

        tDate M = m_pPortfolio->maturityDate();



        if( ! I.m_bSamples || M > m_Maturity )

            m_Maturity = M;

    }



    if( m_Maturity <= base() )

        return Parser.setError( INVALID_MATURITY );



    m_gGlobalWeight = 1.0 / m_nSize;



    return super::parsePostfix( Parser, Info );

}





//

//   t P a t h S p a c e

//



tPathSpace::tPathSpace()



{

    init();

}





//

//   t P a t h S p a c e

//



tPathSpace::tPathSpace( const tPathSpace& Space )



{

    init();

    copyFrom( Space );

}





//

//   ~ t P a t h S p a c e

//



tPathSpace::~tPathSpace()



{

    setObjRefToZeroOrSo( m_pModel );

    setObjRefToZeroOrSo( m_pPortfolio );

}





//

//   o p e r a t o r =

//



tPathSpace& tPathSpace::operator=( const tPathSpace& Space )



{

    if( &Space != this )

        copyFrom( Space );

    return *this;

}





//

//   c l o n e

//



tObject* tPathSpace::clone() const



{

    return new tPathSpace( *this );

}





//

//   f i n a l i z e

//



tRetCode tPathSpace::finalize()



{

    setNumOfSamples( m_Maturity );

    return super::finalize();

}





//

//   h a s M o d e l

//



bool tPathSpace::hasModel() const



{

    MTG_ASSERT( isFinalized() );

    return m_pModel != 0;

}





//

//   m o d e l

//



tModel& tPathSpace::model() const



{

    MTG_ASSERT( isFinalized() && m_pModel != 0 );

    return *m_pModel;

}





//

//   h a s P o r t f o l i o

//



bool tPathSpace::hasPortfolio() const



{

    MTG_ASSERT( isFinalized() );

    return m_pPortfolio != 0;

}





//

//   p o r t f o l i o

//



tPortfolio& tPathSpace::portfolio() const



{

    MTG_ASSERT( isFinalized() && m_pPortfolio != 0 );

    return *m_pPortfolio;

}





//

//   m o d e l M a t c h e s

//



bool tPathSpace::modelMatches( const tModel& Model ) const



{

    MTG_ASSERT( isFinalized() );



    if( m_pModel != 0 )

        return m_pModel == &Model;

    return m_nNumOfFactors == Model.numOfFactors();

}





//

//   g e n e r a t e P a t h

//



void tPathSpace::generatePath( tSamplePath& Path, tRandom& Random,

    int nSeqNo, double& gWeight ) const



{

    MTG_ASSERT( isFinalized() );

    MTG_ASSERT( nSeqNo >= 0 );



    if( nSeqNo == 0 )

        Random.setSeed( m_nSeed1, m_nSeed2 );



    if( m_bAntithetic && nSeqNo % 2 == 1 ) {

        Path.transform( -1 );

    }

    else {

        Path.setNumOfFactors( m_nNumOfFactors );

        Path.setNumOfSamples( numOfSamples() );

        Path.build( Random );

    }



    if( m_bHasWeights )

        gWeight = m_Weight[nSeqNo % m_nSize];

    else

        gWeight = m_gGlobalWeight;

}





//

//   g e n e r a t e P a t h

//



void tPathSpace::generatePath( tSamplePath& Path, tRandom& Random,

    int nSeqNo ) const



{

    double gWeight;

    generatePath( Path, Random, nSeqNo, gWeight );

}





//

//   s a v e

//



tRetCode tPathSpace::save( ostream& Out,

    const tHeap<double>* pWeight ) const



{

    MTG_ASSERT( isFinalized() );

    MTG_ASSERT( pWeight == 0 || pWeight->numOfElems() == m_nSize );



    tRetCode nRet;



    if( pWeight == 0 && m_bHasWeights )

        pWeight = &m_Weight;



    Out << "path_space " << name() << " {\n"

        << "    "

        << static_cast<const tDateRange&>( *this ) << ",\n";



    if( m_pModel != 0 )

        Out << "    model " << m_pModel->name() << ",\n";



    Out << "    maturity \"" << m_Maturity << "\",\n"

        << "    seed " << m_nSeed1 << " " << m_nSeed2 << ",\n"

        << "    width " << m_nNumOfFactors << ",\n"

        << "    size " << m_nSize;



    if( ! Out.good() )

        return WRITE_ERROR;



    if( pWeight != 0 ) {

        Out << ",\n    weight {\n";

        Out << setprecision( DBL_DIG );

        if( ( nRet = pWeight->save( Out, 4, "        " ) ) != OK )

            return nRet;

        Out << "    }";

    }



    Out << "\n}" << endl;



    return Out.good() ? OK : WRITE_ERROR;

}





//

//   s a v e

//



tRetCode tPathSpace::save( ostream& Out,

    const tHeap<double>& Weight ) const



{

    return save( Out, &Weight );

}





//

//   s a v e

//



tRetCode tPathSpace::save( ostream& Out ) const



{

    return save( Out, 0 );

}





//

//   s a v e

//



tRetCode tPathSpace::save( const tFileName& Out,

    const tHeap<double>* pWeight ) const



{

    if( ! Out.isValid() )

        return save( pWeight );



    ofstream S;

    tRetCode nRet;



    if( ( nRet = Out.openWrite( S, 0 ) ) != OK )

        return nRet;



    return save( S, pWeight );

}





//

//   s a v e

//



tRetCode tPathSpace::save( const tFileName& Out,

    const tHeap<double>& Weight ) const



{

    return save( Out, &Weight );

}





//

//   s a v e

//



tRetCode tPathSpace::save( const tFileName& Out ) const



{

    return save( Out, 0 );

}





//

//   s a v e

//



tRetCode tPathSpace::save( const tHeap<double>* pWeight ) const



{

    if( ! m_OutFile.isValid() )

        return OK;



    ofstream S;

    tRetCode nRet;



    if( ( nRet = m_OutFile.openWrite( S, 0 ) ) != OK )

        return nRet;



    return save( S, pWeight );

}





//

//   s a v e

//



tRetCode tPathSpace::save( const tHeap<double>& Weight ) const



{

    return save( &Weight );

}





//

//   s a v e

//



tRetCode tPathSpace::save() const



{

    return save( 0 );

}





//

//   p a r s e

//



tRetCode tPathSpace::parse( tParser& Parser, tSystem& System, tObject*& pObj )



{

    tRetCode nRet;

    tPathSpace* pSpace;

    tParseInfo Info;



    Info.m_bBase = false;

    Info.m_bScale = false;

    Info.m_bDayCount = false;

    Info.m_bSeed = false;

    Info.m_bSamples = false;



    if( ( nRet = Parser.scanBeginOfObj() ) != OK )

        return nRet;



    pSpace = new tPathSpace;

    pSpace->setSystem( System );



    if( ( nRet = pSpace->super::parse( Parser, &Info ) ) != OK ) {

        delete pSpace;

        return nRet;

    }



    pObj = pSpace;

    return OK;

}



MTG_END_NAMESPACE