// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#if ! defined(_MTG_PROFILE_)

#define _MTG_PROFILE_



#include "MtgArrayBounds.h"

#include "MtgSignature.h"



MTG_BEGIN_NAMESPACE



#if ! defined(_MTG_PROFILE_READONLY)



class tTimeAxis;

class tSpaceAxis;

class tSlotLayer;



#endif





//

//   t P r o f i l e

//



class tProfile {



protected:



    struct tItem {

        int m_cType;    // character code

        int m_nId;      // may or may not be used



        tItem() : m_cType( '?' ), m_nId( 0 ) {}

        tItem( int cType ) : m_cType( cType ), m_nId( 0 ) {}

        tItem( int cType, int nId ) : m_cType( cType ), m_nId( nId ) {}



        virtual ~tItem() {}



#if ! defined(_MTG_PROFILE_READONLY)

        virtual tRetCode write( FILE* fFile ) const = 0;

        static tRetCode write( FILE* fFile, const tArrayBounds& Bounds );

#endif



        virtual tRetCode read( FILE* fFile ) = 0;

        static tRetCode read( FILE* fFile, tArrayBounds& Bounds );

        static tRetCode read( FILE* fFile, char cExpected );

    };



        // These are the records that can be inserted into the profile:



    struct tAlternativeItem : public tItem {

		int m_nComplexity;

        int m_nLayer;



#if ! defined(_MTG_PROFILE_READONLY)

        tAlternativeItem( int nComplexity, int nLayer );

        tRetCode write( FILE* fFile ) const;

#endif



        tAlternativeItem() : tItem( 'A' ) {}

        tRetCode read( FILE* fFile );

    };



    struct tEngineItem : public tItem {



#if ! defined(_MTG_PROFILE_READONLY)

        tEngineItem( int nId ) : tItem( 'G', nId ) {}

        tRetCode write( FILE* fFile ) const;

#endif



        tEngineItem() : tItem( 'G' ) {}

        tRetCode read( FILE* fFile );

    };



    struct tExPolicyItem : public tItem {

        int m_nIndex;

        tExPolicy m_nExPolicy;

        double m_gPayoff;



#if ! defined(_MTG_PROFILE_READONLY)

        tExPolicyItem( int nIndex, tExPolicy nExPolicy, double gPayoff );

        tRetCode write( FILE* fFile ) const;

#endif



        tExPolicyItem() : tItem( 'E' ) {}

        tRetCode read( FILE* fFile );

    };



    struct tLatticeItem : public tItem {

        int m_nNumOfSlices;

        tArrayBounds m_Bounds;



        tHeap<double> m_AxesConcat; // All axes concatenated.



#if ! defined(_MTG_PROFILE_READONLY)

        const tTimeAxis* m_pTime;

        const tHeap<tSpaceAxis*>* m_pSpace;



        tLatticeItem( int nId, const tArrayBounds& Bounds,

            const tTimeAxis& Time, const tHeap<tSpaceAxis*>& Space );

        tRetCode write( FILE* fFile ) const;

#endif



        tLatticeItem() : tItem( 'L' ) {}

        tRetCode read( FILE* fFile );



        ~tLatticeItem() {}

    };



    struct tLevelItem : public tItem {

        tHeap<int> m_Level;



#if ! defined(_MTG_PROFILE_READONLY)

        tLevelItem( int nLevel );

        tLevelItem( const tHeap<int>& Level );

        tRetCode write( FILE* fFile ) const;

#endif



        tLevelItem() : tItem( 'P' ) {}

        tRetCode read( FILE* fFile );



        ~tLevelItem() {}

    };



    struct tRepeatItem : public tItem {



#if ! defined(_MTG_PROFILE_READONLY)

        tRetCode write( FILE* fFile ) const { return OK; }

#endif



        tRepeatItem() : tItem( 'R' ) {}

        tRetCode read( FILE* fFile ) { return OK; }

    };



    struct tSlotLayerItem : public tItem {

        tArrayBounds m_Bounds;

        tSignature m_Sig;



#if ! defined(_MTG_PROFILE_READONLY)

        tSlotLayerItem( int nId, const tArrayBounds& Bounds,

            const tSignature& Sig );

        tRetCode write( FILE* fFile ) const;

#endif



        tSlotLayerItem() : tItem( 'S' ) {}

        tRetCode read( FILE* fFile );



        ~tSlotLayerItem() {}

    };



    struct tTaskItem : public tItem {

        int m_nLayer;

        int m_nSlice;

        int m_nDay;

        double m_gFractionOfDay;



#if ! defined(_MTG_PROFILE_READONLY)

        tTaskItem( int nLayer, int nSlice, int nDay, double gFractionOfDay );

        tRetCode write( FILE* fFile ) const;

#endif



        tTaskItem() : tItem( 'T' ) {}

        tRetCode read( FILE* fFile );

    };



    struct tValueItem : public tItem {

		enum tType {

			xVolatility,

			xTotal

		};



        tType m_nType;		

		double m_gValue;

		

#if ! defined(_MTG_PROFILE_READONLY)

        tValueItem( tType m_nType, double gValue );

        tRetCode write( FILE* fFile ) const;

#endif



        tValueItem() : tItem( 'V' ) {}

        tRetCode read( FILE* fFile );

    };



private:



    char* m_sFileName;

    FILE* m_fFile;



#if ! defined(_MTG_PROFILE_READONLY)



    int m_nNextGeneralId;

    int m_nNextLayerId;



    bool m_bExhaustive;

    bool m_bActive;



#endif



    void init();



    tRetCode skipItem();

    tRetCode getType( char& cType );

    tRetCode getItem( tItem*& pItem, char cType );



        // No copying is allowed for profiles, so the copy constructor

        // and the assignment operator are private.



    tProfile( const tProfile& Profile ) {

        throw tException( INTERNAL_ERROR );

    }



    tProfile& operator=( const tProfile& Profile ) {

        throw tException( INTERNAL_ERROR );

        return *this;

    }



#if ! defined(_MTG_PROFILE_READONLY)



		// This one is private, there's a separate function for

		// each type:



	void putValue( tValueItem::tType nType, double gValue );



#endif



protected:



    tRetCode get( tItem*& pItem );

    tRetCode get( tItem*& pItem, char cType );

    tRetCode get( tItem*& pItem, char cType1, char cType2 );

    tRetCode get( tItem*& pItem, const char* sTypeSet );



public:



        // This object is not threadsafe. Every thread/process

        // must create its own Profile!



    tProfile();

    tProfile( const char* sFileName );



    ~tProfile();



        // Leaves the extension intact:



    void setFileName( const char* sFileName );



        // Changes the extension to ".prof":



    void useFileName( const char* sFileName );



    const char* fileName() const {

        return m_sFileName;

    }



    tRetCode openRead();

    tRetCode openRead( const char* sFileName );

	void rewind();



    void close();



    bool isOpen() const {

        return m_fFile != 0;

    }



#if ! defined(_MTG_PROFILE_READONLY)



    tRetCode useArgs( int& argc, const char* argv[] );

    void activate( bool bExhaustive = true );



    tRetCode openWrite();

    tRetCode openWrite( const char* sFileName );



    bool isActive() const {

        return m_bActive;

    }



    bool isExhaustive() const {

        return m_bExhaustive;

    }



    int nextGeneralId() {

        return m_nNextGeneralId++;

    }



    int nextLayerId() {

        return m_nNextLayerId++;

    }



        // Note that if the record cannot be inserted, an exception

        // is thrown!



    void put( const tItem& Item );



    void putAlternative( int nComplexity );

    void putAlternative( int nComplexity, const tSlotLayer* pLayer );



    int putEngine();



    void putExPolicy( int nIndex, tExPolicy nExPolicy, double gPayoff );



    int putLattice( const tArrayBounds& Bounds, const tTimeAxis& Time,

        const tHeap<tSpaceAxis*>& Space );



    void putLevel( int nLevel );

    void putLevel( const tHeap<int>& Level );



    void putRepeat();



    int putSlotLayer( const tArrayBounds& Bounds, const tSignature& Sig );



    void putTask( const tSlotLayer& Layer, int nSlice, int nDay,

        double gFractionOfDay );



		// Different versions for tValueItem:



    void putTotal( double gValue ) {

		putValue( tValueItem::xTotal, gValue );

	}



    void putVolatility( double gValue ) {

		putValue( tValueItem::xVolatility, gValue );

	}



#endif



};



#if ! defined(_MTG_PROFILE_READONLY)



//

//   t W i t h W a k e

//



class tWithProfile {



private :



    tProfile* m_pProfile;

    int m_nId;



protected :



    bool hasProfile() const {

        return m_pProfile != 0;

    }



    tProfile& profile() const {

        return *m_pProfile;

    }



    void setProfileId( int nId ) {

        m_nId = nId;

    }



public: int profileId() const {

        return m_nId;

    }



public :



    tWithProfile() {

        m_pProfile = 0;

        m_nId = 0;

    }



    void setProfile( tProfile& Profile ) {

        setProfile( &Profile );

    }



    void setProfile( tProfile* pProfile = 0 ) {

        m_pProfile = pProfile;

    }



    tWithProfile& copyFrom( const tWithProfile& WithProfile ) {

        m_pProfile = WithProfile.m_pProfile;

        m_nId = WithProfile.m_nId;

        return *this;

    }



    void copyTo( tWithProfile& WithProfile ) const {

        copyTo( &WithProfile );

    }



    void copyTo( tWithProfile* pWithProfile ) const {

        if( pWithProfile != 0 ) {

            pWithProfile->m_pProfile = m_pProfile;

            pWithProfile->m_nId = m_nId;

        }

    }



    friend class tProfile;

};



#endif



MTG_END_NAMESPACE



#endif

