// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgRandom.h"



#include "ranlib.h"



MTG_USING_OTHER_NAMESPACE( ranlib )



MTG_BEGIN_NAMESPACE





//

//   v a r i a b l e s

//



#if defined(_WIN32)



long tRandom::m_nInitialized = 0;

CRITICAL_SECTION tRandom::m_Critical;





//

//   i n i t i a l i z e

//



void tRandom::initialize()



{

    long n = (long) InterlockedCompareExchange(

                        (PVOID*) &m_nInitialized, (PVOID) 1, (PVOID) 0 );



    if( n == 0 ) {

            // not initialized yet; now it's 1

        InitializeCriticalSection( &m_Critical );

            // dummy call

        setall( 1, 1 );

            // set it to 2 to indicate full initialization

        InterlockedExchange( &m_nInitialized, 2 );

    }

    else

    if( n == 1 ) {

            // someone is initializing it; we have to wait...

        while( m_nInitialized != 2 )

            Sleep( 0 );

    }

}



#endif





//

//   s e t S e e d

//



void tRandom::setSeed( long nSeed1, long nSeed2, long nGenId )



{

        // The boundaries are taken from the ranlib documentation.



    if( nSeed1 < 1 )

        nSeed1 = 1;

    else

        if( nSeed1 > 2147483562 )

            nSeed1 = 2147483562;



    if( nSeed2 < 1 )

        nSeed2 = 1;

    else 

        if( nSeed2 > 2147483398 )

            nSeed2 = 2147483398;



    m_nSeed1 = nSeed1;

    m_nSeed2 = nSeed2;

    m_nGenId = nGenId;

}





//

//   s e t S e e d

//



void tRandom::setSeed()



{

    gscgn( 1, &m_nGenId );

    setsd( m_nSeed1, m_nSeed2 );

}





//

//   g e t S e e d

//



void tRandom::getSeed()



{

    gscgn( 0, &m_nGenId );

    getsd( &m_nSeed1, &m_nSeed2 );

}





//

//   l o c k

//



void tRandom::lock()



{

#if defined(_WIN32)

    EnterCriticalSection( &m_Critical );

#endif

    setSeed();

}





//

//   u n l o c k

//



void tRandom::unlock()



{

    getSeed();

#if defined(_WIN32)

    LeaveCriticalSection( &m_Critical );

#endif

}





//

//   t R a n d o m

//



tRandom::tRandom()



{

#if defined(_WIN32)

    initialize();

#endif

    setSeed( 1234567890, 123456789, 1 );

}





//

//   t R a n d o m

//



tRandom::tRandom( long nSeed1, long nSeed2 )



{

#if defined(_WIN32)

    initialize();

#endif

    setSeed( nSeed1, nSeed2, 1 );

}





//

//   t R a n d o m

//



tRandom::tRandom( const tRandom& Random )



{

    m_nSeed1 = Random.m_nSeed1;

    m_nSeed2 = Random.m_nSeed2;

    m_nGenId = Random.m_nGenId;

}





//

//   o p e r a t o r =

//



tRandom& tRandom::operator=( const tRandom& Random )



{

    if( &Random != this ) {

        m_nSeed1 = Random.m_nSeed1;

        m_nSeed2 = Random.m_nSeed2;

        m_nGenId = Random.m_nGenId;

    }



    return *this;

}





//

//   s e t S e e d

//



void tRandom::setSeed( long nSeed1, long nSeed2 )



{

    setSeed( nSeed1, nSeed2, 1 );

}





//

//   g e t S e e d

//



void tRandom::getSeed( long& nSeed1, long& nSeed2 )



{

    nSeed1 = m_nSeed1;

    nSeed2 = m_nSeed2;

}





//

//   u n i f o r m

//



double tRandom::uniform()



{

    lock();

    double g = (double) ranf();

    unlock();

    return g;

}





//

//   u n i f o r m

//



void tRandom::uniform( tHeap<double>& Seq )



{

    int n = Seq.numOfElems();



    lock();

    for( int i = 0; i < n; ++i ) 

        Seq[i] = (double) ranf();

    unlock();

}





//

//   u n i f o r m

//



void tRandom::uniform( tHeap2<double>& Seq )



{

    int r = Seq.numOfRows();

    int c = Seq.numOfCols();



    lock();

    for( int i = 0; i < r; ++i ) {

        double* x = Seq[i];

        for( int j = 0; j < c; ++j )

            x[j] = (double) ranf();

    }

    unlock();

}





//

//   n o r m a l

//



double tRandom::normal()



{

    lock();

    double g = (double) snorm();

    unlock();

    return g;

}





//

//   n o r m a l

//



void tRandom::normal( tHeap<double>& Seq )



{

    int n = Seq.numOfElems();



    lock();

    for( int i = 0; i < n; ++i ) 

        Seq[i] = (double) snorm();

    unlock();

}





//

//   n o r m a l

//



void tRandom::normal( tHeap2<double>& Seq )



{

    int r = Seq.numOfRows();

    int c = Seq.numOfCols();



    lock();

    for( int i = 0; i < r; ++i ) {

        double* x = Seq[i];

        for( int j = 0; j < c; ++j )

            x[j] = (double) snorm();

    }

    unlock();

}



MTG_END_NAMESPACE