// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgScanner.h"



MTG_BEGIN_NAMESPACE





//

//   c l e a n u p

//



void tScanner::cleanup()



{

    if( m_Stack.numOfElems() > 0 ) {

        m_pSrc = m_Stack[0];    // original source, referenced;



        for( int i = 1; i < m_Stack.numOfElems(); ++i ) {

            m_Stack[i]->end();

            delete m_Stack[i];  // temporary sources, owned

        }



        m_Stack.reset();

    }



    if( m_pSrc != 0 ) {

        m_pSrc->end();

        m_pSrc = 0;

    }

}





//

//   c l e a r E r r o r

//



void tScanner::clearError()



{

    m_sErrorMsg[0] = 0;

}





//

//   c h e c k I n c l u d e

//



tRetCode tScanner::checkInclude()



{

    tRetCode nRet;

    bool bTry;



    while( true ) {

        while( curToken() == xTokEOF ) {

            if( m_Stack.numOfElems() == 1 )

                --m_Stack;

            if( m_Stack.numOfElems() == 0 )

                goto end;



                // close and delete temporary file

                

            tSource* pSrc;

            m_Stack.pop( pSrc );



            pSrc->end();

            delete pSrc;



            m_pSrc = m_Stack.last();

            MTG_ASSERT( curToken() == xTokString );



            if( ( nRet = m_pSrc->readToken() ) != OK )

                return setError( nRet );

        }       



        if( curToken() != xTokInclude )

            goto end;



        if( m_pSrc->isSafe() )

            return setError( INVALID_KEYWORD );



            // include new file



        if( ( m_pSrc->readToken() ) != OK )

            return setError( nRet );



        if( curToken() == xTokTry ) {

            if( ( m_pSrc->readToken() ) != OK )

                return setError( nRet );

            bTry = true;

        }

        else {

            bTry = false;

        }



        if( curToken() != xTokString )

            return setError( INVALID_KEYWORD );



        tFileSource* pSrc = new tFileSource( m_pSrc->m_sCurText );



        pSrc->makeSafe();



        if( ( nRet = pSrc->start() ) != OK ) {

            if( ! bTry || nRet != OPEN_ERROR ) {

                delete pSrc;

                return setError( nRet );

            }

                // simply ignore

            if( ( m_pSrc->readToken() ) != OK )

                return setError( nRet );

        }

        else {

            if( m_Stack.numOfElems() == 0 )

                m_Stack.push( m_pSrc );

            m_Stack.push( pSrc );

            m_pSrc = pSrc;

        }

    }



end:



    return OK;

}





//

//   t S c a n n e r

//



tScanner::tScanner()



{

    m_pSrc = 0;

    clearError();

}





//

//   ~ t S c a n n e r

//



tScanner::~tScanner()



{

    cleanup();

}





//

//   s e t S o u r c e

//



tRetCode tScanner::setSource( tSource* pSrc )



{

    tRetCode nRet;



    cleanup();

    clearError();



    if( pSrc != 0 ) {

        if( ( nRet = pSrc->start() ) != OK ) {

            setError( nRet );

            return nRet;

        }

        m_pSrc = pSrc;

        if( ( nRet = checkInclude() ) != OK ) {

            m_pSrc->end();

            m_pSrc = 0;

            return nRet;

        }

    }

    return OK;

}





//

//   s e t E r r o r

//



tRetCode tScanner::setError( const char *sFmt, ... )



{

    va_list Marker;



    if( m_sErrorMsg[0] )

        return SYNTAX_ERROR;



    if( m_pSrc == 0 ) {

        sprintf( m_sErrorMsg, "Runtime : " );

    }

    else {

        sprintf( m_sErrorMsg, "%.*s(%d) : ",

            sizeof(m_sErrorMsg) / 2,

            m_pSrc->name(), m_pSrc->getCurLine() );

    }



    char* s = m_sErrorMsg + strlen( m_sErrorMsg );



    va_start( Marker, sFmt );

    vsprintf( s, sFmt, Marker );



    return SYNTAX_ERROR;

}





//

//   s e t E r r o r

//



tRetCode tScanner::setError( tRetCode nRet )



{

    switch( nRet ) {

        case NOT_FOUND :

                // try to be intelligent

            if( m_pSrc != 0 && curToken() == xTokId ) {

                setError( "object '%.*s' not found",

                    sizeof(m_sErrorMsg) / 4, curText() );

            }

            else {

                setError( "object not found" );

            }

            break;



        case TIMEOUT :

            setError( "timeout" );

            break;



        case FORMAT_ERROR :

            setError( "format error" );

            break;



        case SYNTAX_ERROR :

            setError( "syntax error" );

            break;



        case ATTR_REDEFINITION :

            setError( "redefinition of attribute" );

            break;



        case MALFORMED_EXPRESSION :

            setError( "malformed expression" );

            break;



        case INVALID_KEYWORD :

                // try to be intelligent

            if( m_pSrc != 0 ) {

                setError( "unexpected keyword at or before '%.*s'",

                    sizeof(m_sErrorMsg) / 4, curText() );

            }

            else {

                setError( "unexpected keyword" );

            }

            break;



        case INVALID_CONSTANT :

            setError( "invalid constant" );

            break;



        case INVALID_STRING :

            setError( "invalid string" );

            break;



        case INVALID_DATE :

                // try to be intelligent

            if( m_pSrc != 0 && curToken() == xTokString ) {

                setError( "invalid date '%.*s'",

                    sizeof(m_sErrorMsg) / 4, curText() );

            }

            else {

                setError( "invalid date" );

            }

            break;



        case INVALID_TIME :

                // try to be intelligent

            if( m_pSrc != 0 && curToken() == xTokString ) {

                setError( "invalid time '%.*s'",

                    sizeof(m_sErrorMsg) / 4, curText() );

            }

            else {

                setError( "invalid time" );

            }

            break;



        case INVALID_ID :

            setError( "invalid id" );

            break;



        case INVALID_VOLATILITY :

            setError( "invalid volatility" );

            break;



        case INVALID_MATURITY :

            setError( "invalid maturity" );

            break;



        case INVALID_SETTLEMENT :

            setError( "invalid settlement" );

            break;



        case INVALID_SIZE :

            setError( "invalid size" );

            break;



        case INVALID_WIDTH :

            setError( "invalid width" );

            break;



        case RUNTIME_ERROR :

            setError( "runtime error" );

            break;



        case NOT_AVAILABLE :

            setError( "feature not available" );

            break;



        case DIVISION_BY_ZERO :

            setError( "division by zero" );

            break;



        case OVERFLOW :

            setError( "overflow" );

            break;



        case NO_CONVERGENCE :

            setError( "no convergence" );

            break;



        case OUT_OF_RANGE :

            setError( "out of range" );

            break;



        case DATEBASE_MISMATCH :

            setError( "datebase mismatch" );

            break;



        case SETTLEMENT_CONFLICT :

            setError( "conflict in settlement dates" );

            break;



        case OPEN_ERROR :

                // try to be intelligent

            if( m_pSrc != 0 && curToken() == xTokString ) {

                setError( "open of '%.*s' failed",

                    sizeof(m_sErrorMsg) / 4, curText() );

            }

            else {

                setError( "open failed" );

            }

            break;



        case READ_ERROR :

            setError( "read failed" );

            break;



        case WRITE_ERROR :

            setError( "write failed" );

            break;



        case SCRIPT_ERROR :

            setError( "error in script" );

            break;



        case NOT_IMPLEMENTED :

            setError( "feature not implemented" );

            break;



        case OUT_OF_MEMORY :

            setError( "out of memory" );

            break;



        case INTERNAL_ERROR :

            setError( "internal error; contact buff@cs.nyu.edu" );

            break;



        default :

            setError( "error %d", nRet );

            break;

    }

    return nRet;

}





//

//   b e g i n O f N u m b e r

//



bool tScanner::beginOfNumber()



{

    if( curToken() == xTokNumber ||

        curToken() == xTokPlus ||

        curToken() == xTokMinus ) {

        return true;

    }

    return false;

}





//

//   r e a d T o k e n

//



tRetCode tScanner::readToken()



{

    tRetCode nRet;



    if( ( nRet = m_pSrc->readToken() ) != OK ||

        ( nRet = checkInclude() ) != OK ) {

        setError( nRet );

    }



    return nRet;

}





//

//   s w a l l o w T o k e n

//



tRetCode tScanner::swallowToken( tToken nToken, char* sPrintName )



{

    if( curToken() != nToken )

        return setError( "token '%s' expected", sPrintName );

    if( curToken() != xTokEOF )

        return readToken();

    return OK;

}





//

//   s w a l l o w C o m m a

//



tRetCode tScanner::swallowComma( bool& bGo )



{

    if( curToken() != xTokComma ) {

        bGo = false;

        return OK;

    }

    bGo = true;

    return readToken();

}





//

//   s c a n I n t e g e r

//



tRetCode tScanner::scanInteger( long& nValue, long nLowerBound )



{

    int n;

    tRetCode nRet;



    if( ( nRet = scanInteger( n, (int) nLowerBound ) ) != OK )

        return nRet;



    nValue = (long) n;

    return OK;

}





//

//   s c a n I n t e g e r

//



tRetCode tScanner::scanInteger( int& nValue, int nLowerBound )



{

    bool bOk;

    int nSign;

    tRetCode nRet;



    switch( curToken() ) {

        case xTokMinus :

            if( ( nRet = readToken() ) != OK )

                return nRet;

            nSign = -1;

            break;



        case xTokPlus :

            if( ( nRet = readToken() ) != OK )

                return nRet;

            nSign = 1;

            break;



        default :

            nSign = 1;

            break;

    }



    if( curToken() == xTokEOF )

        return setError( END_OF_FILE );



    if( curToken() != xTokNumber )

        return setError( "integer expected" );



    m_pSrc->m_gCurValue *= nSign;



    if( m_pSrc->m_gCurValue < nLowerBound ) {

        bOk = false;

    }

    else {

        nValue = (int) m_pSrc->m_gCurValue;

        bOk = ( (double) nValue == m_pSrc->m_gCurValue );

    }



    if( ! bOk )

        return setError( "invalid integer" );



    return readToken();

}





//

//   s c a n D o u b l e

//



tRetCode tScanner::scanDouble( double& gValue )



{

    int nSign;

    tRetCode nRet;



    switch( curToken() ) {

        case xTokMinus :

            if( ( nRet = readToken() ) != OK )

                return nRet;

            nSign = -1;

            break;



        case xTokPlus :

            if( ( nRet = readToken() ) != OK )

                return nRet;

            nSign = 1;

            break;



        default :

            nSign = 1;

            break;

    }



    if( curToken() == xTokEOF )

        return setError( END_OF_FILE );



    if( curToken() != xTokNumber )

        return setError( "real number expected" );



    m_pSrc->m_gCurValue *= nSign;

    gValue = m_pSrc->m_gCurValue;



    return readToken();

}





//

//   s c a n N o n N e g D o u b l e

//



tRetCode tScanner::scanNonNegDouble( double& gValue )



{

    tRetCode nRet;



    if( ( nRet = scanDouble( gValue ) ) != OK )

        return nRet;



    if( gValue < 0 )

        return setError( OUT_OF_RANGE );



    return OK;

}





//

//   s c a n P o s D o u b l e

//



tRetCode tScanner::scanPosDouble( double& gValue )



{

    tRetCode nRet;



    if( ( nRet = scanDouble( gValue ) ) != OK )

        return nRet;



    if( gValue <= 0 )

        return setError( OUT_OF_RANGE );



    return OK;

}





//

//   s c a n R a n g e

//



tRetCode tScanner::scanRange( double& gMin, double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanDouble( gMin ) ) != OK )

        return nRet;



    if( curToken() == xTokRange ) {

        if( ( nRet = readToken() ) != OK ||

            ( nRet = scanDouble( gMax ) ) != OK ) {

            return nRet;

        }



        if( gMax < gMin ) {

            double t = gMin; gMin = gMax; gMax = t;

        }

    }

    else {

        gMax = gMin;

    }



    return OK;

}





//

//   s c a n R a n g e

//



tRetCode tScanner::scanRange( double& gMin, double& gMid, double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanRange( gMin, gMax ) ) != OK )

        return nRet;



    if( curToken() == xTokRange ) {

        if( ( nRet = readToken() ) != OK ||

            ( nRet = scanDouble( gMid ) ) != OK ) {

            return nRet;

        }



        if( gMid < gMin ) {

            double t = gMin; gMin = gMid; gMid = t;

        }

        else

            if( gMid > gMax ) {

                double t = gMid; gMid = gMax; gMax = t;

            }

    }

    else {

        gMid = sqrt( gMin * gMax );

    }



    return OK;

}





//

//   s c a n P o s R a n g e

//



tRetCode tScanner::scanPosRange( double& gMin, double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanRange( gMin, gMax ) ) != OK )

        return nRet;



    if( gMin <= 0 )

        return setError( OUT_OF_RANGE );



    return OK;

}





//

//   s c a n P o s R a n g e

//



tRetCode tScanner::scanPosRange( double& gMin, double& gMid, double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanRange( gMin, gMid, gMax ) ) != OK )

        return nRet;



    if( gMin <= 0 )

        return setError( OUT_OF_RANGE );



    return OK;

}





//

//   s c a n N o n N e g R a n g e

//



tRetCode tScanner::scanNonNegRange( double& gMin, double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanRange( gMin, gMax ) ) != OK )

        return nRet;



    if( gMin < 0 )

        return setError( OUT_OF_RANGE );



    return OK;

}





//

//   s c a n N o n N e g R a n g e

//



tRetCode tScanner::scanNonNegRange( double& gMin, double& gMid,

    double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanRange( gMin, gMid, gMax ) ) != OK )

        return nRet;



    if( gMin < 0 )

        return setError( OUT_OF_RANGE );



    return OK;

}





//

//   s c a n P e r c e n t a g e

//



tRetCode tScanner::scanPercentage( double& gValue )



{

    tRetCode nRet;



    if( ( nRet = scanDouble( gValue ) ) != OK )

        return nRet;

   

    if( curToken() == xTokPercentage ) {

        gValue /= 100.0;

        if( ( nRet = readToken() ) != OK )

            return nRet;

    }



    return OK;

}





//

//   s c a n P o s P e r c e n t a g e

//



tRetCode tScanner::scanPosPercentage( double& gValue )



{

    tRetCode nRet;



    if( ( nRet = scanPercentage( gValue ) ) != OK )

        return nRet;



    if( gValue <= 0 )

        return setError( "invalid percentage" );



    return OK;

}





//

//   s c a n N o n N e g P e r c e n t a g e

//



tRetCode tScanner::scanNonNegPercentage( double& gValue )



{

    tRetCode nRet;



    if( ( nRet = scanPercentage( gValue ) ) != OK )

        return nRet;



    if( gValue < 0 )

        return setError( "invalid percentage" );



    return OK;

}





//

//   s c a n P e r c e n t a g e R a n g e

//



tRetCode tScanner::scanPercentageRange( double& gMin, double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanPercentage( gMin ) ) != OK )

        return nRet;



    if( curToken() == xTokRange ) {

        if( ( nRet = readToken() ) != OK ||

            ( nRet = scanPercentage( gMax ) ) != OK ) {

            return nRet;

        }



        if( gMax < gMin ) {

            double t = gMin; gMin = gMax; gMax = t;

        }

    }

    else {

        gMax = gMin;

    }



    return OK;

}





//

//   s c a n P e r c e n t a g e R a n g e

//



tRetCode tScanner::scanPercentageRange( double& gMin, double& gMid,

    double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanPercentageRange( gMin, gMax ) ) != OK )

        return nRet;



    if( curToken() == xTokRange ) {

        if( ( nRet = readToken() ) != OK ||

            ( nRet = scanPercentage( gMid ) ) != OK ) {

            return nRet;

        }



        if( gMid < gMin ) {

            double t = gMin; gMin = gMid; gMid = t;

        }

        else

            if( gMid > gMax ) {

                double t = gMid; gMid = gMax; gMax = t;

            }

    }

    else {

        gMid = sqrt( gMin * gMax );

    }



    return OK;

}





//

//   s c a n P o s P e r c e n t a g e R a n g e

//



tRetCode tScanner::scanPosPercentageRange( double& gMin, double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanPercentageRange( gMin, gMax ) ) != OK )

        return nRet;



    if( gMin <= 0 )

        return setError( OUT_OF_RANGE );



    return OK;

}





//

//   s c a n P o s P e r c e n t a g e R a n g e

//



tRetCode tScanner::scanPosPercentageRange( double& gMin, double& gMid,

    double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanPercentageRange( gMin, gMid, gMax ) ) != OK )

        return nRet;



    if( gMin <= 0 )

        return setError( OUT_OF_RANGE );



    return OK;

}





//

//   s c a n N o n N e g P e r c e n t a g e R a n g e

//



tRetCode tScanner::scanNonNegPercentageRange( double& gMin, double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanPercentageRange( gMin, gMax ) ) != OK )

        return nRet;



    if( gMin < 0 )

        return setError( OUT_OF_RANGE );



    return OK;

}





//

//   s c a n N o n N e g P e r c e n t a g e R a n g e

//



tRetCode tScanner::scanNonNegPercentageRange( double& gMin, double& gMid,

    double& gMax )



{

    tRetCode nRet;



    if( ( nRet = scanPercentageRange( gMin, gMid, gMax ) ) != OK )

        return nRet;



    if( gMin < 0 )

        return setError( OUT_OF_RANGE );



    return OK;

}





//

//   s c a n S t r i n g

//



tRetCode tScanner::scanString( char*& sString )



{

    tRetCode nRet;



    if( curToken() == xTokEOF )

        return setError( END_OF_FILE );



    if( curToken() != xTokString )

        return setError( "string expected" );



    char* s = StrCopy( curText() );



    if( ( nRet = readToken() ) != OK ) {

        delete s;

        return nRet;

    }

    sString = s;

    return OK;

}





//

//   s c a n D a t e

//



tRetCode tScanner::scanDate( tDate& Value )



{

    if( ! tryDate( Value ) )

        return setError( INVALID_DATE );

    return readToken();

}



MTG_END_NAMESPACE



