// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgSlotLayer.h"

#include "MtgClaim.h"



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tSlotLayer::init()



{

    m_pSignature = 0;

    reset();

}





//

//   s o r t S e q

//



void tSlotLayer::sortSeq()



{

    for( int i = 1; i < m_Seq.numOfElems(); ++i ) {

        int j = i;

        tWrapper* p = m_Seq[i].m_pByMaturity;



        while( j > 0 &&

               m_Seq[j - 1].m_pByMaturity->m_pClaim->maturity() <

                   m_Seq[j].m_pByMaturity->m_pClaim->maturity() ) {

            m_Seq[j].m_pByMaturity = m_Seq[j - 1].m_pByMaturity;

            --j;

        }

        m_Seq[j].m_pByMaturity = p;



        j = i;

        p = m_Seq[i].m_pByBirth;



        while( j > 0 &&

               m_Seq[j - 1].m_pByBirth->m_pClaim->birth() <

                   m_Seq[j].m_pByBirth->m_pClaim->birth() ) {

            m_Seq[j].m_pByBirth = m_Seq[j - 1].m_pByBirth;

            --j;

        }

        m_Seq[j].m_pByBirth = p;

    }

}





//

//   t S l o t L a y e r

//



tSlotLayer::tSlotLayer()



{

    init();

}





//

//   t S l o t L a y e r

//



tSlotLayer::tSlotLayer( const tSlotLayer& Layer )



{

    init();

    copyFrom( Layer );

}





//

//   ~ t S l o t L a y e r

//



tSlotLayer::~tSlotLayer()



{

    reset();

}





//

//   r e s e t

//



void tSlotLayer::reset()



{

    m_Seq.reset();

    m_Slot.reset();

    m_XlatSlot.reset();

    m_XlatIndex.reset();

    m_Buffer[0].reset();

    m_Buffer[1].reset();

    m_Prep.reset();

    m_Temp1.reset();

    m_Temp2.reset();

    m_Chain.reset();

    m_Bounds.reset();



    m_pWrapper = 0;

  

    if( m_pSignature != 0 ) {

        delete m_pSignature;

        m_pSignature = 0;

    }



    m_bWithPrep = false;

    m_bWithTemp1 = false;

    m_bWithTemp2 = false;

    m_bWithChain = false;

    m_bIsFinalized = false;



    setProfile();

}





//

//   o p e r a t o r =

//



tSlotLayer& tSlotLayer::operator=( const tSlotLayer& Layer )



{

    if( &Layer != this )

        copyFrom( Layer );

    return *this;

}





//

//   c o p y F r o m

//



void tSlotLayer::copyFrom( const tSlotLayer& Layer )



{

    if( &Layer == this )

        return ;



    reset();



    m_Seq = Layer.m_Seq;

    m_Slot = Layer.m_Slot;

    m_XlatSlot = Layer.m_XlatSlot;

    m_XlatIndex = Layer.m_XlatIndex;

    m_Buffer[0] = Layer.m_Buffer[0];

    m_Buffer[1] = Layer.m_Buffer[1];

    m_Prep = Layer.m_Prep;

    m_Temp1 = Layer.m_Temp1;

    m_Temp2 = Layer.m_Temp2;

    m_Chain = Layer.m_Chain;

    m_bWithPrep = Layer.m_bWithPrep;

    m_bWithTemp1 = Layer.m_bWithTemp1;

    m_bWithTemp2 = Layer.m_bWithTemp2;

    m_bWithChain = Layer.m_bWithChain;

    m_Bounds = Layer.m_Bounds;

    m_pWrapper = Layer.m_pWrapper;

    m_bIsFinalized = Layer.m_bIsFinalized;

    m_nFirstActiveSlot = Layer.m_nFirstActiveSlot;

    m_nLastActiveSlot = Layer.m_nLastActiveSlot;

    m_nNextByMaturity = Layer.m_nNextByMaturity;

    m_nNextByBirth = Layer.m_nNextByBirth;

    m_nClaimPos = Layer.m_nClaimPos;

    m_nTotalPos = Layer.m_nTotalPos;

    m_nCurrent = Layer.m_nCurrent;

    m_nLast = Layer.m_nLast;



    if( Layer.m_pSignature != 0 )

        m_pSignature = Layer.m_pSignature->clone();



    tWithProfile::copyFrom( Layer );

}





//

//   s e t B o u n d s

//



void tSlotLayer::setBounds( const tArrayBounds& Bounds )



{

    m_Bounds = Bounds;

    m_bIsFinalized = false;

}





//

//   s e t B o u n d s

//



void tSlotLayer::setBounds( const tArrayBounds* pBounds )



{

    MTG_ASSERT( pBounds != 0 );

    setBounds( *pBounds );

}





//

//   s e t N u m O f L e v e l s

//



void tSlotLayer::setNumOfLevels( int nNumOfUpLevels, int nNumOfDownLevels )

              

{

    m_Bounds.reset();

    m_Bounds.append( -nNumOfDownLevels, nNumOfUpLevels );

    m_bIsFinalized = false;

}





//

//   s e t P o r t f o l i o

//



void tSlotLayer::setPortfolio( tPortfolio& Pf )



{

    MTG_ASSERT( Pf.isFinalized() );

    setPortfolio( Pf.m_Sorted );

}





//

//   s e t P o r t f o l i o

//



void tSlotLayer::setPortfolio( tPortfolio* pPf )



{

    MTG_ASSERT( pPf != 0 );

    setPortfolio( *pPf );

}





//

//   s e t P o r t f o l i o

//



void tSlotLayer::setPortfolio( tHeap<tWrapper*>& Wrapper )



{

    setPortfolio( &Wrapper );

}





//

//   s e t P o r t f o l i o

//



void tSlotLayer::setPortfolio( tHeap<tWrapper*>* pWrapper )



{

    MTG_ASSERT( pWrapper != 0 );

    m_pWrapper = pWrapper;

    m_bIsFinalized = false;

}





//

//   s e t S i g n a t u r e

//



void tSlotLayer::setSignature( const tSignature& Sig )



{

    setSignature( &Sig );

}





//

//   s e t S i g n a t u r e

//



void tSlotLayer::setSignature( const tSignature* pSig )



{

    if( m_pSignature != 0 )

        delete m_pSignature;

    if( pSig == 0 )

        m_pSignature = 0;

    else

        m_pSignature = pSig->clone();

    m_bIsFinalized = false;

}





//

//   s e t W i t h P r e p

//



void tSlotLayer::setWithPrep( bool bWithPrep )



{

    m_bWithPrep = bWithPrep;

    if( m_bWithPrep && m_bIsFinalized )

        m_Prep.setBounds( m_Bounds );

}





//

//   s e t W i t h T e m p 1

//



void tSlotLayer::setWithTemp1( bool bWithTemp1 )



{

    m_bWithTemp1 = bWithTemp1;

    if( m_bWithTemp1 && m_bIsFinalized )

        m_Temp1.setBounds( m_Bounds );

}





//

//   s e t W i t h T e m p 2

//



void tSlotLayer::setWithTemp2( bool bWithTemp2 )



{

    m_bWithTemp2 = bWithTemp2;

    if( m_bWithTemp2 && m_bIsFinalized )

        m_Temp2.setBounds( m_Bounds );

}





//

//   s e t W i t h C h a i n

//



void tSlotLayer::setWithChain( bool bWithChain )



{

    m_bWithChain = bWithChain;

    if( m_bWithChain && m_bIsFinalized ) {

        tArrayBounds B;

        

        for( int i = 0; i < m_Bounds.dimension() - 1; ++i )

            B.append( m_Bounds.from( i ), m_Bounds.to( i ) );

        m_Chain.setBounds( B );

    }

}





//

//   f i n a l i z e

//



tRetCode tSlotLayer::finalize()



{

    MTG_ASSERT( m_Bounds.dimension() >= 1 );

    MTG_ASSERT( m_pWrapper != 0 );



    if( m_bIsFinalized )

        return OK;



    m_Seq.reset();



    int nMaxSlot = -1;

    int nMaxIndex = -1;



        // Note that the order of m_pWrapper doesn't matter for

        // the purpose of accessing the values. (It might matter

        // from another perspective, of course.)



    for( int i = 0; i < m_pWrapper->numOfElems(); ++i ) {

        int nSlot = (*m_pWrapper)[i]->m_nSlot;

        int nIndex = (*m_pWrapper)[i]->m_nIndex;



        if( m_pSignature == 0 || (*m_pSignature)[nIndex] ) {

            ++m_Seq;

            m_Seq.last().m_pByMaturity = (*m_pWrapper)[i];

            m_Seq.last().m_pByBirth = (*m_pWrapper)[i];

            if( nSlot > nMaxSlot )

                nMaxSlot = nSlot;

            if( nIndex > nMaxIndex )

                nMaxIndex = nIndex;

        }

    }

    sortSeq();



    MTG_ASSERT( nMaxSlot < m_pWrapper->numOfElems() );

    MTG_ASSERT( nMaxIndex < m_pWrapper->numOfElems() );



    if( m_pSignature == 0 ) {

        m_pSignature = new tSignature( nMaxIndex + 1 );

        for( int i = 0; i <= nMaxIndex; ++i )

            m_pSignature->on( i );

    }



    m_XlatSlot.numOfElems( nMaxSlot + 1 );

    for( MTG_FOR_INIT( int ) i = 0; i < m_XlatSlot.numOfElems(); ++i )

        m_XlatSlot[i] = -1;



    m_XlatIndex.numOfElems( nMaxIndex + 1 );

    for( MTG_FOR_INIT( int ) i = 0; i < m_XlatIndex.numOfElems(); ++i )

        m_XlatIndex[i] = -1;



    int nNumOfSlots = 0;

    for( MTG_FOR_INIT( int ) i = 0; i < m_Seq.numOfElems(); ++i ) {

        int nSlot = m_Seq[i].m_pByMaturity->m_nSlot;

        int nIndex = m_Seq[i].m_pByMaturity->m_nIndex;



        if( m_XlatSlot[nSlot] < 0 )

            m_XlatSlot[nSlot] = nNumOfSlots++;

        m_XlatIndex[nIndex] = m_XlatSlot[nSlot];

    }



    m_Slot.numOfElems( nNumOfSlots );



    if( m_bWithChain )

        m_Chain.setBounds( m_Bounds );



        // One extra slot for the total:

    m_Bounds.append( 0, nNumOfSlots );

    m_Buffer[0].setBounds( m_Bounds );

    m_Buffer[1].setBounds( m_Bounds );



    if( m_bWithPrep )

        m_Prep.setBounds( m_Bounds );

    if( m_bWithTemp1 )

        m_Temp1.setBounds( m_Bounds );

    if( m_bWithTemp2 )

        m_Temp2.setBounds( m_Bounds );



    m_nClaimPos = nNumOfSlots - 1;

    m_nTotalPos = nNumOfSlots;

    m_nCurrent = 0;

    m_nLast = 1;



    if( hasProfile() ) {

        try {

            setProfileId( profile().putSlotLayer( m_Bounds, *m_pSignature ) );

        }

        catch( tException e ) {

            return e.ret();

        }

    }



    m_bIsFinalized = true;



    return OK;

}





//

//   h a s M o n i t o r e d C l a i m s

//



bool tSlotLayer::hasMonitoredClaims() const



{

    for( int i = 0; i < m_Seq.numOfElems(); ++i ) {

        if( m_Seq[i].m_pByMaturity->m_pClaim->isMonitored() )

            return true;

    }

    return false;

}





//

//   p r e p a r e R o l l b a c k

//



void tSlotLayer::prepareRollback()



{

    MTG_ASSERT( m_bIsFinalized );



    m_nNextByMaturity = 0;

    m_nNextByBirth = 0;



    m_nFirstActiveSlot = m_Slot.numOfElems();

    m_nLastActiveSlot = -1;



    for( int i = 0; i < m_Slot.numOfElems(); ++i )

        m_Slot[i].m_bIsActive = false;

}





//

//   b e f o r e R o l l b a c k

//



void tSlotLayer::beforeRollback( int nDay )



{

    MTG_ASSERT( m_bIsFinalized );



    while( m_nNextByMaturity < m_Seq.numOfElems() ) {

        tWrapper* pWrapper = m_Seq[m_nNextByMaturity].m_pByMaturity;



        if( pWrapper->m_pClaim->maturity() == nDay ) {

            int nSlot = m_XlatSlot[pWrapper->m_nSlot];

            tSlot& S = m_Slot[nSlot];



            MTG_ASSERT( ! S.m_bIsActive );



            S.m_pWrapper = pWrapper;

            S.m_bIsActive = true;



            if( m_nFirstActiveSlot > nSlot )

                m_nFirstActiveSlot = nSlot;

            if( m_nLastActiveSlot < nSlot )

                m_nLastActiveSlot = nSlot;



            ++m_nNextByMaturity;

        }

        else {

            break;

        }

    }

}





//

//   a f t e r R o l l b a c k

//



void tSlotLayer::afterRollback( int nDay )



{

    MTG_ASSERT( m_bIsFinalized );



    while( m_nNextByBirth < m_nNextByMaturity ) {

        tWrapper* pWrapper = m_Seq[m_nNextByBirth].m_pByBirth;



        if( pWrapper->m_pClaim->birth() == nDay ) {

            int nSlot = m_XlatSlot[pWrapper->m_nSlot];



            MTG_ASSERT( m_Slot[nSlot].m_bIsActive );

            m_Slot[nSlot].m_bIsActive = false;



            if( m_nFirstActiveSlot == nSlot ) {

                do{

                    ++m_nFirstActiveSlot;

                } while( m_nFirstActiveSlot < m_Slot.numOfElems() &&

                            ! m_Slot[m_nFirstActiveSlot].m_bIsActive );

            }



            if( m_nLastActiveSlot == nSlot ) {

                do{

                    --m_nLastActiveSlot;

                } while( m_nLastActiveSlot >= 0 &&

                            ! m_Slot[m_nLastActiveSlot].m_bIsActive );

            }



            ++m_nNextByBirth;

        }

        else {

            break;

        }

    }

}





//

//   r o t a t e

//



void tSlotLayer::rotate()



{

    MTG_ASSERT( m_bIsFinalized );

    m_nCurrent = m_nLast;

    m_nLast = 1 - m_nCurrent;

}





//

//   z e r o

//



void tSlotLayer::zero()



{

    MTG_ASSERT( m_bIsFinalized );



    double g = 0;



    m_Buffer[0].fill( g );

    m_Buffer[1].fill( g );

    m_Prep.fill( g ); 



        // m_Prep, m_Temp1, m_Temp2 are not zeroed.

}





//

//   s e l f C h a i n

//



void tSlotLayer::selfChain()



{

    MTG_ASSERT( m_bIsFinalized );



    if( m_bWithChain )

        m_Chain.fill( this );

}





//

//   c u r C l e a r T o t a l

//



void tSlotLayer::curClearTotal()



{

    MTG_ASSERT( m_bIsFinalized );



    double g = 0;

    current().fillRow( m_nTotalPos, g );

}





//

//   p r e p C l e a r T o t a l

//



void tSlotLayer::prepClearTotal()



{

    MTG_ASSERT( m_bIsFinalized );



    double g = 0;

    m_Prep.fillRow( m_nTotalPos, g );

}





//

//   c u r C o p y F r o m

//



void tSlotLayer::curCopyFrom( const tSlotLayer& Layer, int x1 )



{

    memcpy( current().row( x1 ),

        Layer.m_Buffer[Layer.m_nCurrent].row( x1 ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   c u r C o p y F r o m

//



void tSlotLayer::curCopyFrom( const tSlotLayer& Layer, int x1, int x2 )



{

    memcpy( current().row( x1, x2 ),

        Layer.m_Buffer[Layer.m_nCurrent].row( x1, x2 ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   c u r C o p y F r o m

//



void tSlotLayer::curCopyFrom( const tSlotLayer& Layer, int x1, int x2, int x3 )



{

    memcpy( current().row( x1, x2, x3 ),

        Layer.m_Buffer[Layer.m_nCurrent].row( x1, x2, x3 ), 

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   c u r C o p y F r o m

//



void tSlotLayer::curCopyFrom( const tSlotLayer& Layer, tHeap<int>& x )



{

    memcpy( current().row( x ),

        Layer.m_Buffer[Layer.m_nCurrent].row( x ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   c u r C o p y F r o m

//



void tSlotLayer::curCopyFrom( const tSlotLayer& Layer, int x[] )



{

    memcpy( current().row( x ),

        Layer.m_Buffer[Layer.m_nCurrent].row( x ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   c o p y C u r T o P r e p

//



void tSlotLayer::copyCurToPrep()



{

    m_Prep = current();

}





//

//   s a v e P r e p

//



void tSlotLayer::savePrep()



{

    m_Temp1 = m_Prep;

}





//

//   r e s t o r e P r e p

//



void tSlotLayer::restorePrep()



{

    m_Prep = m_Temp1;

}





//

//   s a v e P r e p

//



void tSlotLayer::savePrep( int x1 )



{

    memcpy( m_Temp1.row( x1 ), m_Prep.row( x1 ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   s a v e P r e p

//



void tSlotLayer::savePrep( int x1, int x2 )



{

    memcpy( m_Temp1.row( x1, x2 ), m_Prep.row( x1, x2 ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   s a v e P r e p

//



void tSlotLayer::savePrep( int x1, int x2, int x3 )



{

    memcpy( m_Temp1.row( x1, x2, x3 ), m_Prep.row( x1, x2, x3 ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   s a v e P r e p

//



void tSlotLayer::savePrep( tHeap<int>& x )



{

    memcpy( m_Temp1.row( x ), m_Prep.row( x ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   s a v e P r e p

//



void tSlotLayer::savePrep( int x[] )



{

    memcpy( m_Temp1.row( x ), m_Prep.row( x ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   r e s t o r e P r e p

//



void tSlotLayer::restorePrep( int x1 )



{

    memcpy( m_Prep.row( x1 ), m_Temp1.row( x1 ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   r e s t o r e P r e p

//



void tSlotLayer::restorePrep( int x1, int x2 )



{

    memcpy( m_Prep.row( x1, x2 ), m_Temp1.row( x1, x2 ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   r e s t o r e P r e p

//



void tSlotLayer::restorePrep( int x1, int x2, int x3 )



{

    memcpy( m_Prep.row( x1, x2, x3 ), m_Temp1.row( x1, x2, x3 ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   r e s t o r e P r e p

//



void tSlotLayer::restorePrep( tHeap<int>& x )



{

    memcpy( m_Prep.row( x ), m_Temp1.row( x ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   r e s t o r e P r e p

//



void tSlotLayer::restorePrep( int x[] )



{

    memcpy( m_Prep.row( x ), m_Temp1.row( x ),

        sizeof(double) * m_Bounds.rowSize() );

}





//

//   l a s t C o p y F r o m

//



void tSlotLayer::lastCopyFrom( const tSlotLayer& Layer )



{

#if defined(_DEBUG)

    MTG_ASSERT( Layer.m_Bounds.dimension() == m_Bounds.dimension() );

    for( int j = 0; j < m_Bounds.dimension(); ++j ) {

        MTG_ASSERT( Layer.m_Bounds.from( j ) == m_Bounds.from( j ) &&

                    Layer.m_Bounds.to( j ) == m_Bounds.to( j ) );

    }

#endif



    m_Buffer[m_nLast] = Layer.m_Buffer[Layer.m_nLast];

}



MTG_END_NAMESPACE

