// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#if ! defined(_MTG_SOURCE_)

#define _MTG_SOURCE_



#include "MtgToken.h"

#include "MtgDate.h"



MTG_BEGIN_NAMESPACE



// This file defines the interface for the tiny scanner that is used

// to parse statements and expressions for generalized basic instruments.

// It's really just a simple Lex-substitute. Possible sources are

// text files and strings. The scanner returns floating point numbers,

// identifiers, predefined identifiers (if, then, else, ...) and

// special tokens (+, -, *, =, +=, ...) as well as calendar dates.

//

// It is not meant for export. The scanner class is made a friend class;

// there's no public interface (and thus no need to elaborate the

// existing interface).



// The length of a token is limited. A token can be longer, but

// it is only stored up to the following number of characters:



const int nSourceMaxTokenLength = 255;



#if ! defined(_MAX_PATH)

  #define _MAX_PATH 128

#endif 





//

//   t S o u r c e

//



class tSource {



        // If the source is safe, certain tokens are disallowed.

        // Among them are xTokShell, xTokTcl, xTokRun, xTokInclude

        // and xTokSave.



    bool m_bIsSafe;



        // The current state of the scanner is: a) the current source line,

        // b) the current token type, and c) the textual representation of

        // the current token as found in the source. Even for predefined

        // tokens and for numbers, m_sCurToken has always the textual

        // representation.



    int m_nCurLine;

    tToken m_nCurToken;

    char m_sCurText[nSourceMaxTokenLength+1];



        // For numbers (token type == xNumber), the numerical value is

        // stored here (automatically).



    double m_gCurValue;



        // Calendar dates (token type == xDate) are stored here:



    tDate m_CurDate;



        // The scanner has two lookahead characters. The character that is

        // to be examined next (and thus the character immediately following

        // the current token) is stored in m_nCurChar (somewhat confusingly,

        // I admit). The following character is stored in m_nNextChar. Both

        // of them can be EOF, of course. Two lookaheads avoid pushbacks.

    

        // The parser only accesses the state variables described above.

        // The following character variables represent a lower level

        // and should not be read from the outside.



    int m_nCurChar;

    int m_nNextChar;

    bool m_bLookAhead;



        // Some helper functions.



    tRetCode readChar();

    tRetCode nextChar( int& nNextChar );



    tRetCode skipSpace();

    tRetCode skipComment();



    tRetCode readId();

    tRetCode readNumber();

    tRetCode readString();

    tRetCode readSpecial();



        // These functions start and finish the scanning process from

        // the current source. They are called by the parser.



    tRetCode start();

    void end();



        // This function loads the state variables with the next token.

        // It is called by the parser. (So the parser basically calls

        // start(), then many times readToken(), then end().)



    tRetCode readToken();



        // Tries to convert the current string token into a 

        // calendar date.



    bool tryDate();

    bool tryDate( tDate& Date );



protected:



        // For error output:



    virtual const char* name() const = 0;



        // The following functions have to be overridden in order

        // to provide a working token source. Only readChar() is pure,

        // as the default behavior of open() and close() is to do

        // nothing.



    virtual tRetCode open();

    virtual tRetCode readChar( int& nNextChar ) = 0;

    virtual void close();



public:



    tSource();

    virtual ~tSource();



    void makeSafe() {

        m_bIsSafe = true;

    }



    bool isSafe() const {

        return m_bIsSafe;

    }



    int getCurLine() const {

        return m_nCurLine;

    }

    

    friend class tScanner;

};





//

//   t S t r i n g S o u r c e

//



class tStringSource : public tSource {



        // The string and the current position. The string is not copied.



    const char *m_sInput;

    int m_nCurPos;



    const char* name() const {

        return "<string>";

    }



    tRetCode open();

    tRetCode readChar( int& nNextChar );



public:



        // setInput() registers the string. Format: zero-terminated ASCII.



    tStringSource();

    tStringSource( const char *sInput );



    void setInput( const char *sInput );

};





//

//   t F i l e S o u r c e

//



class tFileSource : public tSource {



        // One can register the file name (then the file is opened and

        // closed after parsing) or directly a file pointer (then the

        // file is just read, but not closed).



    char m_sFileName[_MAX_PATH];

    FILE* m_pFile;



        // Remember how the file was registered:



    bool m_bByName;



    const char* name() const;



    tRetCode open();

    tRetCode readChar( int& nNextChar );

    void close();



public:



        // Register either by file name or by file pointer (of an open

        // text file). Note that the current source is not closed if

        // a new source is registered with setInput().



    tFileSource();

    tFileSource( const char *sFileName );

    tFileSource( FILE* pFile );



    ~tFileSource();



    void setInput( const char *sFileName );

    void setInput( FILE* pFile );

};



MTG_END_NAMESPACE



#endif

