// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgSummary.h"



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tSummary::init()



{

}





//

//   u p d a t e L o c a t i o n

//



tRetCode tSummary::updateLocation( int nId, tItem* pItem, tLevelItem*& pLevel,

    tTaskItem*& pTask )



{

    tRetCode nRet;



    switch( pItem->m_cType ) {

        case 'P' :

            if( pLevel != 0 )

                delete pLevel;

            pLevel = dynamic_cast<tLevelItem*>( pItem );

            MTG_ASSERT( pLevel != 0 );

            break;



        case 'T' :

            delete pTask;

            pTask = dynamic_cast<tTaskItem*>( pItem );

            MTG_ASSERT( pTask != 0 );



            if( nId >= 0 && pTask->m_nLayer != nId ) {

                delete pTask;

                if( ( nRet = getTaskItem( nId, pTask ) ) != OK ) {

                    pTask = 0;

                    return nRet;

                }

            }            

            if( pLevel != 0 ) {

                delete pLevel;

                pLevel = 0;

            }

            break;



        default :

            delete pItem;

            break;

    }



    return OK;

}





//

//   g e t A l t e r n a t i v e I t e m

//



tRetCode tSummary::getAlternativeItem( tAlternativeItem*& pAlternative )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tItem* pItem;



    if( ( nRet = get( pItem, 'A' ) ) == OK ) {

        pAlternative = dynamic_cast<tAlternativeItem*>( pItem );

        MTG_ASSERT( pAlternative != 0 );

        return OK;

    }



    return nRet;

}





//

//   g e t L a t t i c e I t e m

//



tRetCode tSummary::getLatticeItem( tLatticeItem*& pLattice )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tItem* pItem;



    if( ( nRet = get( pItem, 'L' ) ) == OK ) {

        pLattice = dynamic_cast<tLatticeItem*>( pItem );

        MTG_ASSERT( pLattice != 0 );

        return OK;

    }



    return nRet;

}





//

//   g e t S l o t L a y e r I t e m

//



tRetCode tSummary::getSlotLayerItem( tSlotLayerItem*& pLayer )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tItem* pItem;



    if( ( nRet = get( pItem, 'S' ) ) == OK ) {

        pLayer = dynamic_cast<tSlotLayerItem*>( pItem );

        MTG_ASSERT( pLayer != 0 );

        return OK;

    }



    return nRet;

}





//

//   g e t S l o t L a y e r I t e m

//



tRetCode tSummary::getSlotLayerItem( int nId, tSlotLayerItem*& pLayer )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tItem* pItem;



    if( nId < 0 )

        return getSlotLayerItem( pLayer );



    while( ( nRet = get( pItem, 'S' ) ) == OK ) {

        tSlotLayerItem* p = dynamic_cast<tSlotLayerItem*>( pItem );

        MTG_ASSERT( p != 0 );

        if( p->m_nId == nId ) {

            pLayer = p;

            return OK;

        }

        delete pItem;

    }



    return nRet;

}





//

//   g e t T a s k I t e m

//



tRetCode tSummary::getTaskItem( tTaskItem*& pTask )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tItem* pItem;



    if( ( nRet = get( pItem, 'T' ) ) == OK ) {

        pTask = dynamic_cast<tTaskItem*>( pItem );

        MTG_ASSERT( pTask != 0 );

        return OK;

    }



    return nRet;

}





//

//   g e t T a s k I t e m

//



tRetCode tSummary::getTaskItem( int nId, tTaskItem*& pTask )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tItem* pItem;



    if( nId < 0 )

        return getTaskItem( pTask );



    while( ( nRet = get( pItem, 'T' ) ) == OK ) {

        tTaskItem* p = dynamic_cast<tTaskItem*>( pItem );

        MTG_ASSERT( p != 0 );

        if( p->m_nLayer == nId ) {

            pTask = p;

            return OK;

        }

        delete pItem;

    }



    return nRet;

}





//

//   g e t B o u n d s

//



tRetCode tSummary::getBounds( tArrayBounds& Bounds )



{

    tRetCode nRet;

    tLatticeItem* pLattice;



    if( ( nRet = getLatticeItem( pLattice ) ) != OK )

        return nRet;



    Bounds.reset();

    Bounds.append( 0, pLattice->m_nNumOfSlices - 1 );

    Bounds.append( pLattice->m_Bounds );



    delete pLattice;

    return OK;

}





//

//   g e t S l o t L a y e r s

//



tRetCode tSummary::getSlotLayers( tHeap<tSlotLayerItem*>& Layer )



{

    MTG_ASSERT( isOpen() );

    MTG_ASSERT( Layer.numOfElems() == 0 );



    tRetCode nRet;

    tItem* pItem;



    rewind();



    while( ( nRet = get( pItem, 'S' ) ) == OK ) {

        tSlotLayerItem* p = dynamic_cast<tSlotLayerItem*>( pItem );

        MTG_ASSERT( p != 0 );

        Layer.append( p );

    }



    if( nRet != END_OF_FILE ) {

        for( int i = 0; i < Layer.numOfElems(); ++i ) 

            delete Layer[i];

        Layer.reset();

        return nRet;

    }



    return OK;

}





//

//   g e t A l t O v e r a l l

//



tRetCode tSummary::getAltOverall( int nId, tMultiArray<int>& Aggregate,

    int tAlternativeItem::* pField, int nFill )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tArrayBounds Bounds;



    tTaskItem* pTask = 0;

    tLevelItem* pLevel = 0;

    tAlternativeItem* pAlternative;



    tItem* pItem;



    rewind();



    if( ( nRet = getBounds( Bounds ) ) != OK ||

        ( nRet = getTaskItem( nId, pTask ) ) != OK ) {

        goto end;

    }



    Aggregate.setBounds( Bounds );

    Aggregate.fill( nFill );



    while( ( nRet = get( pItem, "APT" ) ) == OK ) {

        if( pItem->m_cType == 'A' ) {

            if( pLevel == 0 ) {

                delete pItem;

                nRet = FORMAT_ERROR;

                goto end;

            }

            pAlternative = dynamic_cast<tAlternativeItem*>( pItem );

            MTG_ASSERT( pAlternative != 0 );

            if( pAlternative->*pField == -2 ) {

                Aggregate( pTask->m_nSlice, pLevel->m_Level ) = nId;

            }

            else {

                Aggregate( pTask->m_nSlice, pLevel->m_Level ) =

                    pAlternative->*pField;

            }

            delete pAlternative;

        }

        else {

            if( ( nRet = updateLocation( nId, pItem, pLevel, pTask ) ) != OK )

                goto end;

        }

    }



end:



    if( pTask != 0 )  delete pTask;

    if( pLevel != 0 ) delete pLevel;



    return ( nRet == END_OF_FILE ) ? OK : nRet;

}





//

//   g e t C o m p l e x i t y

//



tRetCode tSummary::getComplexity( int nId, tMultiArray<int>& Complexity )



{

    return getAltOverall( nId, Complexity, &tAlternativeItem::m_nComplexity, -1 );

}





//

//   g e t M a x C o m p l e x i t y

//



tRetCode tSummary::getMaxComplexity( int& nMaxComplexity )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tAlternativeItem* pAlternative;



    rewind();

    nMaxComplexity = 1;



    while( ( nRet = getAlternativeItem( pAlternative ) ) == OK ) {

        if( pAlternative->m_nComplexity > nMaxComplexity )

            nMaxComplexity = pAlternative->m_nComplexity;

        delete pAlternative;

    }



    return ( nRet == END_OF_FILE ) ? OK : nRet;

}





//

//   g e t D e p e n d e n c y

//



tRetCode tSummary::getDependency( int nId, tMultiArray<int>& Dependency )



{

    return getAltOverall( nId, Dependency, &tAlternativeItem::m_nLayer, -1 );

}





//

//   g e t B o u n d a r y

//



tRetCode tSummary::getBoundary( int nId, tMultiArray<int>& Boundary )



{

    tRetCode nRet;



    nRet = getAltOverall( nId, Boundary, &tAlternativeItem::m_nLayer, -1 );

    if( nRet != OK )

        return nRet;



    class tI : public tMultiArray<int>::tIterator {

    public:

        tHeap<int> m_Pos;

        int m_nId;

      

        virtual ~tI() {}

     

        void f( tMultiArray<int>& Parent, const tHeap<int>& Pos, bool bInner ) {

            int nId1 = Parent( Pos );



            if( nId1 == m_nId || nId1 < 0 )

                return;



            m_Pos = Pos;

            if( m_Pos.numOfElems() == 2 ) {

                    // In 2 dimensions, also check for diagonal neighbors:



                int nFromI = Pos[0] > Parent.from( 0 ) ? -1 : 0;

                int nToI = Pos[0] < Parent.to( 0 ) ? 1 : 0;

                int nFromJ = Pos[1] > Parent.from( 1 ) ? -1 : 0;

                int nToJ = Pos[1] < Parent.to( 1 ) ? 1 : 0;



                for( int i = nFromI; i <= nToI; ++i ) {

                    for( int j = nFromJ; j <= nToJ; ++j ) {

                        if( i != 0 || j != 0 ) {

                            m_Pos[0] = Pos[0] + i;

                            m_Pos[1] = Pos[1] + j;

                            if( Parent( m_Pos ) == m_nId )

                                return;

                        }

                    }

                }

            }

            else {

                for( int i = 0; i < m_Pos.numOfElems(); ++i ) {

                    if( m_Pos[i] > Parent.from( i ) ) {

                        --m_Pos[i];

                        if( Parent( m_Pos ) == m_nId )

                            return;

                        ++m_Pos[i];

                    }

                    if( m_Pos[i] < Parent.to( i ) ) {

                        ++m_Pos[i];

                        if( Parent( m_Pos ) == m_nId )

                            return;

                        --m_Pos[i];

                    }

                }

            }



            Parent( Pos ) = -1;

        }

    } I; 



    I.m_nId = nId;

    return Boundary.iterate( I );

}





//

//   g e t V a l O v e r a l l

//



tRetCode tSummary::getValOverall( int nId, tMultiArray<double>& Aggregate,

    tValueItem::tType nType, double gFill )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tArrayBounds Bounds;



    tTaskItem* pTask = 0;

    tLevelItem* pLevel = 0;

    tValueItem* pValue;



    tItem* pItem;



    rewind();



    if( ( nRet = getBounds( Bounds ) ) != OK ||

        ( nRet = getTaskItem( nId, pTask ) ) != OK ) {

        goto end;

    }



    Aggregate.setBounds( Bounds );

    Aggregate.fill( gFill );



    while( ( nRet = get( pItem, "VPT" ) ) == OK ) {

        if( pItem->m_cType == 'V' ) {

            if( pLevel == 0 ) {

                delete pItem;

                nRet = FORMAT_ERROR;

                goto end;

            }

            pValue = dynamic_cast<tValueItem*>( pItem );

            MTG_ASSERT( pValue != 0 );

            if( pValue->m_nType == nType ) {

                Aggregate( pTask->m_nSlice, pLevel->m_Level ) =

                    pValue->m_gValue;

            }

            delete pValue;

        }

        else {

            if( ( nRet = updateLocation( nId, pItem, pLevel, pTask ) ) != OK )

                goto end;

        }

    }



end:



    if( pTask != 0 )  delete pTask;

    if( pLevel != 0 ) delete pLevel;



    return ( nRet == END_OF_FILE ) ? OK : nRet;

}





//

//   g e t V o l a t i l i t y

//



tRetCode tSummary::getVolatility( int nId, tMultiArray<double>& Volatility )



{

    return getValOverall( nId, Volatility, tValueItem::xVolatility, 0 );

}





//

//   g e t T o t a l

//



tRetCode tSummary::getTotal( int nId, tMultiArray<double>& Total )



{

    return getValOverall( nId, Total, tValueItem::xTotal, 0 );

}





//

//   g e t T o t a l

//



tRetCode tSummary::getTotal( const tHeap<int>& LayerSign,

    tMultiArray<int>* pValid, tMultiArray<double>* pTotal )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tArrayBounds Bounds;

    tItem* pItem;

    tMultiArray<int> Valid;



        // Need to be careful about repeats:



    typedef unsigned char tIndicator;

    tMultiArray<tIndicator> Indicator;

    tHeap<int> Index2Ind;



    class tI : public tMultiArray<int>::tIterator {

    public:

        int m_nNeedCount;

        tMultiArray<double>* m_pTotal;



        void f( tMultiArray<int>& Parent, const tHeap<int>& Pos, bool bInner ) {

            int& V = Parent( Pos );



            V = ( V == m_nNeedCount ) ? 1 : 0;

            if( V == 0 && m_pTotal != 0 )

                (*m_pTotal)( Pos ) = 0;



        }

    } I; 



    if( pValid == 0 )

        pValid = &Valid;



    tTaskItem* pTask = 0;

    tLevelItem* pLevel = 0;



    Index2Ind.numOfElems( LayerSign.numOfElems() );



    int nNeedCount = 0;

    for( int i = 0; i < LayerSign.numOfElems(); ++i ) {

        if( LayerSign[i] != 0 ) {

            if( (size_t) nNeedCount >= 8 * sizeof(tIndicator) )

                return OUT_OF_MEMORY;

            Index2Ind[i] = nNeedCount++;

        }

        else {

            Index2Ind[i] = -1;

        }

    }



    rewind();

    if( ( nRet = getBounds( Bounds ) ) != OK )

        goto end;



    Indicator.setBounds( Bounds );

    Indicator.fill( 0 );

    pValid->setBounds( Bounds );

    pValid->fill( 0 );

    if( pTotal != 0 ) {

        pTotal->setBounds( Bounds );

        pTotal->fill( 0 );

    }



    if( nNeedCount == 0 )

        goto end;



    while( ( nRet = get( pItem, "VPT" ) ) == OK ) {

        switch( pItem->m_cType ) {

            case 'T' :

                if( pTask != 0 )

                    delete pTask;

                pTask = dynamic_cast<tTaskItem*>( pItem );

                MTG_ASSERT( pTask != 0 );



                while( pTask->m_nLayer >= LayerSign.numOfElems() ||

                       LayerSign[pTask->m_nLayer] == 0 ) {

                    delete pTask;

                    if( ( nRet = getTaskItem( pTask ) ) != OK ) {

                        pTask = 0;

                        goto end;

                    }

                }



                if( pLevel != 0 ) {

                    delete pLevel;

                    pLevel = 0;

                }

                break;



            case 'V' :

                if( pLevel == 0 ) {

                    delete pItem;

                    nRet = FORMAT_ERROR;

                    goto end;

                }

                else {

                    tValueItem* pValue = dynamic_cast<tValueItem*>( pItem );

                    MTG_ASSERT( pValue != 0 );



                    if( pValue->m_nType == tValueItem::xTotal ) {

                        tIndicator& Ind =

                            Indicator( pTask->m_nSlice, pLevel->m_Level );



                        if( ! ( Ind & (tIndicator)

                                ( 1 << Index2Ind[pTask->m_nLayer] ) ) ) {

                            Ind |= (tIndicator) ( 1 << Index2Ind[pTask->m_nLayer] );

                            ++(*pValid)( pTask->m_nSlice, pLevel->m_Level );

                            if( pTotal != 0 ) {

                                (*pTotal)( pTask->m_nSlice, pLevel->m_Level )

                                    += LayerSign[pTask->m_nLayer]

                                        * pValue->m_gValue;

                            }

                        }

                    }

                }

                delete pItem;

                break;



            case 'P' :

                if( pLevel != 0 )

                    delete pLevel;

                pLevel = dynamic_cast<tLevelItem*>( pItem );

                MTG_ASSERT( pLevel != 0 );

                break;



            default :

                throw tException( INTERNAL_ERROR );

        }

    }



    I.m_nNeedCount = nNeedCount;

    I.m_pTotal = pTotal;



    nRet = pValid->iterate( I );



end:



    if( pTask != 0 )  delete pTask;

    if( pLevel != 0 ) delete pLevel;



    return ( nRet == END_OF_FILE ) ? OK : nRet;

}





//

//   g e t E x P o l i c y

//



tRetCode tSummary::getExPolicy( int nId, tExPolicy nExPolicy,

    const tHeap<int>& ClaimSign, tMultiArray<int>* pExPolicy,

    tMultiArray<double>* pPayoff )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tArrayBounds Bounds;

    tItem* pItem;



    tTaskItem* pTask = 0;

    tLevelItem* pLevel = 0;



    int nNeedCount = 0;

    int nCurCount = 0;

    double gCurPayoff = 0;



    for( int i = 0; i < ClaimSign.numOfElems(); ++i ) {

        if( ClaimSign[i] != 0 )

            ++nNeedCount;

    }



    rewind();



    if( ( nRet = getBounds( Bounds ) ) != OK ||

        ( nRet = getTaskItem( nId, pTask ) ) != OK ) {

        goto end;

    }



    if( pExPolicy != 0 ) {

        pExPolicy->setBounds( Bounds );

        pExPolicy->fill( 0 );

    }

    if( pPayoff != 0 ) {

        pPayoff->setBounds( Bounds );

        pPayoff->fill( 0 );

    }



    if( nNeedCount == 0 )

        goto end;



    while( ( nRet = get( pItem, "EPT" ) ) == OK ) {

        if( pItem->m_cType == 'E' ) {

            if( pLevel == 0 ) {

                delete pItem;

                nRet = FORMAT_ERROR;

                goto end;

            }

            tExPolicyItem* pExPolicy = dynamic_cast<tExPolicyItem*>( pItem );

            MTG_ASSERT( pExPolicy != 0 );

            if( pExPolicy->m_nExPolicy == nExPolicy &&

                    ClaimSign.numOfElems() > pExPolicy->m_nIndex &&

                    ClaimSign[pExPolicy->m_nIndex] != 0 ) {

                gCurPayoff += pExPolicy->m_gPayoff *

                                ClaimSign[pExPolicy->m_nIndex];

                ++nCurCount;

            }

            delete pExPolicy;

        }

        else {

            if( nCurCount > 0 ) {

                if( nCurCount == nNeedCount ) {

                    if( pExPolicy != 0 ) {

                        (*pExPolicy)( pTask->m_nSlice,

                            pLevel->m_Level ) = 1;

                    }

                    if( pPayoff != 0 ) {

                        (*pPayoff)( pTask->m_nSlice,

                            pLevel->m_Level ) = gCurPayoff;

                    }

                }

                nCurCount = 0;

                gCurPayoff = 0;

            }

            if( ( nRet = updateLocation( nId, pItem, pLevel, pTask ) ) != OK )

                goto end;

        }

    }



    if( nCurCount == nNeedCount ) {

        if( pExPolicy != 0 )

            (*pExPolicy)( pTask->m_nSlice, pLevel->m_Level ) = 1;

        if( pPayoff != 0 )

            (*pPayoff)( pTask->m_nSlice, pLevel->m_Level ) = gCurPayoff;

    }



end:



    if( pTask != 0 )  delete pTask;

    if( pLevel != 0 ) delete pLevel;



    return ( nRet == END_OF_FILE ) ? OK : nRet;

}





//

//   t S u m m a r y

//



tSummary::tSummary()



{

    init();

}





//

//   t S u m m a r y

//



tSummary::tSummary( const char* sFileName )

    : super( sFileName )



{

    init();

}





//

//   ~ t S u m m a r y

//



tSummary::~tSummary()



{

}



#if defined(_MTG_WITH_MATHLINK)



//

//   s e n d L a t t i c e

//



tRetCode tSummary::sendLattice( MLINK MathLink )



{

    MTG_ASSERT( isOpen() );



    tRetCode nRet;

    tLatticeItem* pLattice;

    long Dim[1];



    rewind();

    if( ( nRet = getLatticeItem( pLattice ) ) != OK )

        return nRet;



    if( MLPutFunction( MathLink, "List",

            pLattice->m_Bounds.dimension() + 1 ) == 0 ) {

        nRet = MATHLINK_ERROR;

        goto end;

    }

    else {

        tArrayBounds& B = pLattice->m_Bounds;

        int n = pLattice->m_nNumOfSlices;



        if( MLPutFunction( MathLink, "List", 2 ) == 0 ||

            MLPutFunction( MathLink, "List", 2 ) == 0 ||

            MLPutInteger( MathLink, 0 ) == 0 ||

            MLPutInteger( MathLink, n - 1 ) == 0 ) {

            nRet = MATHLINK_ERROR;

            goto end;

        }



        Dim[0] = n;

        MLPutRealArray( MathLink, &pLattice->m_AxesConcat[0], Dim, 0, 1 );



        for( int k = 0; k < B.dimension(); ++k ) {

            if( MLPutFunction( MathLink, "List", 2 ) == 0 ||

                MLPutFunction( MathLink, "List", 2 ) == 0 ||

                MLPutInteger( MathLink, B.from( k ) ) == 0 ||

                MLPutInteger( MathLink, B.to( k ) ) == 0 ) {

                nRet = MATHLINK_ERROR;

                goto end;

            }

            Dim[0] = B.size( k );

            MLPutRealArray( MathLink, &pLattice->m_AxesConcat[n], Dim, 0, 1 );

            n += Dim[0];

        }

    }



end:



    delete pLattice;

    return nRet;

}





//

//   s e n d S l o t L a y e r s

//



tRetCode tSummary::sendSlotLayers( MLINK MathLink )



{

    tRetCode nRet;

    tHeap<tSlotLayerItem*> Layer;



    if( ( nRet = getSlotLayers( Layer ) ) != OK )

        return nRet;



    if( MLPutFunction( MathLink, "List", Layer.numOfElems() ) == 0 ) {

        nRet = MATHLINK_ERROR;

        goto end;

    }

    else {

        for( int i = 0; i < Layer.numOfElems(); ++i )  {

            const tSlotLayerItem& I = *Layer[i];



            if( MLPutFunction( MathLink, "List", 3 ) == 0 ||

                MLPutInteger( MathLink, I.m_nId ) == 0 ) {

                nRet = MATHLINK_ERROR;

                goto end;

            }

    

            if( ( nRet = I.m_Sig.toMathLink( MathLink ) ) != OK ||

                ( nRet = I.m_Bounds.toMathLink( MathLink ) ) != OK ) {

                goto end;

            }

        }

    }



end:



    for( int i = 0; i < Layer.numOfElems(); ++i ) 

        delete Layer[i];

    Layer.reset();



    return nRet;

}





//

//   s e n d C o m p l e x i t y

//



tRetCode tSummary::sendComplexity( MLINK MathLink, int nLayer )



{

    tRetCode nRet;

    tMultiArray<int> Complexity;



    if( ( nRet = getComplexity( nLayer, Complexity ) ) != OK )

        return nRet;

    if( MLPutFunction( MathLink, "Transpose", 1 ) == 0 )

        return MATHLINK_ERROR;

    if( ( nRet = Complexity.toMathLink( MathLink ) ) != OK ) 

        return nRet;



    return OK;

}





//

//   s e n d M a x C o m p l e x i t y

//



tRetCode tSummary::sendMaxComplexity( MLINK MathLink )



{

    tRetCode nRet;

    int nMaxComplexity;



    if( ( nRet = getMaxComplexity( nMaxComplexity ) ) != OK )

        return nRet;

    if( MLPutInteger( MathLink, nMaxComplexity ) == 0 )

        return MATHLINK_ERROR;



    return OK;

}





//

//   s e n d D e p e n d e n c y

//



tRetCode tSummary::sendDependency( MLINK MathLink, int nLayer )



{

    tRetCode nRet;

    tMultiArray<int> Dependency;



    if( ( nRet = getDependency( nLayer, Dependency ) ) != OK )

        return nRet;

    if( MLPutFunction( MathLink, "Transpose", 1 ) == 0 )

        return MATHLINK_ERROR;

    if( ( nRet = Dependency.toMathLink( MathLink ) ) != OK )

        return nRet;



    return OK;

}





//

//   s e n d B o u n d a r y

//



tRetCode tSummary::sendBoundary( MLINK MathLink, int nLayer )



{

    tRetCode nRet;

    tMultiArray<int> Boundary;



    if( ( nRet = getBoundary( nLayer, Boundary ) ) != OK )

        return nRet;

    if( MLPutFunction( MathLink, "Transpose", 1 ) == 0 )

        return MATHLINK_ERROR;

    if( ( nRet = Boundary.toMathLink( MathLink ) ) != OK )

        return nRet;



    return OK;

}





//

//   s e n d V o l a t i l i t y

//



tRetCode tSummary::sendVolatility( MLINK MathLink, int nLayer )



{

    tRetCode nRet;

    tMultiArray<double> Volatility;



    if( ( nRet = getVolatility( nLayer, Volatility ) ) != OK )

        return nRet;

    if( MLPutFunction( MathLink, "Transpose", 1 ) == 0 )

        return MATHLINK_ERROR;

    if( ( nRet = Volatility.toMathLink( MathLink ) ) != OK )

        return nRet;



    return OK;

}





//

//   s e n d T o t a l

//



tRetCode tSummary::sendTotal( MLINK MathLink, int nLayer )



{

    tRetCode nRet;

    tMultiArray<double> Total;



    if( ( nRet = getTotal( nLayer, Total ) ) != OK )

        return nRet;

    if( MLPutFunction( MathLink, "Transpose", 1 ) == 0 )

        return MATHLINK_ERROR;

    if( ( nRet = Total.toMathLink( MathLink ) ) != OK )

        return nRet;



    return OK;

}





//

//   s e n d T o t a l

//



tRetCode tSummary::sendTotal( MLINK MathLink, const tHeap<int>& LayerSign )



{

    tRetCode nRet;

    tMultiArray<double> Total;



    if( ( nRet = getTotal( LayerSign, 0, &Total ) ) != OK )

        return nRet;

    if( MLPutFunction( MathLink, "Transpose", 1 ) == 0 )

        return MATHLINK_ERROR;

    if( ( nRet = Total.toMathLink( MathLink ) ) != OK )

        return nRet;



    return OK;

}





//

//   s e n d M a y E x P o l i c y

//



tRetCode tSummary::sendMayExPolicy( MLINK MathLink, int nLayer,

    const tHeap<int>& ClaimSign )



{

    tRetCode nRet;

    tMultiArray<int> MayExPolicy;



    if( ( nRet = getExPolicy( nLayer, xMayExercise, ClaimSign,

            &MayExPolicy, 0 ) ) != OK ) {

        return nRet;

    }

    if( MLPutFunction( MathLink, "Transpose", 1 ) == 0 )

        return MATHLINK_ERROR;

    if( ( nRet = MayExPolicy.toMathLink( MathLink ) ) != OK )

        return nRet;



    return OK;

}





//

//   s e n d F o r c e E x P o l i c y

//



tRetCode tSummary::sendForceExPolicy( MLINK MathLink, int nLayer,

    const tHeap<int>& ClaimSign )



{

    tRetCode nRet;

    tMultiArray<int> ForceExPolicy;



    if( ( nRet = getExPolicy( nLayer, xForceExercise, ClaimSign,

            &ForceExPolicy, 0 ) ) != OK ) {

        return nRet;

    }

    if( MLPutFunction( MathLink, "Transpose", 1 ) == 0 )

        return MATHLINK_ERROR;

    if( ( nRet = ForceExPolicy.toMathLink( MathLink ) ) != OK )

        return nRet;



    return OK;

}





//

//   s e n d M a y E x P a y o f f

//



tRetCode tSummary::sendMayExPayoff( MLINK MathLink, int nLayer,

    const tHeap<int>& ClaimSign )



{

    tRetCode nRet;

    tMultiArray<double> Payoff;



    if( ( nRet = getExPolicy( nLayer, xMayExercise, ClaimSign,

            0, &Payoff ) ) != OK ) {

        return nRet;

    }

    if( MLPutFunction( MathLink, "Transpose", 1 ) == 0 )

        return MATHLINK_ERROR;

    if( ( nRet = Payoff.toMathLink( MathLink ) ) != OK )

        return nRet;



    return OK;

}





//

//   s e n d F o r c e E x P a y o f f

//



tRetCode tSummary::sendForceExPayoff( MLINK MathLink, int nLayer,

    const tHeap<int>& ClaimSign )



{

    tRetCode nRet;

    tMultiArray<double> Payoff;



    if( ( nRet = getExPolicy( nLayer, xForceExercise, ClaimSign,

            0, &Payoff ) ) != OK ) {

        return nRet;

    }

    if( MLPutFunction( MathLink, "Transpose", 1 ) == 0 )

        return MATHLINK_ERROR;

    if( ( nRet = Payoff.toMathLink( MathLink ) ) != OK )

        return nRet;



    return OK;

}



#endif



MTG_END_NAMESPACE

