// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgTclInterest.h"

#include "MtgSource.h"

#include "MtgUSBondMath.h"



MTG_BEGIN_NAMESPACE



#if defined(_MTG_WITH_TCL)





//

//   v a r i a b l e s

//



Tcl_ObjType tTclInterest::m_Spline = {

	"Spline",

	freeSpline, dupSpline, updateSpline, setSpline

};



Tcl_ObjType tTclInterest::m_Drift = {

	"Drift",

	freeDrift, dupDrift, updateDrift, setDrift

};





//

//   s e t S p l i n e

//



int tTclInterest::setSpline( Tcl_Interp *pInterp, Tcl_Obj *pObj )



{

    Tcl_ObjType *pOldType = pObj->typePtr;



    int nLength;

    char* sString = Tcl_GetStringFromObj( pObj, &nLength );



    if( pOldType != 0 && pOldType->freeIntRepProc != 0 )

        pOldType->freeIntRepProc( pObj );



    pObj->internalRep.otherValuePtr = 0;

    pObj->typePtr = &m_Spline;



        // now extract options:



    int objc;

    char** objv1;

    tInterest Interest;



    if( Tcl_SplitList( pInterp, sString, &objc, &objv1 ) != TCL_OK )

        return TCL_ERROR;



    if( objc > 0 ) {

        Tcl_Obj** objv2 = new Tcl_Obj*[objc];



        for( int i = 0; i < objc; ++i ) {

            objv2[i] = Tcl_NewStringObj( objv1[i], strlen( objv1[i] ) );

            Tcl_IncrRefCount( objv2[i] );

        }



        int nNextObj = 0;

        bool bOk = getInterest( pInterp, objc, objv2, nNextObj, &Interest );



        for( MTG_FOR_INIT( int ) i = 0; i < objc; ++i )

            Tcl_DecrRefCount( objv2[i] );

        delete objv2;



        if( ! bOk || nNextObj != objc ) {

            Tcl_Free( (char*) objv1 );

            return TCL_ERROR;   // too many options

        }

    }



    Tcl_Free( (char*) objv1 );

    pObj->internalRep.otherValuePtr =  new tInterestSpline( Interest );



    return TCL_OK;

}





//

//   u p d a t e S p l i n e

//



void tTclInterest::updateSpline( Tcl_Obj *pObj )



{

    if( pObj->internalRep.otherValuePtr == 0 ) {

        pObj->bytes = Tcl_Alloc( 1 );

        strcpy( pObj->bytes, "" );

        pObj->length = 0;

        return;

    }



    tInterestSpline* p = 

        static_cast<tInterestSpline*>( pObj->internalRep.otherValuePtr );



    char* s = "";



    pObj->length = strlen( s );

    pObj->bytes = Tcl_Alloc( pObj->length + 1 );



    strcpy( pObj->bytes, s );

}





//

//   d u p S p l i n e

//



void tTclInterest::dupSpline( Tcl_Obj *pSrc, Tcl_Obj *pDup )



{

    tInterestSpline* p =

        static_cast<tInterestSpline*>( pSrc->internalRep.otherValuePtr );



    if( p )

        pDup->internalRep.otherValuePtr = new tInterestSpline( *p );

    else

        pDup->internalRep.otherValuePtr = 0;



    pDup->typePtr = &m_Spline;

}





//

//   f r e e S p l i n e

//



void tTclInterest::freeSpline( Tcl_Obj *pObj )



{

    if( pObj->internalRep.otherValuePtr != 0 ) {

        delete static_cast<tInterestSpline*>(

            pObj->internalRep.otherValuePtr );

    }

}





//

//   s e t D r i f t

//



int tTclInterest::setDrift( Tcl_Interp *pInterp, Tcl_Obj *pObj )



{

    Tcl_ObjType *pOldType = pObj->typePtr;



    int nLength;

    char* sString = Tcl_GetStringFromObj( pObj, &nLength );



    if( pOldType != 0 && pOldType->freeIntRepProc != 0 )

        pOldType->freeIntRepProc( pObj );



    pObj->internalRep.otherValuePtr = 0;

    pObj->typePtr = &m_Drift;



        // now extract options:



    int objc;

    char** objv;



    if( Tcl_SplitList( pInterp, sString, &objc, &objv ) != TCL_OK )

        return TCL_ERROR;



    if( objc == 0 ) {

        Tcl_Free( (char*) objv );

        return TCL_ERROR;

    }



    tSource* pSource = 0;



    for( int i = 0; i < objc; ++i ) {

        char* sString = StrToLower( objv[i] );



        if( strcmp( sString, "-file" ) ) {

            if( pSource != 0 || i + 1 == objc ) {

                if( pSource != 0 )

                    delete pSource;

                Tcl_Free( (char*) objv );

                delete sString;

                return TCL_ERROR;

            }

            pSource = new tFileSource( objv[++i] );

        }

        else

        if( strcmp( sString, "-spec" ) ) {

            if( pSource != 0 || i + 1 == objc ) {

                if( pSource != 0 )

                    delete pSource;

                Tcl_Free( (char*) objv );

                delete sString;

                return TCL_ERROR;

            }

            pSource = new tStringSource( objv[++i] );

        }

        else {

            if( pSource != 0 )

                delete pSource;

            Tcl_Free( (char*) objv );

            delete sString;

            return TCL_ERROR;

        }



        delete sString;

    }



    Tcl_Free( (char*) objv );



    tObject* pDrift;

    tParser Parser;

    tSystem System;



    pSource->makeSafe();

    if( Parser.setSource( pSource ) != OK ) {

        delete pSource;

        return TCL_ERROR;

    }



    if( tDrift::parse( Parser, System, pDrift ) != OK ) {

        Parser.setSource( 0 );

        delete pSource;

        return TCL_ERROR;

    }



    pObj->internalRep.otherValuePtr = pDrift;

    Parser.setSource( 0 );

    delete pSource;



    return TCL_OK;

}





//

//   u p d a t e D r i f t

//



void tTclInterest::updateDrift( Tcl_Obj *pObj )



{

    if( pObj->internalRep.otherValuePtr == 0 ) {

        pObj->bytes = Tcl_Alloc( 1 );

        strcpy( pObj->bytes, "" );

        pObj->length = 0;

        return;

    }



    tDrift* p = static_cast<tDrift*>( pObj->internalRep.otherValuePtr );



    char* s = "";



    pObj->length = strlen( s );

    pObj->bytes = Tcl_Alloc( pObj->length + 1 );



    strcpy( pObj->bytes, s );

}





//

//   d u p D r i f t

//



void tTclInterest::dupDrift( Tcl_Obj *pSrc, Tcl_Obj *pDup )



{

    tDrift* p = static_cast<tDrift*>( pSrc->internalRep.otherValuePtr );



    if( p )

        pDup->internalRep.otherValuePtr = p->clone();

    else

        pDup->internalRep.otherValuePtr = 0;



    pDup->typePtr = &m_Drift;

}





//

//   f r e e D r i f t

//



void tTclInterest::freeDrift( Tcl_Obj *pObj )



{

    if( pObj->internalRep.otherValuePtr != 0 )

        delete static_cast<tDrift*>( pObj->internalRep.otherValuePtr );

}





//

//   g e t R e a d e r

//



tInterestSpline* tTclInterest::getSpline( Tcl_Interp* pInterp, Tcl_Obj* pObj )



{

    if( pObj->typePtr != &m_Spline ||

        pObj->internalRep.otherValuePtr == 0 ) {

        if( pInterp != 0 ) {

            Tcl_Obj* pObj = Tcl_GetObjResult( pInterp );

            Tcl_AppendToObj( pObj,

                "arg of type Spline expected", -1 );

        }

        return 0;

    }



    return static_cast<tInterestSpline*>( pObj->internalRep.otherValuePtr );

}





//

//   g e t I n t e r e s t

//



bool tTclInterest::getInterest( Tcl_Interp* pInterp, int objc,

    Tcl_Obj *CONST objv[], int& nNextObj, tInterest* pInterest,

    tDate* pBase )



{

    static struct tKeyWord {

        const char* m_sName;

        int m_nId;

        tDayCount* m_pDayCount;

    } KeyWord[] = {

        { "-act/act", 0, &DayCountACT_ACT },

        { "-act/365", 0, &DayCountACT_365 },

        { "-act/365.25", 0, &DayCountACT_365_25 },

        { "-act/365nl", 0, &DayCountACT_365_NL },

        { "-act/365isda", 0, &DayCountACT_365_ISDA },

        { "-act/360", 0, &DayCountACT_360 },

        { "-30/360isda", 0, &DayCount30_360_ISDA },

        { "-30/360psa", 0, &DayCount30_360_PSA },

        { "-30/360sia", 0, &DayCount30_360_SIA },

        { "-30e/360", 0, &DayCount30E_360 },

        { "-exponential", 1, 0 },

        { "-linear", 2, 0 },

        { "-moneymarket", 2, 0 },

        { "-yield", 3, 0 },

        { "-discount", 4, 0 },

        { "-periods", 5, 0 },

        { "-today", 6, 0 },

        { "-base", 6, 0 }

    };



    bool b1 = false;

    bool b2 = false;

    bool b3 = false;

    bool b4 = false;

    bool b5 = false;



    while( nNextObj < objc ) {

        int nLength;

        char* sString = Tcl_GetStringFromObj( objv[nNextObj], &nLength );



        if( nLength == 0 || *sString != '-' )

            break;



        char* s = StrToLower( sString );

        int k = 0;



        for( k = 0; k < sizeof(KeyWord) / sizeof(tKeyWord); ++k ) {

            if( strcmp( s, KeyWord[k].m_sName ) == 0 )

                break;

        }

        delete s;



        if( k == sizeof(KeyWord) / sizeof(tKeyWord) )

            break;



        tKeyWord& W = KeyWord[k];



        if( W.m_nId == 0 ) {

            if( b1 )

                return false;

            if( pInterest )

                pInterest->set( *W.m_pDayCount );

            b1 = true;

        }

        else

        if( W.m_nId == 1 || W.m_nId == 2 ) {

            if( b2 )

                return false;

            if( pInterest ) {

                pInterest->set( W.m_nId == 1 ?

                    tInterest::xExponential : tInterest::xLinear );

            }

            b2 = true;

        }

        else

        if( W.m_nId == 3 || W.m_nId == 4 ) {

            if( b3 )

                return false;

            if( pInterest ) {

                pInterest->set( W.m_nId == 3 ?

                    tInterest::xYield : tInterest::xDiscount );

            }

            b3 = true;

        }

        else

        if( W.m_nId == 5 ) {

            int nPeriods;



            if( b4 || ++nNextObj >= objc ||

                Tcl_GetIntFromObj( pInterp, objv[nNextObj],

                    &nPeriods ) != TCL_OK ) {

                return false;

            }

            if( pInterest )

                pInterest->set( tPeriodCompounder( nPeriods ) );

            b4 = true;

        }

        else {

            MTG_ASSERT( W.m_nId == 6 );



            if( pBase == 0 )

                break;



            if( b5 || ++nNextObj >= objc ||

                    ! getDate( objv[nNextObj], *pBase ) ) {

                return false;

            }

            b5 = true;

        }



        ++nNextObj;

    }



    return true;

}





//

//   g e t C o m p o u n d e r

//



bool tTclInterest::getCompounder( Tcl_Interp* pInterp, int objc,

    Tcl_Obj *CONST objv[], int& nNextObj, bool& bHasCompounder,

    double& gCoupon, tBondMath*& pMath )



{

    gCoupon = 0;

    pMath = &USBondMath;



    static struct tKeyWord {

        const char* m_sName;

        int m_nId;

        tBondMath* m_pMath;

    } KeyWord[] = {

        { "-coupon", 0, 0 },

        { "-canada", 1, &USBondMath },

        { "-france", 1, &USBondMath },

        { "-germany", 1, &USBondMath },

        { "-italy", 1, &USBondMath },

        { "-japan", 1, &USBondMath },

        { "-uk", 1, &USBondMath },

        { "-us", 1, &USBondMath },

    };



    bool b1 = false;

    bool b2 = false;



    while( nNextObj < objc ) {

        int nLength;

        char* sString = Tcl_GetStringFromObj( objv[nNextObj], &nLength );



        if( nLength == 0 || *sString != '-' )

            break;



        char* s = StrToLower( sString );

        int k = 0;



        for( k = 0; k < sizeof(KeyWord) / sizeof(tKeyWord); ++k ) {

            if( strcmp( s, KeyWord[k].m_sName ) == 0 )

                break;

        }

        delete s;



        if( k == sizeof(KeyWord) / sizeof(tKeyWord) )

            break;



        tKeyWord& W = KeyWord[k];



        if( W.m_nId == 0 ) {

            if( b1 || ++nNextObj >= objc ||

                Tcl_GetDoubleFromObj( pInterp, objv[nNextObj],

                    &gCoupon ) != TCL_OK ) {

                return false;

            }

            b1 = true;

        }

        else {

            MTG_ASSERT( W.m_nId == 1 );



            if( b2 )

                return false;

            pMath = W.m_pMath;

            b2 = true;

        }



        ++nNextObj;

    }



    bHasCompounder = b1;

    return true;

}





//

//   g e t D a t e

//



bool tTclInterest::getDate( Tcl_Obj* pObj, tDate& Date )



{

    int nLength;

    char* sString = Tcl_GetStringFromObj( pObj, &nLength );

    const char* sRest = Date.set( sString );



    if( sRest == 0 )

        return false;



    while( isspace( *sRest ) )

        ++sRest;



    return ! *sRest;

}





//

//   s e r v i c e

//



int tTclInterest::service( Tcl_Interp *pInterp,

    int objc, Tcl_Obj *CONST objv[], char* sUsage,

    double (tInterest::*Svc)( tDate, tDate, double ) const )



{

    tDate Start, Value;

    double gQuantity;

    tInterest Interest;



    int nNextObj = 1;



    if( ! getInterest( pInterp, objc, objv, nNextObj, &Interest ) )

        return TCL_ERROR;



    int n = objc - nNextObj;



    if( n < 2 || n > 3 ) {

        Tcl_WrongNumArgs( pInterp, 1, objv, sUsage );

        return TCL_ERROR;

    }



    if( n == 3 ) {

        if( ! getDate( objv[nNextObj++], Start ) )

            return TCL_ERROR;

    }

    else {

        Start = tDate::today();

    }



    if( ! getDate( objv[nNextObj++], Value ) )

        return TCL_ERROR;



    if( Tcl_GetDoubleFromObj( pInterp, objv[nNextObj++],

            &gQuantity ) != TCL_OK ) {

        return TCL_ERROR;

    }



    Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(

        (Interest.*Svc)( Start, Value, gQuantity ) ) );



    return TCL_OK;

}





//

//   p r e s e n t V a l u e

//



int tTclInterest::presentValue( ClientData ClientData,

    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )



{

    return service( pInterp, objc, objv,

        "?options? ?startdate? valuedate yield",

        tInterest::presentValue );

}





//

//   f u t u r e V a l u e

//



int tTclInterest::futureValue( ClientData ClientData,

    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )



{

    return service( pInterp, objc, objv,

        "?options? ?startdate? valuedate yield",

        tInterest::futureValue );

}





//

//   i n v P r e s e n t V a l u e

//



int tTclInterest::invPresentValue( ClientData ClientData,

    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )



{

    return service( pInterp, objc, objv,

        "?options? ?startdate? valuedate yield",

        tInterest::invPresentValue );

}





//

//   i n v F u t u r e V a l u e

//



int tTclInterest::invFutureValue( ClientData ClientData,

    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )



{

    return service( pInterp, objc, objv,

        "?options? ?startdate? valuedate yield",

        tInterest::invFutureValue );

}





//

//   c r e a t e S p l i n e

//



int tTclInterest::createSpline( ClientData ClientData,

    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )



{

    tInterest Interest;



    int nNextObj = 1;



    if( ! getInterest( pInterp, objc, objv, nNextObj, &Interest ) )

        return TCL_ERROR;



    if( objc > nNextObj ) {

        Tcl_WrongNumArgs( pInterp, 1, objv, "?options?" );

        return TCL_ERROR;

    }



    Tcl_Obj* pObj = Tcl_NewObj();



    pObj->bytes = 0;

    pObj->internalRep.otherValuePtr = new tInterestSpline( Interest );

    pObj->typePtr = &m_Spline;



    Tcl_SetObjResult( pInterp, pObj );

    return TCL_OK;

}





//

//   c r e a t e D r i f t

//



int tTclInterest::createDrift( ClientData ClientData,

    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )



{

    tInterest Interest;



    int nNextObj = 1;



    if( ! getInterest( pInterp, objc, objv, nNextObj, &Interest ) )

        return TCL_ERROR;



    if( objc > nNextObj ) {

        Tcl_WrongNumArgs( pInterp, 1, objv, "?options?" );

        return TCL_ERROR;

    }



    Tcl_Obj* pObj = Tcl_NewObj();



    pObj->bytes = 0;

    pObj->internalRep.otherValuePtr = new tInterestSpline( Interest );

    pObj->typePtr = &m_Spline;



    Tcl_SetObjResult( pInterp, pObj );

    return TCL_OK;

}





//

//   s p l i n e A d d P a y m e n t

//



int tTclInterest::splineAddPayment( ClientData ClientData,

    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )



{

    bool bHasCompounder;

    double gCoupon;

    tBondMath* pMath;



    int nNextObj = 1;



    if( ! getCompounder( pInterp, objc, objv, nNextObj,

            bHasCompounder, gCoupon, pMath ) ) {

        return TCL_ERROR;

    }



    int n = objc - nNextObj;

    if( n < 4 || bHasCompounder && n > 5 ) {

        Tcl_WrongNumArgs( pInterp, 1, objv,

            "?options? spline startdate presentvalue valuedate ?payment?" );

        return TCL_ERROR;

    }



    tInterestSpline* pSpline = getSpline( pInterp, objv[nNextObj++] );

    if( pSpline == 0 )

        return TCL_ERROR;



    tDate Start;

    double gPV;



    if( ! getDate( objv[nNextObj++], Start ) ||

        Tcl_GetDoubleFromObj( pInterp, objv[nNextObj++], &gPV ) != TCL_OK ) {

        return TCL_ERROR;

    }



    if( pMath )

        Start = pMath->toSettlement( Start );



    tHeap<tPayment> Payment;



    while( nNextObj < objc ) {

        tDate Value;

        double gAmount;



        if( ! getDate( objv[nNextObj++], Value ) )

            return TCL_ERROR;

        if( nNextObj < objc ) {

            if( Tcl_GetDoubleFromObj( pInterp,

                    objv[nNextObj++], &gAmount ) != TCL_OK ) {

                return TCL_ERROR;

            }

        }

        else {

            gAmount = 100;

        }

        Payment.append( tPayment( Value, gAmount ) );

    }



    if( bHasCompounder ) {

        MTG_ASSERT( pMath != 0 );



        tCompounder* p = pMath->createCompounder( Start, Payment[0].m_Date );

        tCouponCompounder* q = dynamic_cast<tCouponCompounder*>( p );



        if( q != 0 ) {

            pSpline->addPayment( Start, gPV,

                Payment[0].m_gAmount, gCoupon, *q );

        }

        else {

            pSpline->addPayment( Start, gPV, Payment[0].m_Date,

                Payment[0].m_gAmount, gCoupon, *p );

        }

        delete p;

    }

    else {

        pSpline->addPayment( Start, gPV, Payment );

    }



    return TCL_OK;

}





//

//   s p l i n e G e t F o r w a r d C u r v e

//



int tTclInterest::splineGetForwardCurve( ClientData ClientData,

    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )



{

    tDate Start = tDate::today();



        // first do it without actually initializing

        // the interest object:



    int nNextObj = 1;

    if( ! getInterest( pInterp, objc, objv, nNextObj, 0, &Start ) )

        return TCL_ERROR;



    int n = objc - nNextObj;

    if( n != 1 ) {

        Tcl_WrongNumArgs( pInterp, 1, objv, "?options? spline" );

        return TCL_ERROR;

    }



    tInterestSpline* pSpline = getSpline( pInterp, objv[nNextObj++] );

    if( pSpline == 0 )

        return TCL_ERROR;



    tInterest Interest( pSpline->interest() );



        // now initialize the interest object by using the other

        // one as a template:

    nNextObj = 1;

    if( ! getInterest( pInterp, objc, objv, nNextObj, &Interest, &Start ) )

        return TCL_ERROR;



    n = 3 * pSpline->numOfLegs();

    if( n == 0 )

        return TCL_OK;



    Tcl_Obj** v = new Tcl_Obj*[n];

    int m = 0;



    tDate Value;

    double gYield;



    pSpline->getFirstLeg( Value, gYield, Interest );

    do{

        MTG_ASSERT( m + 1 < n );



        char s[32];



        sprintf( s, "%02d/%02d/%02d",

            Value.month(), Value.day(), Value.year() % 100 );

        v[m++] = Tcl_NewStringObj( s, strlen( s ) );

        v[m++] = Tcl_NewLongObj( Value - Start );

        v[m++] = Tcl_NewDoubleObj( gYield );



    } while( pSpline->getNextLeg( Value, gYield, Interest ) );

    MTG_ASSERT( m == n );



    Tcl_SetObjResult( pInterp, Tcl_NewListObj( n, v ) );

    delete v;



    return TCL_OK;

}





//

//   c r e a t e E x t e n s i o n

//



tRetCode tTclInterest::createExtension( tTclKernel& Kernel )



{

    Tcl_RegisterObjType( &m_Spline );

    Tcl_RegisterObjType( &m_Drift );



    #define MTG_CREATE_CMD( f ) \

        Tcl_CreateObjCommand( Kernel.interp(), "Mtg::" #f, f, 0, 0 );



    MTG_CREATE_CMD( presentValue )

    MTG_CREATE_CMD( futureValue )



    MTG_CREATE_CMD( invPresentValue )

    MTG_CREATE_CMD( invFutureValue )

    

    MTG_CREATE_CMD( createSpline )

    MTG_CREATE_CMD( createDrift )



    MTG_CREATE_CMD( splineAddPayment )

    MTG_CREATE_CMD( splineGetForwardCurve )



    #undef MTG_CREATE_CMD



    return OK;

}



#endif



MTG_END_NAMESPACE

