// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"



#if defined(_WIN32)

    #include <windows.h>

#endif



#include "MtgTclKernel.h"

#include "MtgJobServer.h"



#if defined(_WIN32)

    #include <io.h>

    #include <fcntl.h>

    #include <sys/types.h>

    #include <sys/stat.h>

    #include <share.h>

    #include <process.h>

#endif



#if defined(_MTG_WITH_TCL)



MTG_BEGIN_NAMESPACE





//

//   v a r i a b l e s

//



bool tTclKernel::m_bInitialized = false;

bool tTclKernel::m_bInteractive = false;





//

//   c r e a t e

//



void tTclKernel::create()



{

    m_pInterp = Tcl_CreateInterp();



    if( m_pInterp == 0 ) {  // check with isOpen()

        MTG_TRACE( "Creation of Tcl interpreter failed\n" );

        return;

    }



    if( Tcl_Init( m_pInterp ) == TCL_ERROR /* ||

        Tk_Init( m_pInterp ) == TCL_ERROR */ ) {

        cleanup();

        MTG_TRACE( "Creation of Tcl interpreter failed\n" );

	    return;

    }



    // Tcl_StaticPackage( m_pInterp, "Tk", Tk_Init, Tk_SafeInit );



    Tcl_SetVar( m_pInterp, "tcl_interactive",

	    m_bInteractive ? "1" : "0", TCL_GLOBAL_ONLY );



    #define MTG_CREATE_CMD( f ) \

        Tcl_CreateObjCommand( m_pInterp, "Mtg::" #f, f, 0, 0 );



    MTG_CREATE_CMD( system )

    MTG_CREATE_CMD( fork )

    MTG_CREATE_CMD( tclify )

    MTG_CREATE_CMD( lock )

    MTG_CREATE_CMD( unlock )



    #undef MTG_CREATE_CMD

}





//

//   c l e a n u p

//



void tTclKernel::cleanup()



{

    if( m_pInterp != 0 ) {

        Tcl_DeleteInterp( m_pInterp );

        m_pInterp = 0;

    }

}





//

//   s h o w E r r o r

//



void tTclKernel::showError()



{

    Tcl_Channel ErrChannel = Tcl_GetStdChannel( TCL_STDERR );



    if( ErrChannel ) {

	    Tcl_AddErrorInfo( m_pInterp, "" );

		Tcl_Write( ErrChannel,

		    Tcl_GetVar( m_pInterp, "errorInfo", TCL_GLOBAL_ONLY ), -1 );

		Tcl_Write( ErrChannel, "\n", 1 );

	}

}





//

//   s y s t e m

//



int tTclKernel::system( ClientData ClientData, Tcl_Interp *pInterp,

    int objc, Tcl_Obj *CONST objv[] )



{

    if( objc < 2 ) {

        Tcl_WrongNumArgs( pInterp, 1, objv, "?options? arg ?arg?" );

        return TCL_ERROR;

    }



    int nNextObj = 1;

    bool bDetach = false;



    while( nNextObj < objc ) {

        int nLength;

        char* sString = Tcl_GetStringFromObj( objv[nNextObj], &nLength );



        if( nLength > 0 && *sString == '-' ) { 

            char* s = StrToLower( sString );

            int c = strcmp( s, "-detach" );

            delete s;

            

            if( c == 0 ) {

                if( bDetach )

                    return TCL_ERROR;

                bDetach = true;

            }

            else {

                return TCL_ERROR;

            }



            ++nNextObj;

        }

        else {

            break;

        }

    }



    if( nNextObj == objc )

        return TCL_ERROR;



    char** argv = new char*[objc - nNextObj + 1];

    int nLength;



    for( int i = nNextObj; i < objc; ++i )

        argv[i - nNextObj] = Tcl_GetStringFromObj( objv[i], &nLength );



    if( bDetach ) {

        argv[objc - nNextObj] = 0;

        Tcl_SetObjResult( pInterp,

            Tcl_NewIntObj( spawnv( _P_DETACH, argv[0], argv ) ) );

    }

    else {

        char* arg = Tcl_Concat( objc - nNextObj, argv );

        delete argv;

        ::system( arg );

        Tcl_Free( arg );

        Tcl_SetObjResult( pInterp, Tcl_NewIntObj( 0 ) );

    }



    return TCL_OK;

}





//

//   f o r k

//



int tTclKernel::fork( ClientData ClientData, Tcl_Interp *pInterp,

    int objc, Tcl_Obj *CONST objv[] )



{

    char* sArg = 0;



    if( objc > 1 ) {

        char** argv = new char*[objc - 1];

        int nLength;



        for( int i = 1; i < objc; ++i )

            argv[i - 1] = Tcl_GetStringFromObj( objv[i], &nLength );



        sArg = Tcl_Concat( objc - 1, argv );

        delete argv;

    }



        // fork() doesn't fork directly, but through the job

        // server, which uses a pipe to connect to a background

        // daemon that controls forking



    Tcl_SetObjResult( pInterp, Tcl_NewIntObj( tJobServer::submit( sArg ) ) );



    if( sArg != 0 )

        Tcl_Free( sArg );



    return TCL_OK;

}





//

//   t c l i f y

//



int tTclKernel::tclify( ClientData ClientData, Tcl_Interp *pInterp,

    int objc, Tcl_Obj *CONST objv[] )



{

        // This functions takes its argument string and converts

        // it into a string that, enclosed in quotes, can be

        // used in Tcl scripts to represent the same string.



    if( objc != 2 ) {

        Tcl_WrongNumArgs( pInterp, 1, objv, "arg" );

        return TCL_ERROR;

    }



    int nLength;

    char* sString;

    sString = Tcl_GetStringFromObj( objv[1], &nLength );



    int nFlags;

    nLength = Tcl_ScanElement( sString, &nFlags );



    char* sRes = new char[nLength + 1];

    Tcl_ConvertElement( sString, sRes, nFlags | TCL_DONT_USE_BRACES );



    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( sRes, -1 ) );



    delete sRes;

    return TCL_OK;

}





//

//   l o c k

//



int tTclKernel::lock( ClientData ClientData, Tcl_Interp *pInterp,

    int objc, Tcl_Obj *CONST objv[] )



{

    if( objc < 2 ) {

        Tcl_WrongNumArgs( pInterp, 1, objv, "?options? filename" );

        return TCL_ERROR;

    }



    int nNextObj = 1;

    char* sFileName = 0;



    int nTry = -1;

    int nWait = -1;



    while( nNextObj < objc ) {

        int nLength;

        char* sString = Tcl_GetStringFromObj( objv[nNextObj], &nLength );



        if( nLength > 0 && *sString == '-' ) { 

            char* s = StrToLower( sString );

            

            if( strcmp( s, "-try" ) == 0 ) {

                if( nTry > 0 ||

                    ++nNextObj == objc ||

                    Tcl_GetIntFromObj( pInterp,

                        objv[nNextObj], &nTry ) != TCL_OK ||

                            nTry <= 0 ) {

                    delete s;

                    return TCL_ERROR;

                }

            }

            else

            if( strcmp( s, "-wait" ) == 0 ) {

                if( nWait >= 0 ||

                    ++nNextObj == objc ||

                    Tcl_GetIntFromObj( pInterp,

                        objv[nNextObj], &nWait ) != TCL_OK ||

                            nWait < 0 ) {

                    delete s;

                    return TCL_ERROR;

                }

            }

            else {

                delete s;

                return TCL_ERROR;

            }



            delete s;

            ++nNextObj;

        }

        else {

            sFileName = sString;

            if( ++nNextObj != objc )

                return TCL_ERROR;

        }

    }



    if( sFileName == 0 )

        return TCL_ERROR;



    if( nTry < 0 )

        nTry = 3;

    if( nWait < 0 )

        nWait = 1;



    int fd = -1;

    

    for( int k = 0; k < nTry && fd < 0; ++k ) {

        if( k > 0 && nWait > 0 )

            WaitFor( nWait );

        fd = sopen( sFileName,

                _O_WRONLY | _O_CREAT | _O_TRUNC,

                _SH_DENYRW, _S_IREAD | _S_IWRITE );

    }



    Tcl_SetObjResult( pInterp, Tcl_NewIntObj( fd ) );

    return TCL_OK;

}





//

//   u n l o c k

//



int tTclKernel::unlock( ClientData ClientData, Tcl_Interp *pInterp,

    int objc, Tcl_Obj *CONST objv[] )



{

    if( objc != 2 ) {

        Tcl_WrongNumArgs( pInterp, 1, objv, "filename" );

        return TCL_ERROR;

    }



    int fd;



    if( Tcl_GetIntFromObj( pInterp, objv[1], &fd ) != TCL_OK )

        return TCL_ERROR;



    if( fd >= 0 )

        ::close( fd );



    return TCL_OK;

}





//

//   i n i t

//



void tTclKernel::init( const char* sExecutable, bool bInteractive )



{

    if( sExecutable != 0 )

        Tcl_FindExecutable( const_cast<char*>( sExecutable ) );

    m_bInteractive = bInteractive;

    m_bInitialized = true;

}





//

//   e x i t

//



void tTclKernel::exit()



{

    if( m_bInitialized )

        Tcl_Finalize();

}





//

//   t T c l K e r n e l

//



tTclKernel::tTclKernel()



{

    MTG_ASSERT( m_bInitialized );

    create();

}





//

//   ~ t T c l K e r n e l

//



tTclKernel::~tTclKernel()



{

    cleanup();

}





//

//   u s e A r g s

//



tRetCode tTclKernel::useArgs( int& argc, const char* argv[] )



{

    for( int k = 1; k < argc; ++k ) {

        if( argv[k][0] == '-' || argv[k][0] == '/' ) {

        }

    }



        // Remove used arguments.



    int j = 1;

    for( MTG_FOR_INIT( int ) k = 1; k < argc; ++k ) {

        if( argv[k] != 0 )

            argv[j++] = argv[k];

    }

    argc = j;



    return OK;

}





//

//   s e t A r g s

//



void tTclKernel::setArgs( int argc, const char* argv[] )



{

    MTG_ASSERT( isOpen() && argc > 0 );



    char* sArgs = Tcl_Merge( argc - 1, const_cast<char**>( argv ) + 1 );



    setVar( "argv", sArgs );

    Tcl_Free( sArgs );



    char sBuffer[16];



    sprintf( sBuffer, "%d", argc - 1 );

    setVar( "argc", sBuffer );

}





//

//   s e t V a r

//



void tTclKernel::setVar( const char* sName, const char* sValue )



{

    MTG_ASSERT( isOpen() );



    Tcl_SetVar( m_pInterp, const_cast<char*>( sName ),

        const_cast<char*>( sValue ), TCL_GLOBAL_ONLY );

}





//

//   r u n F i l e

//



tRetCode tTclKernel::runFile( const char* sFileName )



{

    MTG_ASSERT( isOpen() );



#if defined(_WIN32)



    char s[_MAX_PATH];



    if( _fullpath( s, sFileName, _MAX_PATH ) != 0 )

        setVar( "argv0", s );

    else

        setVar( "argv0", sFileName );



#else

    setVar( "argv0", sFileName );

#endif



    tRetCode nRet;

	int nCode = Tcl_EvalFile( m_pInterp, const_cast<char*>( sFileName ) );



	if( nCode != TCL_OK ) {

        showError();

        nRet = SCRIPT_ERROR;

	}

    else {

        nRet = OK;

    }



    return nRet;

}





//

//   r u n F i l e

//



tRetCode tTclKernel::runFile( int argc, const char* argv[] )



{

    setArgs( argc, argv );

    return runFile( argv[0] );

}





//

//   r u n B a t c h

//



tRetCode tTclKernel::runBatch( const char* sFileMask, bool bIgnoreError )



{

#if defined(_WIN32)



    struct _finddata_t FD;

    long hFile;



    if( ( hFile = _findfirst( sFileMask, &FD ) ) == -1L )

        return OK;



    tRetCode nRet = OK;

    char* sDir = GetDir();



    do{

        if( ( FD.attrib & _A_SUBDIR ) == 0 ) {

            tRetCode nRet1 = runFile( FD.name );

            SetDir( sDir );



            if( nRet1 != OK ) {

                if( nRet == OK )

                    nRet = nRet1;

                if( ! bIgnoreError )

                    break;

            }

        }

    } while( _findnext( hFile, &FD ) == 0 );



    _findclose( hFile );

    delete sDir;



    return nRet;



#else

    return NOT_IMPLEMENTED;

#endif

}





//

//   c l o s e

//



void tTclKernel::close()



{

    cleanup();

}



MTG_END_NAMESPACE



#endif