// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#if ! defined(_MTG_TERM_STRUCT_)

#define _MTG_TERM_STRUCT_



#include "MtgMap.h"

#include "MtgParser.h"



MTG_BEGIN_NAMESPACE





//

//   t T e r m S t r u c t

//

    

class tTermStruct {



        // A general remark: *maturities* are counters of time units,

        // whereas *units* are indexed starting from 0. Thus, a maturity

        // value of 60 means the corresponding instrument is alive

        // during time units 0..59. Variables nUnit, nFromUnit, nToUnit are

        // indexes of units, if not otherwise stated.



        // Time units can be anything, days, fractions of years and so on.

        // The most common time unit is a day.



    struct tSpec {

        int m_nUnit;

        double m_gFwd;

        double m_gFwd2;

        double m_gImp2;

    };



    struct tImpSpec {

        double m_gFwd;

        double m_gImp;

        int m_nMaturity;

    };



    typedef tMap<tSpec,3,10> tSpecMap;

    typedef tHeap<tImpSpec> tImpSpecHeap;



    tSpecMap m_Spec;

    tImpSpecHeap m_ImpSpec;



    double m_gBand;



    bool m_bFinalized;



    double m_gMin;

    double m_gMax;



    int m_nConstantUntil;



    void cleanup();



    tRetCode diff( const tTermStruct& TS, int nSign );

    tRetCode bound( const tTermStruct& TS, int nSign, bool& bAdjusted );



    int cmp( const tTermStruct& TS, int& nEqualBefore,

        int& nEqualAfter ) const;



protected:



    void copyFrom( const tTermStruct &TS );



    virtual tRetCode calcFwd( int nMaturity1, double gImp1,

        int nMaturity2, double gImp2, double& gFwd ) const = 0;



    virtual tRetCode testImp( double gImp ) const  { return OK; }

    virtual tRetCode testFwd( double gFwd ) const  { return OK; }



    virtual void trim( double& gFwd ) const        {}

    virtual double scaleUp( double gFwd ) const    { return gFwd; }

    virtual double scaleDown( double gFwd ) const  { return gFwd; }



public:



    tTermStruct();

    tTermStruct( const tTermStruct &TS );



    virtual ~tTermStruct();



    tTermStruct& operator=( const tTermStruct &TS );



        // gImp is continuously compounded.



    tRetCode addForward( int nMaturity, double gFwd );

    tRetCode addImplied( int nMaturity, double gImp );



    void setBand( double gBand = 0 );



    tTermStruct& operator+=( double gBand ) {

        setBand( gBand );

        return *this;

    }



    tTermStruct& operator-=( double gBand ) {

        setBand( -gBand );

        return *this;

    }



        // Sometimes one has two term structures R and Q and needs

        // to know the difference R-Q, defined on a time basis (this is

        // the case in VOP, for instance). For this reason, the operator

        // -= is provided. It subtracts the drift term structure TS

        // from the current term structure and finalizes the result.



        // Instead of R = R - Q (or R -= Q), one sometimes has to compute

        // Q = R - Q. This operation is captured in Q ^= R.



        // Note that this operation does not make sense for all

        // instantiations of the class.



    tRetCode operator-=( const tTermStruct& TS ) {

        return diff( TS, 1 );

    }



    tRetCode operator^=( const tTermStruct& TS ) {

        return diff( TS, -1 );

    }



    bool operator<=( const tTermStruct& TS ) const;



    tRetCode boundFromBelow( const tTermStruct& TS );

    tRetCode boundFromAbove( const tTermStruct& TS );



        // All functions above require to finalize the result by

        // calling the following function.



    tRetCode finalize();



    bool isFinalized() const {

        return m_bFinalized;

    }



    void getFwdRange( double& gMin, double& gMax ) const;



        // Keep in mind that the time unit count starts at 0.

        // Also, in the integer version, the integral is over

        // nToUnit - nFromUnit full time units; the time unit

        // nToUnit is excluded!



    double forward( int nUnit ) const;

    double forward( int nFromUnit, int nToUnit ) const;

    double forward( double gFromUnit, double gToUnit ) const;



        // The implied rate of maturity nMaturity is simply the

        // forward rate from units 0 to (excluding) nMaturity.



    double implied( int nMaturity ) const;



    int maturity() const;

    int constantUntil() const;



    int certainUntil( const tTermStruct& TS ) const;

    int certainAfter( const tTermStruct& TS ) const;



    tRetCode parseParam( tParser& Parser, double& gPlusBand,

        double& gMinusBand, tTermStruct* pMinTS = 0,

        tTermStruct* pMaxTS = 0 );



    tRetCode parsePostfix( tParser& Parser, double gPlusBand,

        double gMinusBand, tTermStruct* pMinTS = 0,

        tTermStruct* pMaxTS = 0 );

};



MTG_END_NAMESPACE



#endif

