// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgTimeAxis.h"

#include "MtgSpaceAxis.h"

#include "MtgPortfolio.h"



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tTimeAxis::init()



{

    reset();

}





//

//   a d d S l i c e

//



void tTimeAxis::addSlice( int nDay, tDtGenerator& DtGen )



{

    MTG_ASSERT( m_aSlice.numOfElems() >= 1 );

    MTG_ASSERT( m_aSlice.last().m_nDay < nDay );

    MTG_ASSERT( m_aSlice.last().m_gFractionOfDay == 0 );



    int nFromDay = m_aSlice.last().m_nDay;



    while( nFromDay < nDay ) {

        double gNextDt = DtGen.nextDt( nFromDay );



        if( gNextDt > m_gStableDt )

            gNextDt = m_gStableDt;



        if( gNextDt < 1 ) {

            int nMiniSteps = (int) ceil( 1 / gNextDt );

            gNextDt = 1.0 / nMiniSteps;



            int j = m_aSlice.numOfElems();

            m_aSlice += nMiniSteps;



            for( int k = 0; k < nMiniSteps - 1; ++k, ++j ) {

                m_aSlice[j].m_nDay = nFromDay;

                m_aSlice[j].m_gFractionOfDay =

                    m_aSlice[j - 1].m_gFractionOfDay + gNextDt;

            }

            m_aSlice[j].m_nDay = ++nFromDay;

            m_aSlice[j].m_gFractionOfDay = 0;

        }

        else {

            int nMiniSteps = 4;

            int nStep = (int) floor( gNextDt );



            if( nFromDay + nMiniSteps * nStep >= nDay ) {

                int nDistance = nDay - nFromDay;

                int nDept = 0;



                nMiniSteps = nDistance / nStep;

                if( nDistance % nStep > 0 ) {

                    nStep = nDistance / ++nMiniSteps;

                    nDept = nDistance % nStep;

                }



                while( nFromDay < nDay ) {

                    nFromDay += nStep;

                    if( nDept > 0 ) {

                        ++nFromDay;

                        --nDept;

                    }

                    if( nFromDay > nDay )

                        nFromDay = nDay;

                    ++m_aSlice;

                    m_aSlice.last().m_nDay = nFromDay;

                    m_aSlice.last().m_gFractionOfDay = 0;

                }

            }

            else {

                int j = m_aSlice.numOfElems();

                m_aSlice += nMiniSteps;

                for( int k = 0; k < nMiniSteps; ++k, ++j ) {

                    nFromDay += nStep;

                    m_aSlice[j].m_nDay = nFromDay;

                    m_aSlice[j].m_gFractionOfDay = 0;

                }

            }

        }

    }

}





//

//   t T i m e A x i s

//



tTimeAxis::tTimeAxis()



{

    init();

}





//

//   t T i m e A x i s

//



tTimeAxis::tTimeAxis( const tTimeAxis& Axis )



{

    init();

    copyFrom( Axis );

}





//

//   ~ t T i m e A x i s

//



tTimeAxis::~tTimeAxis()



{

}





//

//   r e s e t

//



void tTimeAxis::reset()



{

    m_pEvGen = 0;

    m_pDtGen = 0;

    m_pPf = 0;

    m_nMaturity = -1;

    m_aSlice.reset();

    m_bFinalized = false;

}





//

//   o p e r a t o r =

//



tTimeAxis& tTimeAxis::operator=( const tTimeAxis& Axis )



{

    if( this != &Axis )

        copyFrom( Axis );

    return *this;

}





//

//   c o p y F r o m

//



void tTimeAxis::copyFrom( const tTimeAxis& Axis )



{

    if( this == &Axis )

        return;



    reset();



    m_pSpace = Axis.m_pSpace;

    m_pEvGen = Axis.m_pEvGen;

    m_pDtGen = Axis.m_pDtGen;

    m_pPf = Axis.m_pPf;

    m_nMaturity = Axis.m_nMaturity;

    m_gStableDt = Axis.m_gStableDt;

    m_aSlice = Axis.m_aSlice;

    m_bFinalized = Axis.m_bFinalized;

}





//

//   c l o n e

//



tTimeAxis* tTimeAxis::clone() const



{

    return new tTimeAxis( *this );

}





//

//   s e t S p a c e A x i s

//



void tTimeAxis::setSpaceAxis( tHeap<tSpaceAxis*>& Space )



{

    setSpaceAxis( &Space );

}





//

//   s e t S p a c e A x i s

//



void tTimeAxis::setSpaceAxis( tHeap<tSpaceAxis*>* pSpace )



{

    m_pSpace = pSpace;

    m_bFinalized = false;

}





//

//   s e t E v G e n e r a t o r

//



void tTimeAxis::setEvGenerator( tEvGenerator& EvGen )



{

    setEvGenerator( &EvGen );

}





//

//   s e t E v G e n e r a t o r

//



void tTimeAxis::setEvGenerator( tEvGenerator* pEvGen )



{

    m_pEvGen = pEvGen;

    m_bFinalized = false;

}





//

//   s e t D t G e n e r a t o r

//



void tTimeAxis::setDtGenerator( tDtGenerator& DtGen )



{

    setDtGenerator( &DtGen );

}





//

//   s e t D t G e n e r a t o r

//



void tTimeAxis::setDtGenerator( tDtGenerator* pDtGen )



{

    m_pDtGen = pDtGen;

    m_bFinalized = false;

}





//

//   s e t P o r t f o l i o

//



void tTimeAxis::setPortfolio( const tPortfolio& Pf )



{

    setPortfolio( &Pf );

}





//

//   s e t P o r t f o l i o

//



void tTimeAxis::setPortfolio( const tPortfolio* pPf )



{

    m_pPf = pPf;

    m_bFinalized = false;

}





//

//   s e t M a t u r i t y

//



void tTimeAxis::setMaturity( int nMaturity )



{

    m_nMaturity = nMaturity;

    m_bFinalized = false;

}





//

//   f i n a l i z e

//



void tTimeAxis::finalize()



{

    MTG_ASSERT( m_pSpace != 0 && m_pSpace->numOfElems() >= 1 );

    MTG_ASSERT( m_nMaturity > 0 || m_pEvGen != 0 || m_pPf != 0 );



    tSeqEvGenerator SeqEvGen;

    int nDay;



    if( m_bFinalized )

        return;



    if( m_pPf != 0 ) {

        m_pPf->getEvents( SeqEvGen );

        if( m_nMaturity > m_pPf->maturity() )

            SeqEvGen += m_nMaturity;

    }

    else {

        SeqEvGen += m_nMaturity;

    }



    if( m_pEvGen != 0 )

        SeqEvGen += *m_pEvGen;



        // Adjust maturity:



    m_nMaturity = SeqEvGen.lastEvent();



        // Default: one timestep per day.



    tLinDtGenerator LinDtGen( 1, 1, m_nMaturity );

    tDtGenerator* pDtGen;



    if( m_pDtGen != 0 )

        pDtGen = m_pDtGen;

    else

        pDtGen = &LinDtGen;



    m_gStableDt = (*m_pSpace)[0]->stableDt();

    for( int i = 1; i < m_pSpace->numOfElems(); ++i ) {

        if( (*m_pSpace)[i]->stableDt() < m_gStableDt ) {

            throw tException( INTERNAL_ERROR );

            m_gStableDt = (*m_pSpace)[i]->stableDt();

        }

    }



    m_aSlice.numOfElems( 1 );

    m_aSlice[0].m_nDay = 0;

    m_aSlice[0].m_gFractionOfDay = 0;



    SeqEvGen.rewind();

    while( ( nDay = SeqEvGen.nextEvent() ) >= 0 ) {

        if( nDay > 0 )

            addSlice( nDay, *pDtGen );

    }



    m_bFinalized = true;



#if defined(_DEBUG)

    //for( int k = 0; k < m_aSlice.numOfElems(); ++k ) {

    //    MTG_TRACE( "Slice %d at day %lg\n",

    //        k, m_aSlice[k].m_nDay + m_aSlice[k].m_gFractionOfDay );            

    //}

#endif



    for( MTG_FOR_INIT( int ) i = 0; i < m_pSpace->numOfElems(); ++i )

        (*m_pSpace)[i]->finalize();

}





//

//   a b s D u r a t i o n

//



double tTimeAxis::absDuration( int nSlice ) const



{

    MTG_ASSERT( m_bFinalized );



    if( nSlice == m_aSlice.numOfElems() - 1 )

        return 0;



    return

        m_aSlice[nSlice + 1].m_nDay + m_aSlice[nSlice + 1].m_gFractionOfDay -

        m_aSlice[nSlice].m_nDay - m_aSlice[nSlice].m_gFractionOfDay;

}





//

//   r e l D u r a t i o n

//



double tTimeAxis::relDuration( int nSlice ) const



{

    return absDuration( nSlice ) / stableDt();

}



MTG_END_NAMESPACE

