// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgUSTBond.h"

#include "MtgCouponCashflow.h"

#include "MtgFactor.h"

#include "MtgFDEngine.h"

#include "MtgInterestSpline.h"

#include "MtgParser.h"

#include "MtgUSBondMath.h"



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tUSTBond::init()



{

    m_gPrincipal = -1;

    m_gCoupon = -1;

}





//

//   c o p y F r o m

//



void tUSTBond::copyFrom( const tUSTBond& Bond )



{

    if( &Bond == this )

        return;



    m_gPrincipal = Bond.m_gPrincipal;

    m_gCoupon = Bond.m_gCoupon;

    m_Settlement = Bond.m_Settlement;



    super::copyFrom( Bond );

}





//

//   p a y o f f

//



double tUSTBond::payoff( tEngine& Engine )



{

    return m_gPrincipal + m_gCoupon / 2;

}





//

//   p a r s e P a r a m

//



tRetCode tUSTBond::parseParam( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;

    tDayShift* pShift;

    tParseInfo& I = static_cast<tParseInfo&>( Info );



    switch( Parser.curToken() ) {

        case xTokUpAndOut :

        case xTokDownAndOut :

            return Parser.setError( INVALID_KEYWORD );



        case xTokPrincipal :

            if( m_gPrincipal >= 0 )

                return Parser.setError( "principal already defined" );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.scanNonNegDouble( m_gPrincipal ) ) != OK ) {

                return nRet;

            }

            break;



        case xTokCoupon :

            if( m_gCoupon >= 0 )

                return Parser.setError( "coupon already defined" );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = Parser.scanNonNegDouble( m_gCoupon ) ) != OK ) {

                return nRet;

            }

            break;



        case xTokSettlement :

            if( I.m_bParam1 )

                return Parser.setError( "settlement already defined" );

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = tDayShift::parse( Parser, pShift ) ) != OK ) {

                return nRet;

            }

            if( pShift == 0 )

                return Parser.setError( INVALID_SETTLEMENT );

            m_Settlement = pShift->shift( system().base() );

            delete pShift;

            if( m_Settlement < system().base() )

                return Parser.setError( INVALID_SETTLEMENT );

            I.m_bParam1 = true;

            break;



        default :

            return super::parseParam( Parser, Info );

    }



    return OK;

}





//

//   p a r s e P o s t f i x

//



tRetCode tUSTBond::parsePostfix( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;

    tParseInfo& I = static_cast<tParseInfo&>( Info );



        // assume the prices are clean



    if( ( nRet = super::parsePostfix( Parser, Info ) ) != OK )

        return nRet;



    if( m_gPrincipal < 0 )

        m_gPrincipal = 100;

    if( m_gCoupon < 0 )

        m_gCoupon = 0;



    if( ! I.m_bParam1 )

        m_Settlement = USBondMath.toSettlement( system().base() );



    setSettlement( m_Settlement );



    tCompounder* pCompounder = 

        USBondMath.createCompounder( m_Settlement, maturityDate() );



    tCouponCompounder* p = dynamic_cast<tCouponCompounder*>( pCompounder );

    if( p != 0 )

        modifyMaturity( p->maturity() );



    if( m_Settlement > maturityDate() )

        return Parser.setError( INVALID_SETTLEMENT );



    if( m_gCoupon > 0 ) {

        tDate Start, End;



        pCompounder->getCompoundingPeriod( m_Settlement, Start, End );

        while( End < maturityDate() ) {

            tCouponCashflow *pCC =

                new tCouponCashflow( system(), End, m_gCoupon / 2 );



            regCashflow( pCC );

            pCompounder->getNextCompoundingPeriod( Start, End );

        }



            // Notice that the last coupon belongs to the principal;

            // there is no cashflow entry.

    }



    if( m_gCoupon > 0 ) {

        double gAI = USBondMath.accruedInterest( m_Settlement,

                maturityDate(), m_gCoupon, tIdentityDayShift() );



        setUnitPriceShift( gAI );



        if( isPriced() ) {

            // Now convert the clean prices into dirty prices,

            // by adding accrued interest:



            modifyUnitBidPrice( unitBidPrice() + gAI );

            modifyUnitAskPrice( unitAskPrice() + gAI );

        }

    }



    delete pCompounder;

    return OK;

}





//

//   t U S T B o n d

//



tUSTBond::tUSTBond()



{

    init();

}





//

//   t U S T B o n d

//



tUSTBond::tUSTBond( const tUSTBond& Bond )



{

    copyFrom( Bond );

}





//

//   ~ t U S T B o n d

//



tUSTBond::~tUSTBond()



{

}





//

//   o p e r a t o r =

//



tUSTBond& tUSTBond::operator=( const tUSTBond& Bond )



{

    if( &Bond != this )

        copyFrom( Bond );

    return *this;

}





//

//   c l o n e

//



tObject* tUSTBond::clone() const



{

    return new tUSTBond( *this );

}



    

//

//   r e s o l v e

//



tRetCode tUSTBond::resolve( tLookup<int>& Claim,

    tLookup<int>& Factor, tHeap<tFactor*>& FactorPtr, tPortfolio& Pf )



{

    if( isResolved() )

        return OK;

    return super::resolve( Claim, Factor, FactorPtr, Pf );

}





//

//   a d d P a y m e n t S t r e a m

//



bool tUSTBond::addPaymentStream( tInterestSpline& Spline ) const



{

    if( ! isPriced() )

        return false;



        // assume the prices are dirty



    tHeap<tPayment> Payment;



    for( int i = 0; i < m_Cashflow.numOfElems(); ++i ) {

        tCouponCashflow* p =

            dynamic_cast<tCouponCashflow*>( m_Cashflow[i] );



        MTG_ASSERT( p != 0 );                             



            // notice that here we don't have to divide the coupon

        Payment.append( tPayment( p->date(), p->coupon() ) );

    }



    Payment.append( tPayment( maturityDate(), m_gPrincipal + m_gCoupon / 2 ) );



    Spline.addPayment( m_Settlement, ( unitAskPrice() + unitBidPrice() ) / 2,

        Payment );



    return true;

}





//

//   f i n a l i z e

//



tRetCode tUSTBond::finalize()



{

    if( isFinalized() )

        return OK;

    return super::finalize();

}



MTG_END_NAMESPACE



