// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgYieldCurve.h"

#include "MtgCurveInstance.h"

#include "MtgEngine.h"

#include "MtgForwardCurve.h"

#include "MtgInterestConvention.h"

#include "MtgParser.h"



#include <float.h>



MTG_BEGIN_NAMESPACE





//

//   t Y i e l d C u r v e I n s t a n c e

//



tYieldCurve::tYieldCurveInstance::tYieldCurveInstance(

    const tYieldCurve& Curve, tEngine& Engine )

    : tCurveInstance( Curve, Engine )



{

}





//

//   ~ t Y i e l d C u r v e I n s t a n c e

//



tYieldCurve::tYieldCurveInstance::~tYieldCurveInstance()



{

}





//

//   e v a l u a t e

//



tRetCode tYieldCurve::tYieldCurveInstance::evaluate( double gWeight )



{

    try {

        for( int k = firstSample(); k <= lastSample(); ++k ) {

            double gSub = xlat( k );

            double gPV = m_Engine.presentValueSI( 0, gSub );



            m_Sample[k] += gWeight * gPV;

        }

    } 

    catch( tException( e ) ) {

        return e.ret();

    }



    return OK;

}





//

//   c o n t a i n s E v a l u a t i o n O f

//



bool tYieldCurve::tYieldCurveInstance::containsEvaluationOf(

    const tCurveInstance& Instance ) const



{

    if( &Instance == this )

        return false;



        // various tests... (forward and yield curves have the same

        // evaluation step)



    if( dynamic_cast<const tYieldCurveInstance*>( &Instance ) == 0 &&

        dynamic_cast<const tForwardCurve::tForwardCurveInstance*>( &Instance ) == 0 ) {

        return false;

    }



    if( ! Instance.curve().sameDateBaseAs( m_Curve ) )

        return false;



    if( Instance.curve().numOfSamples() > m_Curve.numOfSamples() )

        return false;



    return true;

}





//

//   p o s t P r o c e s s

//



tRetCode tYieldCurve::tYieldCurveInstance::postProcess( double gWeight )



{

    const tYieldCurve* pYield =

        dynamic_cast<const tYieldCurve*>( &m_Curve );



    if( pYield == 0 )

        throw tException( INTERNAL_ERROR );



    const tInterest& I = pYield->interest();

    const tDateBase& DB = m_Engine.dateBase();



    int nInherit = 0;



    try {

        for( int k = firstSample(); k <= lastSample(); ++k ) {

            double f = DB.getFractionSI( xlat( k ) );



            if( f < DBL_EPSILON ) {

                ++nInherit;

            }

            else {

                if( m_Sample[k] == 0 )

                    m_Sample[k] = DBL_MAX;

                else

                    m_Sample[k] = I.invPresentValue( f, m_Sample[k] );

                for( int i = 1; i <= nInherit; ++i )

                    m_Sample[k - i] = m_Sample[k];

                nInherit = 0;

            }

        }

    } 

    catch( tException( e ) ) {

        return e.ret();

    }



    if( nInherit != 0 )

        return RUNTIME_ERROR;



    return super::postProcess( gWeight );

}





//

//   c o p y F r o m

//



void tYieldCurve::copyFrom( const tYieldCurve& Curve )



{

    if( &Curve == this )

        return;



    m_Interest = Curve.m_Interest;

    super::copyFrom( Curve );

}





//

//   p a r s e P a r a m

//



tRetCode tYieldCurve::parseParam( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;

    tObject* pObj;

    tInterestConvention* pConvention;



    tParseInfo& I = static_cast<tParseInfo&>( Info );



    switch( Parser.curToken() ) {

        case xTokInterestConvention :

            if( I.m_bParam1 )

                return Parser.setError( ATTR_REDEFINITION );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() != xTokId )

                return Parser.setError( "interest convention expected" );

            if( ( pObj = Parser.findObject() ) == 0 )

                return Parser.setError( NOT_FOUND );

            if( ( pConvention =

                    dynamic_cast<tInterestConvention*>( pObj ) ) == 0 ) {

                return Parser.setError( OBJECT_MISMATCH );

            }

            m_Interest = pConvention->interest();

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            I.m_bParam1 = true;

            break;



        default :

            return super::parseParam( Parser, Info );

    }



    return OK;

}





//

//   p a r s e P o s t f i x

//



tRetCode tYieldCurve::parsePostfix( tParser& Parser, tParseInfoStub& Info )



{

    tParseInfo& I = static_cast<tParseInfo&>( Info );



    if( ! I.m_bParam1 ) {

        m_Interest.set( system().dayCount() );

        m_Interest.set( tInterest::xYield );

        m_Interest.set( tInterest::xExponential );

    }



    return super::parsePostfix( Parser, Info );

}





//

//   t Y i e l d C u r v e

//



tYieldCurve::tYieldCurve()



{

}





//

//   t Y i e l d C u r v e

//



tYieldCurve::tYieldCurve( tScale nScale )

    : tCurve( nScale )



{

}





//

//   t Y i e l d C u r v e

//



tYieldCurve::tYieldCurve( tScale nScale, int nNumOfPeriods )

    : tCurve( nScale, nNumOfPeriods )



{

}





//

//   t Y i e l d C u r v e

//



tYieldCurve::tYieldCurve( tScale nScale, const tDayCount& DayCount )

    : tCurve( nScale, DayCount )



{

}





//

//   t Y i e l d C u r v e

//



tYieldCurve::tYieldCurve( tScale nScale, int nNumOfPeriods,

    const tDayCount& DayCount )

    : tCurve( nScale, nNumOfPeriods, DayCount )



{

}





//

//   t Y i e l d C u r v e

//



tYieldCurve::tYieldCurve( tScale nScale, const tDayCount& DayCount,

    tDate Base )

    : tCurve( nScale, DayCount, Base )



{

}





//

//   t Y i e l d C u r v e

//



tYieldCurve::tYieldCurve( tScale nScale, int nNumOfPeriods,

    const tDayCount& DayCount, tDate Base )

    : tCurve( nScale, nNumOfPeriods, DayCount, Base )



{

}





//

//   t Y i e l d C u r v e

//



tYieldCurve::tYieldCurve( const tYieldCurve& Curve )



{

}





//

//   ~ t Y i e l d C u r v e

//



tYieldCurve::~tYieldCurve()



{

}





//

//   o p e r a t o r =

//



tYieldCurve& tYieldCurve::operator=( const tYieldCurve& Curve )



{

    if( this != &Curve )

        copyFrom( Curve );

    return *this;

}





//

//   c l o n e

//



tObject* tYieldCurve::clone() const



{

    return new tYieldCurve( *this );

}





//

//   f i n a l i z e

//



tRetCode tYieldCurve::finalize()



{

    return super::finalize();

}





//

//   c r e a t e I n s t a n c e

//



tRetCode tYieldCurve::createInstance( tEngine& Engine,

    tCurveInstance*& pCurve ) const



{

    pCurve = new tYieldCurveInstance( *this, Engine );

    return OK;

}



MTG_END_NAMESPACE





