(* :Title: MtgMath *)

(* :Context: RB`MtgMath` *)

(* :Author: Robert Buff *)

(* :Summary:
    This package contains functions for plotting
    financial data generated by the Mtg programs.
*)

(* :Copyright: Copyright 1998, Robert Buff *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 3.0 *)

(* :History:
    V1.0 May 1998 by Robert Buff
*)

(* :Keywords:
*)

(* :Sources:
*)

(* :Discussion:
*)

BeginPackage["RB`MtgMath`",
        {"Graphics`Graphics`",
         "Graphics`Colors`",
         "Graphics`Legend`",
         "Statistics`DataManipulation`",
         "Statistics`DescriptiveStatistics`",
         "Utilities`FilterOptions`"}]

EndPackage[]

Install["RB\MtgMath.exe"];

Iff[Context[] != "RB`MtgMath`",
    Needs["Graphics`Graphics`"];
    Needs["Graphics`Colors`"];
    Needs["Graphics`Legend`"];
    Needs["Statistics`DataManipulation`"];
    Needs["Statistics`DescriptiveStatistics`"];
    Needs["Utilities`FilterOptions`"]]

(* Settings *)

IgnoreLastSlice = False;
AlwaysGrayScale = False;

(* Global variables *)

CurrentFile = "";
CurrentLattice = {};
CurrentLayers = {};

(* Helper functions *)

FmtString[num_?NumericQ]:=
    With[{s = ToString[N[num]]},
        If[StringMatchQ[s, "*."],
            StringDrop[s,-1],
            s]]

LookupLayerColor[layer_Integer] :=
    CurrentLayers[[layer, 4]]

LookupComplexityColor[r_?NumericQ, gb_?NumericQ] :=
    If[Head[LookupColor[1]] === GrayLevel,
        GrayLevel[gb],
        RGBColor[r, gb, gb]]

LookupVolatilityColor[r_?NumericQ, b_?NumericQ] :=
    If[Head[LookupColor[1]] === GrayLevel,
        GrayLevel[r],
        RGBColor[r, Max[r, b] / 2, b]]

LookupColor[layer_Integer] :=
    CurrentLayers[[layer, 4]]

LookupWhite[] :=
    If[Head[LookupColor[1]] === GrayLevel,
        GrayLevel[1],
        White]

LookupBlack[] :=
    If[Head[LookupColor[1]] === GrayLevel,
        GrayLevel[0],
        Black]

LookupSpaceCoord[pos_Integer] :=
    Table[
        With[{x = CurrentLattice[[i + 1]]},
            x[[2, 1 + pos - x[[1, 1]]]]],
        {i, 1, Length[CurrentLattice] - 1}]

LookupSpaceCoord[pos_List] :=
    Table[
        With[{x = CurrentLattice[[i + 1]]},
            x[[2, 1 + pos[[i]] - x[[1, 1]]]]],
        {i, 1, Length[CurrentLattice] - 1}]

LookupSpaceDisp[layer_Integer, pos_Integer] :=
    Module[{x},
        x = CurrentLayers[[layer, 3]];
        Table[1 + pos - x[[i, 1]], {i, 1, Length[x] - 1}]]

LookupSpaceDisp[layer_Integer, pos_List] :=
    Module[{x},
        x = CurrentLayers[[layer, 3]];
        Table[1 + pos[[i]] - x[[i, 1]], {i, 1, Length[x] - 1}]]

FindLayer[claim_] :=
    FindLayer[{claim}, 0]

FindLayer[claim_Integer, tag_Integer] :=
    FindLayer[{claim}, tag]

FindLayer[claimList_List, tag_Integer] :=
    Module[{x},
        x = Sort[claimList];
        Catch[
            Scan[(
                If[#1[[2, 1]] == tag && Sort[#1[[2, 2]]] == x,
                    Throw[#1[[1]]]])&,
                CurrentLayers]]]

AssignColorToLayers[] :=
	Module[{n, c},
        n = Length[CurrentLayers];
        l = {Blue, Cyan, Green, Red, Magenta, Yellow,
             Brown, Gold, Orange, Black, Pink, Eggshell,
             Chocolate, DarkSeaGreen, Khaki, Raspberry,
             Tomato, Titanium, Olive, Salmon};
        CurrentLayers =
            If[AlwaysGrayScale || n > Length[l],
                With[{m = 1 / n, c = -1 / n},
                    Table[            
                        Append[
                            Take[CurrentLayers[[i]], 3],
                            GrayLevel[N[m i + c]]],
                        {i, 1, n}]],
                Table[Append[
                        Take[CurrentLayers[[i]], 3],
                        l[[i]]],
                    {i, 1, n}]];]

TrimData[data_List?(Depth[#] == 3&)] :=
    If[IgnoreLastSlice, Map[(Drop[#1, -1])&, data], data]

AssignColorToLayerData[data_List?(Depth[#] == 2&)] :=
    Module[{w},
        w = LookupWhite[];
        Map[(If[# < 0, w, LookupColor[#]])&, data]]

AssignColorToLayerData[data_List?(Depth[#] == 3&)] :=
    Map[(AssignColorToLayerData[#])&, data];

AssignColorToComplexityData[data_List?(Depth[#] == 2&)] :=
    Module[{mgb, cgb, mr, cr, w, o},
        {mgb, mr} = {-1 / CurrentMaxComplexity, 0};
        {cgb, cr} = {1, 1};
        w = LookupWhite[];
        o = LookupComplexityColor[mgb + cgb, mgb + cgb];
        Map[(
            Which[
                # <= 0, w,
                # == 1, o,
                True, LookupComplexityColor[mr # + cr, mgb # + cgb]])&,
            data]]

AssignColorToComplexityData[data_List?(Depth[#] == 3&)] :=
    Map[(AssignColorToComplexityData[#])&, data]

AssignColorToMaskData[data_List?(Depth[#] == 2&)] :=
    Module[{w, b},
        w = LookupWhite[];
        b = LookupBlack[];
        Map[(If[# == 0, b, w])&, data]]

AssignColorToMaskData[data_List?(Depth[#] == 3&)] :=
    Map[(AssignColorToMaskData[#])&, data]

AssignColorToVolatilityData[data_List?(Depth[#] == 2&),
    minVol_?Positive, maxVol_?Positive] :=
    Module[{w, r, b, d, x},
        w = LookupWhite[];
        r = LookupVolatilityColor[1, 0];
        b = LookupVolatilityColor[0, 1];
        d = If[minVol == maxVol, 1, maxVol - minVol];
        Map[(
            Which[
                # < minVol, w,
                # == minVol, b,
                # == maxVol, r,
                True, 
                    x = (# - minVol) / d;
                    LookupVolatilityColor[x, 1 - x]])&,
            data]]

AssignColorToVolatilityData[data_List?(Depth[#] == 3&)] :=
    Module[{v, minVol, maxVol},
        v = Select[Flatten[data], Positive];
        {minVol, maxVol} = {Min[v], Max[v]};
        Map[(AssignColorToVolatilityData[#, minVol, maxVol])&, data]]

GenerateTicks[layer_Integer, data_List?(Depth[#] == 4&)] :=
    {{{0, "today"}, {Length[data[[1]]], "maturity"}},
     {{LookupSpaceDisp[layer, 0][[1]],
       Map[FmtString, LookupSpaceCoord[0]][[1]]}}}

DataToGraphics[layer_Integer, data_List?(Depth[#] == 4&)] :=
    Module[{},
        Graphics[RasterArray[data],
            Axes -> True,
            AxesOrigin -> {-1, -1},
            AxesLabel -> {"time", "stock"},
            Ticks -> GenerateTicks[layer, data]]]

PrintLayerListing[] :=
    Module[{t},
        t = Table[
            {StyleForm["     ",
                Background -> LookupLayerColor[i],
                FontFamily -> "Courier"],
             " Layer ", i, ": ",
             MapIndexed[(If[#2[[1]] > 1, {", ", #1}, #1])&,                
                 CurrentLayers[[i, 2, 2]]],
             " (tag ", CurrentLayers[[i, 2, 1]], ")\n"},
            {i, Length[CurrentLayers]}];
        Print @@ Flatten[t]]

(* Script functions *)

ExtractName[{obj_, name_?AtomQ}, __] := 
    ToString[name]

ExtractName[obj_, def_?AtomQ] :=
    If[Depth[obj] < 2 || ! AtomQ[obj[[1]]] || NumericQ[obj[[1]]],
        ToString[def],
        ToString[obj[[1]]]]

RemoveName[obj_List] :=
    obj

RemoveName[obj_] :=
    If[Depth[obj] >= 2 && AtomQ[obj[[1]]] && ! NumericQ[obj[[1]]],
        Drop[obj, 1],
        obj]

ToScript[{}] :=
    ""

ToScript[{}, sep_String] :=
    ""

ToScript[{obj_?(Depth[#] > 1&), name_?AtomQ}] :=
    ToScript[obj, name]

ToScript[itemList_List, sep_String] :=
    StringJoin[
        Drop[Flatten[Map[({ToScript[#], sep})&, itemList]], -1]]

ToScript[IMPLIED[mat_Integer?Positive, rate_?NumericQ]] :=
    StringJoin["implied ", ToString[mat], " ",
        FmtString[rate / 100]]

ToScript[IMPLIED[mat_Integer?Positive, {rate_?NumericQ}]] :=
    ToScript[IMPLIED[mat, rate]]

ToScript[IMPLIED[mat_Integer?Positive,
    {minRate_?NumericQ, maxRate_?NumericQ}]] :=
    If[minRate <= maxRate,
        StringJoin["implied ", FmtString[mat], " ",
            FmtString[minRate / 100], "..", FmtString[maxRate / 100]],
        ToScript[IMPLIED[mat, {maxRate, minRate}]]]

ToScript[IMPLIED[mat_Integer?Positive,
    {minRate_?NumericQ, priorRate_?NumericQ, maxRate_?NumericQ}]] :=
    Module[{a, b, c},
        {a, b, c } = Sort[{minRate, priorRate, maxRate}] / 100;
        StringJoin["implied ", FmtString[mat], " ",
            FmtString[a], "..", FmtString[b], "..", FmtString[c]]]

ToScript[BAND[rate_?NumericQ]] :=
    StringJoin["band ", FmtString[Abs[rate / 100]]];

ToScript[BAND[Plus, rate_?NumericQ]] :=
    StringJoin["band +", FmtString[Abs[rate / 100]]];

ToScript[BAND[Minus, rate_?NumericQ]] :=
    StringJoin["band -", FmtString[Abs[rate / 100]]];

ToScript[BAND[rate1_?NumericQ, rate2_?NumericQ]] :=
    With[{
        a = Which[
                rate1 > 0, StringJoin["band +", FmtString[Abs[rate1 / 100]]],
                rate1 < 0, StringJoin["band -", FmtString[Abs[rate1 / 100]]],
                True, ""],
        b = Which[
                rate2 > 0, StringJoin["band +", FmtString[Abs[rate2 / 100]]],
                rate2 < 0, StringJoin["band -", FmtString[Abs[rate2 / 100]]],
                True, ""]},
        If[a == "", b, If[b == "", a, StringJoin[a, ", ", b]]]]

TermStructToScript[termStruct_List] :=
    ToScript[termStruct, ", "]        

ToScript[VOL[name_?AtomQ, termStruct___]] :=
    ToScript[VOL[termStruct], name]

ToScript[VOL[termStruct___]] :=
    ToScript[VOL[termStruct], "v"]

ToScript[VOL[termStruct___], name_?AtomQ] :=
    StringJoin["vol ", ToString[name],
        " {", TermStructToScript[Flatten[{termStruct}]], "}"]

ToScript[DRIFT[name_?AtomQ, termStruct___]] :=
    ToScript[DRIFT[termStruct], name]

ToScript[DRIFT[termStruct___]] :=
    ToScript[DRIFT[termStruct], "r"]

ToScript[DRIFT[termStruct___], name_?AtomQ] :=
    StringJoin["drift ", ToString[name],
        " {", TermStructToScript[Flatten[{termStruct}]], "}"]

ToScript[FACTOR[name_?AtomQ]] :=
    ToScript[FACTOR[], name]

ToScript[FACTOR[]] :=
    ToScript[FACTOR[], "s"]

ToScript[FACTOR[], name_?AtomQ] :=
    StringJoin["factor ", ToString[name], " {}"]

ClaimToScript[type_String, mat_Integer, strike_?Positive,
    olist___?OptionQ] :=
    Module[{upBarrier, downBarrier, pay},
        {upBarrier, downBarrier, pay} = 
            {UpAndOut, DownAndOut, Payoff} /. Flatten[{olist}] /.
            {UpAndOut -> None, DownAndOut -> None, Payoff -> Linear};
        StringJoin[Flatten[{
            "type ", ToString[type],
            ", maturity ", FmtString[mat],
            ", strike ", FmtString[strike],
            Which[
                pay === Linear, ", payoff linear",
                pay === Digital, ", payoff digital"],
            If[NumericQ[upBarrier],
                {", up_and_out ", FmtString[upBarrier]}, ""],
            If[NumericQ[downBarrier],
                {", down_and_out ", FmtString[downBarrier]}, ""]}]]]

ClaimToScript[name_String, type_String, mat_Integer, strike_?Positive,
    olist___?OptionQ] :=
    StringJoin["claim",
        If[ToString[name] == "", "", " " <> ToString[name]],
        " {", ClaimToScript[type, mat, strike, olist], "}"]

ToScript[ECALL[mat_Integer, strike_?Positive, olist___?OptionQ]] :=
    ToScript[ECALL[mat, strike, olist], ""]

ToScript[ECALL[name_?AtomQ, mat_Integer, strike_?Positive,
    olist___?OptionQ]] :=
    ToScript[ECALL[mat, strike, olist], name]

ToScript[ECALL[mat_Integer, strike_?Positive, olist___?OptionQ],
    name_?AtomQ] :=
    ClaimToScript[ToString[name], "european_call", mat, strike, olist]

ToScript[ACALL[mat_Integer, strike_?Positive, olist___?OptionQ]] :=
    ToScript[ACALL[mat, strike, olist], ""]

ToScript[ACALL[name_?AtomQ, mat_Integer, strike_?Positive,
    olist___?OptionQ]] :=
    ToScript[ACALL[mat, strike, olist], name]

ToScript[ACALL[mat_Integer, strike_?Positive, olist___?OptionQ],
    name_?AtomQ] :=
    ClaimToScript[ToString[name], "american_call", mat, strike, olist]

ToScript[EPUT[mat_Integer, strike_?Positive, olist___?OptionQ]] :=
    ToScript[EPUT[mat, strike, olist], ""]

ToScript[EPUT[name_?AtomQ, mat_Integer, strike_?Positive,
    olist___?OptionQ]] :=
    ToScript[EPUT[mat, strike, olist], name]

ToScript[EPUT[mat_Integer, strike_?Positive, olist___?OptionQ],
    name_?AtomQ] :=
    ClaimToScript[ToString[name], "european_put", mat, strike, olist]

ToScript[APUT[mat_Integer, strike_?Positive, olist___?OptionQ]] :=
    ToScript[APUT[mat, strike, olist], ""]

ToScript[APUT[name_?AtomQ, mat_Integer, strike_?Positive,
    olist___?OptionQ]] :=
    ToScript[APUT[mat, strike, olist], name]

ToScript[APUT[mat_Integer, strike_?Positive, olist___?OptionQ],
    name_?AtomQ] :=
    ClaimToScript[ToString[name], "american_put", mat, strike, olist]

ToScript[LONG[name_?AtomQ]] :=
    ToScript[LONG[name, 1]]

ToScript[LONG[name_?AtomQ, pos_?NumericQ]] :=
    StringJoin["claim ", ToString[name], " pos ", FmtString[pos]]

ToScript[LONG[claim_]] :=
    ToScript[LONG[claim, 1]]

ToScript[LONG[claim_, pos_?NumericQ]] :=
    StringJoin[ToScript[claim], " pos ", FmtString[pos]]

ToScript[SHORT[name_?AtomQ]] :=
    ToScript[SHORT[name, 1]]

ToScript[SHORT[name_?AtomQ, pos_?NumericQ]] :=
    StringJoin["claim ", ToString[name], " pos ", FmtString[-pos]]

ToScript[SHORT[claim_]] :=
    ToScript[SHORT[claim, 1]]

ToScript[SHORT[claim_, pos_?NumericQ]] :=
    StringJoin[ToScript[claim], " pos ", FmtString[-pos]]

ClaimListToScript[claimList_List] :=
    ToScript[claimList, ", "]        

ToScript[PORTFOLIO[name_?AtomQ, claimList___]] :=
    ToScript[PORTFOLIO[claimList], name]
            
ToScript[PORTFOLIO[claimList___]] :=
    ToScript[PORTFOLIO[claimList], "p"]

ToScript[PORTFOLIO[claimList___], name_?AtomQ] :=
    StringJoin["portfolio ", ToString[name],
        " {", ClaimListToScript[Flatten[{claimList}]], "}"]

ToScript[SCENARIO[WORSTCASE[pos_?AtomQ]]] :=
    ToScript[SCENARIO[WORSTCASE[pos]], "w"]

ToScript[SCENARIO[name_?AtomQ, WORSTCASE[pos_?AtomQ]]] :=
    ToScript[SCENARIO[WORSTCASE[pos]], name]
        
ToScript[SCENARIO[WORSTCASE[SELLER]], name_?AtomQ] :=
    StringJoin["scenario ", ToString[name], " {type worst_case, seller}"]
        
ToScript[SCENARIO[WORSTCASE[BUYER]], name_?AtomQ] :=
    StringJoin["scenario ", ToString[name], " {type worst_case, buyer}"]

ToScript[SCENARIO[name_?AtomQ, SHOCK[pos_?AtomQ], olist___?OptionQ]] :=
    ToScript[SCENARIO[SHOCK[pos], olist], name]

ShockListToScript[olist___?OptionQ] :=
    Module[{d, p, r},
        {d, p, r} =
            {Duration, Periodicity, Repetitions} /. Flatten[{olist}] /.
            {Duration -> 1, Periodicity -> 1, Repetitions -> 1};
        StringJoin[
            ", duration ", FmtString[d],
            ", periodicity ", FmtString[p],
            ", repetitions ", FmtString[r]]]
        
ToScript[SCENARIO[SHOCK[SELLER], olist___?OptionQ], name_?AtomQ] :=
    StringJoin["scenario ", ToString[name],
        " {type shock, seller",
        ShockListToScript[olist],
        "}"]
        
ToScript[SCENARIO[SHOCK[BUYER], olist___?OptionQ], name_?AtomQ] :=
    StringJoin["scenario ", ToString[name],
        " {type shock, buyer",
        ShockListToScript[olist],
        "}"]
    
ToScript[argList___] :=
    Module[{},
        Print["*** not all variables resolved:\n", argList];
        Throw["*** not all variables resolved ***"]]
            
ArgsIntoTempl[templ_List, arg_Rule] :=
    templ /. arg

ArgsIntoTempl[templ_List, name_?AtomQ, obj_] :=
    With[{h = Switch[Head[obj],
                FACTOR, 1,
                VOL, 2,
                DRIFT,
                    Which[
                        ToString[name] == "r", 3,
                        ToString[name] == "q", 4,
                        templ[[3]] === None, 3,
                        True, 4],
                PORTFOLIO, 5,
                SCENARIO, 6,
                _, 7]},
        If[h == 7,
            ReplacePart[templ, Append[templ[[h]], {name, obj}], h],
            ReplacePart[templ, {name, obj}, h]]]

ArgsIntoTempl[templ_List, obj_] :=
    With[{def = Switch[Head[obj],
            FACTOR, "s",
            VOL, "v",
            DRIFT, If[templ[[3]] === None, "r", "q"],
            PORTFOLIO, "p",
            SCENARIO, "w",
            _, "x"]},
        ArgsIntoTempl[templ, ExtractName[obj, def], RemoveName[obj]]]

ArgsToObjList[argList_List] :=
    Module[{templ},
        templ = Fold[ArgsIntoTempl,
                    {None, None, None, None, None, None, {},
                     AssetPrice, DaysPerYear, Shape, Deviations,
                     TimeStep, Accuracy, Method, Profile},
                    Flatten[argList]];
        templ = templ /. {
                    AssetPrice -> 100,
                    DaysPerYear -> 365,
                    Shape -> Tree,
                    Deviations -> 3.5,
                    TimeStep -> 1,
                    Accuracy -> high,
                    Method -> explicit,
                    Profile -> None};
        If[templ[[1]] === None,
            templ[[1]] = {"s", FACTOR[]}];
        If[templ[[2]] === None,
            templ[[2]] = {"v", VOL[IMPLIED[365, 15]]}];
        If[templ[[3]] === None,
            templ[[3]] = {"r", DRIFT[]}];
        If[templ[[4]] === None,
            templ[[4]] = {"q", DRIFT[]}];
        If[templ[[5]] === None,
            templ[[5]] = {"p", PORTFOLIO[]}];
        If[templ[[6]] === None,
            templ[[6]] = {"w", SCENARIO[WORSTCASE[SELLER]]}];
        templ]

MakeScript[f:{_String, _FACTOR}, v:{_String, _VOL}, r:{_String, _DRIFT},
    q:{_String, _DRIFT}, p:{_String, _PORTFOLIO}, s:{_String, _SCENARIO},
    c_List] :=
    ToScript[
        Map[#[[{2,1}]]&, Join[{f, v, r, q}, c, {p, s}]], "\n"]

MakeScript[f:{_String, _FACTOR}, v:{_String, _VOL}, r:{_String, _DRIFT},
    q:{_String, _DRIFT}, d_?Positive, a_?Positive] :=
    StringJoin[
        "\nmodel m {type bs",
        ", vol ", v[[1]],
        ", discount ", r[[1]],
        ", carry ", q[[1]],
        ", days_per_year ", FmtString[d],
        ", ", f[[1]], " ", FmtString[a],
        "}"]

MakeScript[p:{_String, _PORTFOLIO}, h_?AtomQ, n_?AtomQ, t_?Positive, m_?AtomQ] :=
    StringJoin[Flatten[{
        "\nlattice l {model m",
        ", portfolio ", p[[1]],
        ", ", ToLowerCase[ToString[h]],
        If[NumericQ[n], {" ", FmtString[n]}, ""],
        ", time_step ", FmtString[t],
        ", method ", ToString[m], 
        "}"}]]

MakeScript[p:{_String, _PORTFOLIO}, s:{_String, _SCENARIO},
    y_?AtomQ] :=
    StringJoin[
        "\nevaluate {model m, lattice l",
        ", portfolio ", p[[1]],
        ", scenario ", s[[1]],
        ", accuracy ", ToLowerCase[ToString[y]],
        "}"]

ObjListToScript[objList_List?(Length[#] == 15&)] :=
    Module[{f, v, r, q, p, s, c, a, d, h, n, t, y, m, i, j, k,
            profile, script, desc},
        {f, v, r, q, p, s, c, a, d, h, n, t, y, m, profile} = objList;
        a = Flatten[{a}];
        t = Flatten[{t}];
        y = Flatten[{y}];
        script = {MakeScript[f, v, r, q, p, s, c]};
        desc = {};
        For[i = 1, i <= Length[a], ++i,
            script = Append[script,
                MakeScript[f, v, r, q, d, a[[i]]]];
            For[j = 1, j <= Length[t], ++j,
                script = Append[script, MakeScript[p, h, n, t[[j]], m]];
                For[k = 1, k <= Length[y], ++k,
                    desc = Append[desc,
                        {AssetPrice -> a[[i]],
                         TimeStep -> t[[j]],
                         Accuracy -> y[[k]]}];
                    script = Append[script, MakeScript[p, s, y[[k]]]]]]];
        {StringJoin @@ script, desc, profile}]

Script[argList___] :=
    Module[{objList, script, profile},
        objList = ArgsToObjList[{argList}];
        ObjListToScript[objList][[1]]]

RunScript[script_String] :=
    MtgRunScript[script]

RunScript[script_String, profile_String] :=
    MtgRunScript[script, profile]

RunScript[argList___] :=
    Module[{objList, script, desc, profile, result},
        objList = ArgsToObjList[{argList}];
        {script, desc, profile} = ObjListToScript[objList];
        If[profile === None,
            result = RunScript[script],
            result = RunScript[script, profile]];
        If[Depth[result] == 1,
            result,
            Table[Append[result[[i]],desc[[i]]],{i, Length[result]}]]]

MakeScriptFunc[func_, Hold[{param1_?AtomQ}, argList___]] :=
    MakeScriptFunc[func, {param1}, Hold[argList]]

MakeScriptFunc[func_, Hold[{param1_?AtomQ, param2_?AtomQ}, argList___]] :=
    MakeScriptFunc[func, {param1, param2}, Hold[argList]]

MakeScriptFunc[func_, {param1_?AtomQ}, argList___] :=
    With[{body = Flatten[{{argList} /. param1 -> $x$}]},
        Function[$x, func @@ ReleaseHold[body]]]

MakeScriptFunc[func_, {param1_?AtomQ, param2_?AtomQ}, argList___] :=
    With[{body = Flatten[{{argList} /. {param1 -> $x$, param2 -> $y$}}]},
        Function[{$x, $y}, func @@ ReleaseHold[body]]]

MakeScriptFunc[func_, argList___] :=
    Function[a,
        func @@ {argList, AssetPrice -> a}]

MakePriceFunc[argList___] :=
    MakeScriptFunc[PriceFunc, argList]
    
PriceFunc[argList___] :=
    With[{x = RunScript[argList]},
        If[AtomQ[x], x, x[[1, 1, 1]]]]

PriceColumn[result_?AtomQ] :=
    result

PriceColumn[result_List?(Depth[#] == 4&)] :=
    Column[result, 1]

PriceColumn[argList___] :=
    With[{x = RunScript[argList]},
        If[AtomQ[x], x, Column[Column[x, 1], 1]]]

MakeDeltaFunc[argList___] :=
    MakeScriptFunc[DeltaFunc, argList]

DeltaFunc[argList___] :=
    With[{x = RunScript[argList]},
        If[AtomQ[x], x, x[[1, 1, 2]]]]

DeltaColumn[result_?AtomQ] :=
    result

DeltaColumn[result_List?(Depth[#] == 4&)] :=
    Column[result, 2]

DeltaColumn[argList___] :=
    With[{x = RunScript[argList]},
        If[AtomQ[x], x, Column[Column[x, 1], 2]]]

MakeGammaFunc[argList___] :=
    MakeScriptFunc[GammaFunc, argList]

GammaFunc[argList___] :=
    With[{x = RunScript[argList]},
        If[AtomQ[x], x, x[[1, 1, 3]]]]

GammaColumn[result_?AtomQ] :=
    result

GammaColumn[result_List?(Depth[#] == 4&)] :=
    Column[result, 3]

GammaColumn[argList___] :=
    With[{x = RunScript[argList]},
        If[AtomQ[x], x, Column[Column[x, 1], 3]]]

MakeTimeFunc[argList___] :=
    MakeScriptFunc[TimeFunc, argList]

TimeFunc[argList___] :=
    With[{x = RunScript[argList]},
        If[AtomQ[x], x, x[[1, 1, 4]]]]

MakeNumOfTasksFunc[argList___] :=
    MakeScriptFunc[NumOfTasksFunc, argList]

NumOfTasksFunc[argList___] :=
    With[{x = RunScript[argList]},
        If[AtomQ[x], x, x[[1, 1, 5]]]]

MakeGradientFunc[argList___] :=
    MakeScriptFunc[GradientFunc, argList]

GradientFunc[argList___] :=
    With[{x = RunScript[argList]},
        If[AtomQ[x], x, x[[1, 2]]]]

(* Main functions *)

OpenFile[dir_String, fName_String] :=
    Module[{},
        SetDirectory[dir];
        OpenFile[fName]]

OpenFile[fName_String] :=
    Module[{},
        CurrentFile = ToFileName[Directory[],fName];
        CurrentLattice = MtgGetLattice[CurrentFile];
        CurrentLayers = MtgGetLayers[CurrentFile];
        AssignColorToLayers[];
        CurrentMaxComplexity = MtgGetMaxComplexity[CurrentFile];
        Print["File name: ", CurrentFile];
        Print["Number of layers: ",Length[CurrentLayers]];
        Print["Max complexity: ",CurrentMaxComplexity];
        PrintLayerListing[]]

GetComplexityData[layer_Integer] :=
    TrimData[MtgGetComplexity[CurrentFile, layer]]

GetComplexityPlot[layer_Integer] :=
    DataToGraphics[layer,
        AssignColorToComplexityData[GetComplexityData[layer]]]

GetDependencyData[layer_Integer] :=
    TrimData[MtgGetDependency[CurrentFile, layer]]

GetDependencyPlot[layer_Integer]:=
    DataToGraphics[layer,
        AssignColorToLayerData[GetDependencyData[layer]]]

GetBoundaryData[layer_Integer] :=
    TrimData[MtgGetBoundary[CurrentFile, layer]]

GetBoundaryPlot[layer_Integer] :=
    DataToGraphics[layer,
        AssignColorToLayerData[GetBoundaryData[layer]]]

GetMayExPolicyData[layer_Integer] :=
    GetMayExPolicyData[layer, {1}]
	
GetMayExPolicyData[layer_Integer, claim_Integer] :=
    GetMayExPolicyData[layer, {claim}]
	
GetMayExPolicyData[layer_Integer, claimList_List] :=
	TrimData[MtgGetMayExPolicy[CurrentFile, layer, claimList]]

GetMayExPolicyPlot[layer_Integer, claim_Integer] :=
    GetMayExPolicyPlot[layer, {1}]

GetMayExPolicyPlot[layer_Integer, claim_Integer] :=
    GetMayExPolicyPlot[layer, {claim}]

GetMayExPolicyPlot[layer_Integer, claimList_List] :=
    DataToGraphics[layer,
        AssignColorToMaskPlot[GetMayExPolicy[layer, claimList]]]

GetMayExPayoff[layer_Integer] :=
    GetMayExPayoff[layer, {1}]

GetMayExPayoff[layer_Integer, claim_Integer] :=
    GetMayExPayoff[layer, {claim}]

GetMayExPayoff[layer_Integer, claimList_List] :=
	TrimData[MtgGetMayExPayoff[CurrentFile, layer, claimList]]

GetForceExPolicyData[layer_Integer] :=
    GetForceExPolicyData[layer, {1}]
	
GetForceExPolicyData[layer_Integer, claim_Integer] :=
    GetForceExPolicyData[layer, {claim}]
	
GetForceExPolicyData[layer_Integer, claimList_List] :=
	TrimData[MtgGetForceExPolicy[CurrentFile, layer, claimList]]

GetForceExPolicyPlot[layer_Integer, claim_Integer] :=
    GetForceExPolicyPlot[layer, {1}]

GetForceExPolicyPlot[layer_Integer, claim_Integer] :=
    GetForceExPolicyPlot[layer, {claim}]

GetForceExPolicyPlot[layer_Integer, claimList_List] :=
    DataToGraphics[layer,
        AssignColorToMaskPlot[GetForceExPolicy[layer, claimList]]]

GetForceExPayoff[layer_Integer] :=
    GetForceExPayoff[layer, {1}]

GetForceExPayoff[layer_Integer, claim_Integer] :=
    GetForceExPayoff[layer, {claim}]

GetForceExPayoff[layer_Integer, claimList_List] :=
	TrimData[MtgGetForceExPayoff[CurrentFile, layer, claimList]]

GetTotal[layer_Integer] :=
    GetTotal[{layer}]

GetTotal[layer_List] :=
    TrimData[MtgGetTotalList[CurrentFile, layer]]

GetVolatilityData[layer_Integer] :=
    TrimData[MtgGetVolatility[CurrentFile, layer]]

GetVolatilityPlot[layer_Integer] :=
    DataToGraphics[layer,
        AssignColorToVolatilityData[GetVolatilityData[layer]]]

TestDataWithMask[dataOne_List?(Depth[#] >= 3&),
        dataTwo_List?(Depth[#] >= 3&),
        mask_List?(Depth[#] >= 3&), compareF_] :=
    Module[{i,a,b,c},
        a = Flatten[mask];
        b = Flatten[dataOne];
        c = Flatten[dataTwo];
        ! TrueQ[
            Catch[
                For[i = 1, i <= Length[a], ++i,
                    If[a[[i]] != 0,
                        If[! Compare[b[[i]], c[[i]]],
                            Print[{b[[i]], c[[i]]}];
                            Throw[True]]]]]]]

TestHypothesisOne[layer_Integer, largeClaim_Integer, smallClaim_Integer] :=
    Module[{a,b,m},
        a = GetTotal[{FindLayer[largeClaim, 0], -FindLayer[smallClaim, 1]}];
        b = GetMayExPayoff[layer, {largeClaim, -smallClaim}];
        m = GetMayExPolicy[layer, {largeClaim, smallClaim}];
        TestLatticeWithMask[a, b, m, Less]]
