package samples.business;

import javacard.framework.APDU;
import javacard.framework.ISO7816;
import javacard.framework.ISOException;
import javacard.framework.Applet;
import javacard.framework.OwnerPIN;
import javacard.framework.Util;

/**
 * <tt>BizCardApplet</tt> is a demo Java Card applet for
 * the IBM Java Card.
 *
 * It can store a certain number of business card
 * information entries containing the fields title,
 * name, E-mail address, phone and address.
 * The information can be accessed only after a successful
 * card holder verification. As this is a Demo-Applet,
 * the pin is "guess".
 *
 * Note: This applet was tested with the JCOP10 Java Card from
 * IBM, as available in 2001.  For details on this card, please 
 * see http://www.zurich.ibm.com/JavaCard
 *
 * @author  Thomas Schaeck (schaeck@de.ibm.com)
 * @author  Frank Seliger  (seliger@de.ibm.com)
 */
public class BizCardApplet
                       extends javacard.framework.Applet {

  final static byte PIN_TRY_LIMIT = (byte) 10;
  final static byte MAX_PIN_SIZE  = (byte)  8;

  /** A BizCard has two business info entries. */
  private BizCardRecord bizCardRecord0_;
  private BizCardRecord bizCardRecord1_;

  /** The OwnerPin must be matched before this applet
      cooperates */
  private OwnerPIN pin_
           = new OwnerPIN(PIN_TRY_LIMIT, MAX_PIN_SIZE);

/**
 * Create a new BizCardApplet.
 */
protected BizCardApplet(byte[] pin_passed) {
  super();
  bizCardRecord0_ = new BizCardRecord();
  bizCardRecord1_ = new BizCardRecord();
  byte[] pin      = new byte[pin_passed.length];
  Util.arrayCopy(pin_passed, (short)0, pin, (short)0, (short)pin_passed.length);
  /** Set the pin to the value passed during install */
  pin_.update(pin, (short) 0, (byte)pin.length);	
}

/**
 * Actions to be performed it this applet is deselected.
 */
public void deselect() {  // Switching the applet
  pin_.reset();           // .. resets PIN-validated flag
}

/**
 * Gets a field as specified in the given APDU.
 *
 * @param apdu The APDU to be processed.
 */
private void getField(APDU apdu) {
  byte[] buffer = apdu.getBuffer();

  // Check class byte. Only 0x80 is allowed here.
  if (buffer[ISO7816.OFFSET_CLA] != (byte) 0x80)
    ISOException.throwIt(ISO7816.SW_CONDITIONS_NOT_SATISFIED);

  // Check P2. This is the field ID of the field
  // to be read. It must be between 0 and 4.
  if ((buffer[ISO7816.OFFSET_P2]) > 4)
    ISOException.throwIt(ISO7816.SW_INCORRECT_P1P2);

  // Check if the CHV has already been verified.
  if (pin_.isValidated() != true)
    ISOException.throwIt(ISO7816.SW_SECURITY_STATUS_NOT_SATISFIED);

  // Retrieve the information to send outgoing
  BizCardRecord bcRecord = null;

  switch (buffer[ISO7816.OFFSET_P1]) {
    case 0x00 : bcRecord = bizCardRecord0_; break;
    case 0x01 : bcRecord = bizCardRecord1_; break;
    default : // Index is out of bounds
      ISOException.throwIt(ISO7816.SW_INCORRECT_P1P2);
  }
  BizCardField bcField =
    bcRecord.getField(buffer[ISO7816.OFFSET_P2]);

  // Le currently 	
  short le = apdu.setOutgoing();
  apdu.setOutgoingLength(bcField.length);

  // Send the response.
  apdu.sendBytesLong(
    bcField.data, (short) 0, bcField.length);
}

/**
 * Install the applet on the Java Card.
 *
 * @param All parameters are currently ignored.
 *
 */
public static void install(byte[] bArray, short bOffset, byte bLength){
//  BizCardApplet me = new BizCardApplet(bArray);
    
BizCardApplet me = new BizCardApplet( new byte[] {(byte) 0x42, (byte) 0x69, (byte) 0x7A});
  me.register();
}

/**
 * Perform card holder verification as specified
 * in the given APDU.
 *
 * @param apdu The APDU to be processed.
 */
private void performCHV(APDU apdu) {
  byte[] buffer = apdu.getBuffer();
  byte lc = buffer[4];

  // Check class byte. Only 0x80 is allowed here.
  if (buffer[ISO7816.OFFSET_CLA] != (byte) 0x80)
    ISOException.throwIt(ISO7816.SW_CONDITIONS_NOT_SATISFIED);

  // Check P1: Only 0x01 for CHV 1 is allowed here.
  if (buffer[ISO7816.OFFSET_P1] != (byte) 0x01)
    ISOException.throwIt(ISO7816.SW_INCORRECT_P1P2);

  // Check P2. Only 0x00 is allowed here.
  if (buffer[ISO7816.OFFSET_P2] != (byte) 0x00)
    ISOException.throwIt(ISO7816.SW_INCORRECT_P1P2);

  // Check the pin.	
  if (pin_.check(buffer, ISO7816.OFFSET_CDATA, lc) == false)
    ISOException.throwIt(
       ISO7816.SW_SECURITY_STATUS_NOT_SATISFIED);

  // Return successfully, passing no data:
    apdu.setOutgoingAndSend((short) 0, (short) 0);
}

/**
 * Processes incoming APDUs.
 * When a Java Card Applet receives an APDU,
 * it calls this method to process it.
 *
 * @param apdu The APDU to be processed.
 */
public void process(APDU apdu) throws ISOException {
  byte[] apduBuffer = apdu.getBuffer();
  
  // Dispatch commands depending on instruction byte.
  switch ( apduBuffer[ISO7816.OFFSET_INS] ) { 

	case (byte) 0x01 :
	  performCHV(apdu);
	  break;
	case (byte) 0x02 :
	  getField(apdu);
	  break;
	case (byte) 0x03 :
	  setField(apdu);
	  break;
	case (byte) 0xA4 :
	  selectFile(apdu);
	  break;
	default :
	  ISOException.throwIt(ISO7816.SW_INS_NOT_SUPPORTED);
	  break;
   }
}

/**
 * Return the select response when the applet is selected.
 *
 * @param apdu The select APDU that was sent to the card.
 */
private void selectFile(APDU apdu) {
  byte[] buffer = apdu.getBuffer();
  short lc = (short) ((short) 0x00FF & buffer[4]);
  short offset = ISO7816.OFFSET_CDATA;

  // Check class byte. Only 0x00 is allowed here.
  if (buffer[ISO7816.OFFSET_CLA] != (byte) 0x00)
    ISOException.throwIt(ISO7816.SW_CONDITIONS_NOT_SATISFIED);

  // Check P1. Only SELECT by name is supported.
  if (buffer[ISO7816.OFFSET_P1] != (byte) 0x04)
    ISOException.throwIt(ISO7816.SW_INCORRECT_P1P2);

  // Check P2. Only 0x00 (return FCI template) is allowed here.
  if (buffer[ISO7816.OFFSET_P2] != (byte) 0x00)
    ISOException.throwIt(ISO7816.SW_INCORRECT_P1P2);

  // Check AID length.
  if ((lc < ISO7816.OFFSET_CDATA) || (lc > (short) 0x10))
    ISOException.throwIt(ISO7816.SW_WRONG_LENGTH);
  
  // Calculate length to make SELECT processing general
  short proLen
    = (short) 0x00; // 13 bytes proprietary data
  short start
    = (offset -= 4); // to use aid in APDU buffer

  // -> command ok, now prepare response
  buffer[offset++] = (byte) 0x6F;
  buffer[offset++] = (byte) ((byte) 0xFF
                            & (4 + proLen + lc));
  buffer[offset++] = (byte) 0x84;
  buffer[offset++] = (byte) ((byte) 0xFF & (lc));
  offset += lc;
  buffer[offset++] = (byte) 0xA5;
  buffer[offset++] = (byte) ((byte) 0xFF & (proLen));

  // Send the response.
  apdu.setOutgoingAndSend(start, (short) (start + 1
                  + (short) ((byte) 0xFF & buffer[2])));
}

/**
 * Sets a field as specified in the given APDU.
 *
 * @param apdu The SetField Command APDU to be processed.
 */
private void setField(APDU apdu) {
  byte[] buffer = apdu.getBuffer();
  short lc = (short) ((short) 0x00FF & buffer[4]);

  // Check class byte. Only 0x80 is allowed here.
  if (buffer[ISO7816.OFFSET_CLA] != (byte) 0x80)
    ISOException.throwIt(ISO7816.SW_CONDITIONS_NOT_SATISFIED);

  // Check P2. This is the field to be read:
  //  1 <= field <= 5  (field 1 has index 0)    
  if ((buffer[ISO7816.OFFSET_P2]) > 4)
    ISOException.throwIt(ISO7816.SW_INCORRECT_P1P2);

  // Check if the CHV has already been verified.
  if (pin_.isValidated() != true)
    ISOException.throwIt(ISO7816.SW_SECURITY_STATUS_NOT_SATISFIED);

  // Determine which information to update
  BizCardRecord bcRecord = null;

  switch (buffer[ISO7816.OFFSET_P1]) {
    case 0x00 : bcRecord = bizCardRecord0_; break;
    case 0x01 : bcRecord = bizCardRecord1_; break;
    default : // Index is out of bounds
      ISOException.throwIt(ISO7816.SW_INCORRECT_P1P2);
  }

  // Set the field to the value transmitted in the APDU.
  bcRecord.setField(buffer[ISO7816.OFFSET_P2],
                    buffer, ISO7816.OFFSET_CDATA, lc);
  	
  // Return successfully, passing no data:
  apdu.setOutgoingAndSend((short) 0, (short) 0);
}
}
