package samples.business;

import java.util.Hashtable;
import opencard.core.terminal.CommandAPDU;
import opencard.core.terminal.ResponseAPDU;
import opencard.core.terminal.CardTerminalIOControl;
import opencard.core.terminal.CHVControl;
import opencard.core.terminal.CHVEncoder;
import opencard.core.terminal.CardTerminalException;
import opencard.core.service.SmartCard;
import opencard.core.service.CardChannel;
import opencard.core.service.CardServiceInvalidParameterException;
import opencard.core.service.CardServiceInvalidCredentialException;
import opencard.core.service.CardServiceOperationFailedException;
import opencard.core.service.CardServiceException;
import opencard.core.service.CardServiceScheduler;
import opencard.opt.service.CardServiceUnexpectedResponseException;
import opencard.opt.management.ApplicationID;

/**
 * <tt>AnyAppletProxy</tt> is a Card Applet Proxy for any
 * Card Applet for the JavaCard.
 *
 * It offers methods for convenient access to the information
 * securely stored in the CardApplet. Before accessing
 * data, it automatically asks for the password using the OpenCard
 * Framework's card holder verification mechanism.
 *
 * This is a copy of the BusinessCardProxy class - its only purpose
 * in life is to test selection changes and the associated PIN handling.
 * @see samples.business.BizCardApplet
 *
 * @author  Thomas Schaeck (schaeck@de.ibm.com)
 * @author  Frank Seliger  (seliger@de.ibm.com)
 *
 * @version $Id: AnyAppletProxy.java,v 1.2 1998/09/03 12:17:08 cvsusers Exp $
 */
public class AnyAppletProxy extends AppletProxy {
	// Return codes.
	protected final static int OK = 0x9000;
	protected final static int INVALID_PASSWORD = 0x6A80;
	protected final static int INDEX_OUT_OF_RANGE = 0x6A83;
	protected final static int CHV_MISSING = 0x6982;

	// Field identifiers.
	protected final static int NAME = 0;
	protected final static int TITLE = 1;
	protected final static int EMAIL = 2;
	protected final static int PHONE = 3;
	protected final static int ADDRESS = 4;

	/** Reusable command APDU for getting an information entry field. */
	private CommandAPDU getFieldAPDU = new CommandAPDU(14);

	/** Reusable command APDU for setting an information entry field. */
	private CommandAPDU setFieldAPDU = new CommandAPDU(255);

	/** Application identifier of the AnyCardApplet applet */
	private static final ApplicationID ANYCARDAPPLET_AID = new ApplicationID(new byte[] {(byte) 0x42, (byte) 0x43, (byte) 0x61, (byte) 0x72, (byte) 0x64, (byte) 0x00});
/**
 * Gets the address field from entry with the given index.
 *
 * @param index The index of the entry, from which the address shall be obtained.
 *
 * @return The address as a string.
 */
public String getAddress(int index) throws CardServiceException, CardTerminalException {
	return getField(index, ADDRESS);
}
/**
 * Gets the state object of the associated Card Applet Proxy. Or, to be more
 * exact, the state object representing the state of the card applet to
 * which the Card Applet Proxy is associated.
 *
 * @return The state object representing the state of the card
 *         Applet Proxy.
 */
protected AnyAppletState getAnyAppletState(CardChannel channel) {
	return (AnyAppletState) ((Hashtable) channel.getState()).get(ANYCARDAPPLET_AID);
}
/**
 * Gets the E-mail address field from entry with the given index.
 *
 * @param index The index of the entry, from which the E-mail
 *              address shall be obtained.
 *
 * @return The E-mail address as a string.
 */
public String getEmail(int index) throws CardServiceException, CardTerminalException {
	return getField(index, EMAIL);
}
/**
 * Gets a field from entry with the given index.
 *
 * @param index   The index of the entry from which a field shall be obtained.
 * @param filed   The field ID of the field to be obtained.
 * @return The field contents.
 */
public String getField(int index, int field) throws CardServiceInvalidCredentialException, CardServiceOperationFailedException, CardServiceInvalidParameterException, CardServiceUnexpectedResponseException, CardServiceException, CardTerminalException {
	// Class and instruction byte for the get field command.
	final byte[] GET_FIELD_COMMAND_PREFIX = {(byte) 0x80, (byte) 0x02};
	try {
		allocateCardChannel();

		// Perform Card Holder Verification if necessary
		if (!getAnyAppletState(getCardChannel()).isCHVPerformed()) {
			performCHV(getCardChannel(), 1);
		}

		// Set up the command APDU and send it to the card.
		getFieldAPDU.setLength(0);
		getFieldAPDU.append(GET_FIELD_COMMAND_PREFIX); // Class, Instruction
		getFieldAPDU.append((byte) index); // Info Set Index
		getFieldAPDU.append((byte) field); // Field identifier for Name
		getFieldAPDU.append((byte) 0x00); // Lc
		getFieldAPDU.append((byte) 0x00); // Le

		// Send command APDU and check the response.
		ResponseAPDU response = sendCommandAPDU(getCardChannel(), ANYCARDAPPLET_AID, getFieldAPDU);
		switch (response.sw() & 0xFFFF) {
			case OK :
				return new String(response.data());
			case INDEX_OUT_OF_RANGE :
				throw new CardServiceInvalidParameterException("Index out of range");
			default :
				throw new CardServiceUnexpectedResponseException("sw = 0x" + Integer.toHexString((short) (response.sw() & 0xFFFF)));
		}
	} finally {
		releaseCardChannel();
	}
}
/**
 * Gets the name field from entry with the given index.
 *
 * @param index The index of the entry, from which the name shall be obtained.
 *
 * @return The name as a string.
 */
public String getName(int index) throws CardServiceException, CardTerminalException {
	return getField(index, NAME);
}
/**
 * Gets the phone number field from entry with the given index.
 *
 * @param index The index of the entry, from which the phone
 *              number shall be obtained.
 *
 * @return The phone number as a string.
 */
public String getPhone(int index) throws CardServiceException, CardTerminalException {
	return getField(index, PHONE);
}
/**
 * Gets the title field from entry with the given index.
 *
 * @param index The index of the entry, from which the title shall be obtained.
 *
 * @return The title as a string.
 */
public String getTitle(int index) throws CardServiceException, CardTerminalException {
	return getField(index, TITLE);
}
/**
 * Create a <tt>AnyAppletProxy</tt> instance.
 *
 * @param scheduler The Scheduler from which channels have to be obtained.
 * @param card      The SmartCard object to which this service belongs.
 * @param blocking  Currently not used.
 *
 * @throws opencard.core.service.CardServiceException
 *         Thrown when instantiation fails.
 */
protected void initialize(CardServiceScheduler scheduler, SmartCard card, boolean blocking) throws CardServiceException {
	super.initialize(ANYCARDAPPLET_AID, scheduler, card, blocking);
	try {
		// Allocate the card channel. This gives us eclusive access to the
		// card until we release the channel.
		allocateCardChannel();

		// Get the card state. If not already there, create it.
		Hashtable cardState = (Hashtable) getCardChannel().getState();

		// Get the card applet state. If not already there, create it.
		AnyAppletState state = (AnyAppletState) cardState.get(ANYCARDAPPLET_AID);
		if (state == null) {
			state = new AnyAppletState();
			state.setCHVPerformed(false);
			cardState.put(ANYCARDAPPLET_AID, state);
		}
	} finally {
		releaseCardChannel();
	}
}
/**
 * Performs Card Holder Verification.
 *
 * @param channel The card channel to be used.
 * @param numCHV  The number of the CHV to be given to the card.
 */
protected void performCHV(CardChannel channel, int numCHV) throws CardServiceInvalidCredentialException, CardServiceOperationFailedException, CardServiceUnexpectedResponseException, CardServiceException, CardTerminalException {
	// Class and instruction for VerifyCHV command.
	final byte[] DO_CHV_COMMAND_PREFIX = {(byte) 0x80, (byte) 0x01};
	final byte[] PLACEHOLDER = {(byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0};

	// Set up the command APDU.
	CommandAPDU doCHVAPDU = new CommandAPDU(13);
	doCHVAPDU.append(DO_CHV_COMMAND_PREFIX); // Class, Instruction
	doCHVAPDU.append((byte) numCHV); // CHV number
	doCHVAPDU.append((byte) 0x00); // Reserved
	doCHVAPDU.append((byte) 0x08); // Reserved
	doCHVAPDU.append(PLACEHOLDER); // Placeholder for password to be filled in.

	// Let the card terminal ask for the password. If it is unable to do it,
	// OpenCard will use the CHVDialog set in the service.
	CardTerminalIOControl ioctrl = new CardTerminalIOControl(8, 30, null, null);
	CHVControl chvctrl = new CHVControl("Enter your password", numCHV, CHVEncoder.STRING_ENCODING, 0, ioctrl);
	ResponseAPDU response = sendVerifiedAPDU(channel, doCHVAPDU, chvctrl, -1);
	switch (response.sw() & 0xFFFF) {
		case OK :
			// The card applet keeps in mind that we now have performed
			// the CHV. Our state must reflect this.
			getAnyAppletState(getCardChannel()).setCHVPerformed(true);
			return;
		case INVALID_PASSWORD :
			throw new CardServiceInvalidCredentialException("Wrong CHV");
		default :
			throw new CardServiceUnexpectedResponseException("RC = " + response.sw());
	}
}
/**
 * Set the address field in the entry with the given index.
 *
 * @param index   The index of the entry in which
 *                the address shall be set.
 * @param address The new address.
 */
public void setAddress(int index, String address) throws CardServiceException, CardTerminalException {
	setField(index, ADDRESS, address);
}
/**
 * Set the E-mail address field in the entry with the given index.
 *
 * @param index   The index of the entry in which
 *                the E-mail address shall be set.
 * @param address The new E-mail address.
 */
public void setEmail(int index, String email) throws CardServiceException, CardTerminalException {
	setField(index, EMAIL, email);
}
/**
 * Set a field in the entry with the given index.
 *
 * @param index The index of the entry in which a field shall be set.
 * @param field The field ID of the field to be set.
 * @param value The new value.
 */
public void setField(int index, int field, String value) throws CardServiceInvalidCredentialException, CardServiceOperationFailedException, CardServiceInvalidParameterException, CardServiceUnexpectedResponseException, CardTerminalException, CardServiceException {
	// Class and instruction for the SetField Command
	final byte[] SET_NAME_COMMAND_PREFIX = {(byte) 0x80, (byte) 0x03};
	try {
		allocateCardChannel();

		// Perform Card Holder Verification if necessary
		if (!getAnyAppletState(getCardChannel()).isCHVPerformed()) {
			performCHV(getCardChannel(), 1);
		}

		// Set up the command APDU and send it to the card.
		setFieldAPDU.setLength(0);
		setFieldAPDU.append(SET_NAME_COMMAND_PREFIX); // Class, Instruction
		setFieldAPDU.append((byte) index); // Info Set Index
		setFieldAPDU.append((byte) field); // Field identifier
		setFieldAPDU.append((byte) value.length()); // Lc
		setFieldAPDU.append(value.getBytes()); // Data

		ResponseAPDU response = sendCommandAPDU(getCardChannel(), ANYCARDAPPLET_AID, setFieldAPDU);
		switch (response.sw() & 0xFFFF) {
			case OK :
				return;
			case INDEX_OUT_OF_RANGE :
				throw new CardServiceInvalidParameterException("Index out of range");
			default :
				throw new CardServiceUnexpectedResponseException("RC=" + response.sw());
		}
	} finally {
		releaseCardChannel();
	}
}
/**
 * Set the name field in the entry with the given index.
 *
 * @param index   The index of the entry in which
 *                the name shall be set.
 * @param name    The new name.
 */
public void setName(int index, String name) throws CardServiceException, CardTerminalException {
	setField(index, NAME, name);
}
/**
 * Set the phone field in the entry with the given index.
 *
 * @param index The index of the entry in which the phone number shall be set.
 *
 * @param phone The phone number address.
 */
public void setPhone(int index, String phone) throws CardServiceException, CardTerminalException {
	setField(index, PHONE, phone);
}
/**
 * Set the title field in the entry with the given index.
 *
 * @param index   The index of the entry in which the title shall be set.
 * @param address The new title.
 */
public void setTitle(int index, String title) throws CardServiceException, CardTerminalException {
	setField(index, TITLE, title);
}
}
