package samples.business;

import javacard.framework.*;

/**
 * <tt>AnyCardApplet</tt> is a demo JavaCard applet for the
 * IBM JavaCard.
 *
 * It can store a certain number of business card information entries
 * containing the fields title, name, E-mail address, phone and address.
 * The information can be accessed only after a successful card holder
 * verification. As this is a Demo-Applet, the pin is "guess".
 *
 * @author  Thomas Schaeck (schaeck@de.ibm.com)
 * @author  Frank Seliger  (seliger@de.ibm.com)
 *
 */
public class AnyCardApplet extends javacard.framework.Applet {

	/** A AnyCard has two business info entries. */
       private BizCardRecord bizCardRecord0_;
       private BizCardRecord bizCardRecord1_;
	
       /** The OwnerPin must be matched before this applet cooperates */
       private OwnerPIN pin_ = new OwnerPIN((byte)0x10, (byte)0x08);

/**
 * Creates a new AnyCardApplet.
 */
public AnyCardApplet() {
	super();
       bizCardRecord0_ = new BizCardRecord();
       bizCardRecord1_ = new BizCardRecord();

       /** Set the pin to "d...y" */
       byte[] pin = {(byte)'d', (byte)'.', (byte)'.',
               (byte)'.', (byte)'y', (byte) 0, (byte) 0, (byte) 0};
       pin_.update(pin, (short) 0, (byte)pin.length);	
}

/**
 * Actions to be performed it this applet is deselected.
 */
public void deselect() {
       pin_.resetAndUnblock(); // Switching the applet resets PIN-validated flag.
}

/**
 * Gets a field as specified in the given APDU.
 *
 * @param apdu The APDU to be processed.
 */
private void getField(APDU apdu) {
	byte[] buffer = apdu.getBuffer();

	// Check class byte. Only 0x80 is allowed
	if (buffer[ISO.OFFSET_CLA] != (byte) 0x80)
		ISOException.throwIt(ISO.SW_CONDITIONS_NOT_SATISFIED);

	// Check P2. This is the field ID of the field
	// to be read. It must be between 0 and 4.
	if ((buffer[ISO.OFFSET_P2] & 0xFF) > 4)
		ISOException.throwIt(ISO.SW_INCORRECT_P1P2);
	
	// Check if the CHV has already been verified.
	if (pin_.isValidated() != true)
		ISOException.throwIt(ISO.SW_SECURITY_STATUS_NOT_SATISFIED);

       // Retrieve the information to send outgoing
       BizCardRecord bcr = null;

       switch (buffer[ISO.OFFSET_P1] & 0xFF) {
           case 0x00 : bcr = bizCardRecord0_; break;
           case 0x01 : bcr = bizCardRecord1_; break;
           default : // Index is out of bounds
		ISOException.throwIt(ISO.SW_INCORRECT_P1P2);
       }

	// Le currently 	
	short le = apdu.setOutgoing();
       apdu.setOutgoingLength(le);
	
	// Send the response.
	apdu.sendBytesLong(
               bcr.getField((byte)(buffer[ISO.OFFSET_P2] & 0xFF)).data,
               (short) 0,
	        bcr.getField((byte)(buffer[ISO.OFFSET_P2] & 0xFF)).length);
}
/**
 * Install the applet on the JavaCard.
 *
 * @param apdu Currently ignored.
 */
public static void install(APDU apdu) throws ISOException {
	AnyCardApplet me = new AnyCardApplet();
	me.register();
}
/**
 * Perform card holder verification as specified in the given APDU.
 *
 * @param apdu The APDU to be processed.
 */
private void performCHV(APDU apdu) {
	byte[] buffer = apdu.getBuffer();
	byte lc = buffer[4];

	// Check class byte. Only 0x80 is allowed.
	if (buffer[ISO.OFFSET_CLA] != (byte) 0x80)
		ISOException.throwIt(ISO.SW_CONDITIONS_NOT_SATISFIED);

	// Check P1: Only 0x01 for CHV 1 is allowed.
	if (buffer[ISO.OFFSET_P1] != (byte) 0x01)
		ISOException.throwIt(ISO.SW_INCORRECT_P1P2);

	// Check P2. Only 0x00 is allowed.
	if (buffer[ISO.OFFSET_P2] != (byte) 0x00)
		ISOException.throwIt(ISO.SW_INCORRECT_P1P2);

	// Check the pin.	
       if (pin_.check(buffer, (short)5, lc) == false)
       	ISOException.throwIt(ISO.SW_SECURITY_STATUS_NOT_SATISFIED);

	// Send the response: No data, return code 0x9000
	apdu.setOutgoingAndSend((short) 0, (short) 0);
}

/**
 * Processes incoming APDUs. When a JavaCard Applet receives an APDU,
 * it calls this method to process it.
 *
 * @param apdu The APDU to be processed.
 */
public void process(APDU apdu) throws ISOException {
	byte[] apduBuffer = apdu.getBuffer();

	// Check Class
	if (!(((apduBuffer[javacard.framework.ISO.OFFSET_CLA] & (byte) 0x00) == (byte) 0x00) | ((apduBuffer[javacard.framework.ISO.OFFSET_CLA] & (byte) 0x80) == (byte) 0x80)))
		ISOException.throwIt(ISO.SW_CLA_NOT_SUPPORTED);

	// Dispatch commands depending on class and instruction bytes.
	switch (Util.makeShort((byte) (apduBuffer[javacard.framework.ISO.OFFSET_CLA] & (byte) 0xF0), apduBuffer[javacard.framework.ISO.OFFSET_INS])) {
		case (short) 0x8001 :
			performCHV(apdu);
			break;
		case (short) 0x8002 :
			getField(apdu);
			break;
		case (short) 0x8003 :
			setField(apdu);
			break;
		case (short) 0x00A4 :
			selectFile(apdu);
			break;
		default :
			ISOException.throwIt(ISO.SW_INS_NOT_SUPPORTED);
			break;
	}
}

/**
 * Return the select response when the applet is selected.
 *
 * @param apdu The select APDU that was sent to the card.
 */
private void selectFile(APDU apdu) {
	byte[] buffer = apdu.getBuffer();
	short lc = (short) ((short) 0x00FF & buffer[4]);
	short offset = (short) 5;

	// Check class byte. Only 0x00 is allowed
	if (buffer[ISO.OFFSET_CLA] != (byte) 0x00)
		ISOException.throwIt(ISO.SW_CONDITIONS_NOT_SATISFIED);

	// Check P1. Only SELECT by name is supported.
	if (buffer[ISO.OFFSET_P1] != (byte) 0x04)
		ISOException.throwIt(ISO.SW_INCORRECT_P1P2);

	// Check P2. Only 0x00 (return FCI template) is allowed.
	if (buffer[ISO.OFFSET_P2] != (byte) 0x00)
		ISOException.throwIt(ISO.SW_INCORRECT_P1P2);

	// Check AID length.
	if ((lc < (short) 5) || (lc > (short) 0x10))
		ISOException.throwIt(ISO.SW_WRONG_LENGTH);

	// Compare AIDs.	
	if (!(javacard.framework.System.getAID().equals(buffer, offset, buffer[ISO.OFFSET_LC])))
		ISOException.throwIt(ISO.SW_WRONG_DATA);

	// Calculate length to make SELECT processing more general
	short proLen = (short) 0x00; // 13 bytes proprietary data
	short start = (offset -= 4); // to use aid in APDU buffer

	// -> command ok, now prepare response
	buffer[offset++] = (byte) 0x6F;
	buffer[offset++] = (byte) ((byte) 0xFF & (4 + proLen + lc));
	buffer[offset++] = (byte) 0x84;
	buffer[offset++] = (byte) ((byte) 0xFF & (lc));
	offset += lc;
	buffer[offset++] = (byte) 0xA5;
	buffer[offset++] = (byte) ((byte) 0xFF & (proLen));

	// Send the response.
	apdu.setOutgoingAndSend(start, (short) (start + 1 + (short) ((byte) 0xFF & buffer[2])));
}

/**
 * Sets a field as specified in the given APDU.
 *
 * @param apdu The SetField Command APDU to be processed.
 */
private void setField(APDU apdu) {
	byte[] buffer = apdu.getBuffer();
	short lc = (short) ((short) 0x00FF & buffer[4]);

	// Check class byte. Only 0x80 is allowed.
	if (buffer[ISO.OFFSET_CLA] != (byte) 0x80)
		ISOException.throwIt(ISO.SW_CONDITIONS_NOT_SATISFIED);

	// Check P2. This is the field to be read: 1 <= field <= 5
	if ((buffer[ISO.OFFSET_P2] & 0xFF) > 4)
		ISOException.throwIt(ISO.SW_INCORRECT_P1P2);

	// Check if the CHV has already been verified.
	if (pin_.isValidated() != true)
		ISOException.throwIt(ISO.SW_SECURITY_STATUS_NOT_SATISFIED);

       // Determine which information to update
       BizCardRecord bcr = null;

       switch (buffer[ISO.OFFSET_P1] & 0xFF) {
           case 0x00 : bcr = bizCardRecord0_; break;
           case 0x01 : bcr = bizCardRecord1_; break;
           default : // Index is out of bounds
		ISOException.throwIt(ISO.SW_INCORRECT_P1P2);
       }
	
       // Set the field to the value transmitted in the APDU.
	bcr.setField((byte)(buffer[ISO.OFFSET_P2] & 0xFF), buffer, (short)5, lc);
		
	// Send the response: No data, status word 0x9000.
	apdu.setOutgoingAndSend((short) 0, (short) 0);
}
}
