package samples.business;

import opencard.core.service.SmartCard;
import opencard.core.service.CardServiceException;
import opencard.core.terminal.CardTerminalException;
import opencard.core.terminal.ResponseAPDU;
import opencard.core.service.CardServiceScheduler;
import opencard.core.service.CardChannel;
import opencard.core.terminal.CommandAPDU;
import opencard.core.terminal.CHVControl;
import opencard.opt.management.ApplicationID;

/**
 * Base class for Open Platform Applet proxies.
 *
 * @author Thomas Schaeck (schaeck@de.ibm.com)
 *
 * @version $Id: AppletProxy.java,v 1.2 1998/09/03 12:17:08 cvsusers Exp $
 */
public abstract class AppletProxy extends CardExecutiveCardService
{
  /** Application Identifier of the applet to which this proxy belongs */
  private ApplicationID appletAID_ = null;
  /**
   * Gets the application identifier of the applet to which this proxy belongs.
   *
   * @return The application identifier of the associated applet.
   */
  public ApplicationID getAppletAID()
  {
	return appletAID_;
  }
  /**
   * Instantiate an <tt>AppletProxy</tt> and associates it with the card applet
   * with the given application identifier.
   *
   * @param appletAID The application identifier of the applet to be associated
   *                  with the new <tt>AppletProxy</tt>.
   * @param scheduler The scheduler of this <tt>CardExecutiveCardService</tt>.
   * @param card      The controlling </tt>SmartCard</tt> object.
   * @param blocking  Specify the wait behavior for obtaining a
   *                  <tt>CardChannel</tt> from the
   *                  <tt>CardServiceScheduler</tt>.
   */
  protected void initialize(ApplicationID appletAID, CardServiceScheduler scheduler,
					        SmartCard card, boolean blocking)
  throws CardServiceException
  {
	super.initialize(scheduler, card, blocking);
	appletAID_ = appletAID;
  }
  /**
   * Sends a command APDU using the given channel.
   *
   * @param channel      The <tt>CardChannel</tt> to be used for sending the
   *                     command APDU to the JavaCard.
   * @param commandAPDU  The <tt>CommandAPDU</tt> to be sent.
   *
   * @return The cards response to the command APDU sent.
   */
  public ResponseAPDU sendCommandAPDU(CardChannel channel, CommandAPDU commandAPDU)
  throws CardTerminalException, CardServiceException
  {
	return sendCommandAPDU(channel, appletAID_, commandAPDU);
  }
  /**
   * Sends an APDU to the applet to which this proxy belongs.
   *
   * @param commandAPDU The <tt>CommandAPDU</tt> to be sent to the applet
   *
   * @return The <tt>ResponseAPDU</tt> returned by the applet.
   */
  public ResponseAPDU sendCommandAPDU(CommandAPDU commandAPDU)
  throws CardTerminalException, CardServiceException
  {
	return sendCommandAPDU(appletAID_, commandAPDU);
  }
  /**
   * Send a verify CHV command APDU to the JavaCard after filling in the
   * password obtained from the CHV dialog currently associated with this
   * card service.
   *
   * @param channel          The <tt>CardChannel</tt> to be used for sending the
   *                         command APDU.
   * @param verificationAPDU The command APDU for password verification into which the
   *                         password shall be inserted.
   * @param chvControl       The CHV control to be used for password input.
   * @param timeout          The timeout to be used.
   *
   * @return The response APDU returned by the JavaCard as response to the
   *         verify password command.
   */
  public ResponseAPDU sendVerifiedAPDU(CardChannel channel,
			   CommandAPDU verificationAPDU,
                       CHVControl chvControl, int timeout)
throws CardServiceException, CardTerminalException
{
  return sendVerifiedAPDU(channel, appletAID_, verificationAPDU, chvControl, timeout);
}
}
