package samples.business;

import opencard.core.event.CTListener;
import opencard.core.event.CardTerminalEvent;
import opencard.core.service.SmartCard;
import opencard.core.service.CardService;
import opencard.core.terminal.CardTerminalRegistry;
import opencard.core.terminal.CardTerminalException;

/**
 * BusinessCard represents a JavaCard with a BusinessCardApplet to
 * applications. A BusinessCard object offers methods to get and
 * set BusinessInfo entries on the card.
 *
 * @author  Thomas Schaeck (schaeck@de.ibm.com)
 *
 * @version $Id: BusinessCard.java,v 1.1 1998/09/02 16:12:08 cvsusers Exp $
 */
public class BusinessCard implements CTListener
{
  /** The object representing the Smart Card and OCF. */
  SmartCard card_ = null;

  /** The Applet Proxy for the BusinessCardApplet. */
  BusinessCardProxy businessCardProxy_ = null;
/**
 * Creates a new BusinessCard object.
 */
BusinessCard() throws BusinessCardException {
	try {
		// Start the OpenCard Framework. For historic reasons, this
		// must be done by calling a static method of the class
		// SmartCard.
		SmartCard.start();

		// Register the new SignatureCard as a Card Terminal Event Listener
		CardTerminalRegistry.getRegistry().addCTListener(this);

		// Let the registry create events for cards which are already present
		CardTerminalRegistry.getRegistry().createEventsForPresentCards(this);
	} catch (Exception e) {
		// Do exception handling
	}
}
/**
 * React on card insertion events sent by OCF:
 * Get new card and card service
 *
 * @param ctEvent The card insertion event.
 */
public void cardInserted(CardTerminalEvent ctEvent) {
	try {
		// Get a SmartCard object
		card_ = SmartCard.getSmartCard(ctEvent);

		// Get a Proxy instance for the BusinessCardProxy.
		businessCardProxy_ = (BusinessCardProxy) card_.getCardService(BusinessCardProxy.class, true);
	} catch (Exception e) {
		e.printStackTrace();
	}
}
/**
 * React on card removal events sent by OCF:
 * Invalidate card and card service.
 *
 * @param ctEvent The card removal event.
 */
public synchronized void cardRemoved(CardTerminalEvent ctEvent) {
	card_ = null;
	businessCardProxy_ = null;
}
/**
 * Close a <tt>BusinessCard</tt> object.
 */
public void close() throws CardTerminalException {
	// Shut down the OpenCard Framework. For historic reasons,
	// this is done by calling a static method of the class
	// SmartCard.
	SmartCard.shutdown();
}
  /**
   * Gets a <tt>BusinessInfo</tt> object for the entry with the given index.
   *
   * @param index The index of the entry for which a <tt>BusinessInfo</tt>
   *              object shall be obtained.
   *
   * @exception samples.business.BusinessCardException
   *            Thrown when problems occurr while reading the info
   *            from card.
   */
  public BusinessInfo getBusinessInfo(int index) throws BusinessCardException {
	if (card_ == null)
	   throw new BusinessCardException("No card found.");
	try {
		// We need mutual exclusion here to prevent other Card Services from
		// modifying data on the card while we read the information fields.
		card_.beginMutex();

		// Read the fields using an Card Applet Proxy.
		return new BusinessInfo(businessCardProxy_.getName(index), businessCardProxy_.getTitle(index), businessCardProxy_.getEmail(index),
					   businessCardProxy_.getPhone(index), businessCardProxy_.getAddress(index));
	} catch (Exception e) {
			   e.printStackTrace();
		throw new BusinessCardException(e.getMessage());
	} finally {
	  // End mutual exclusion. From now on other Card Services may
	  // access the card again.
		card_.endMutex();
	}
  }  
  /**
   * Sets the entry with the given index to the given
   * <tt>BusinessInfo</tt> object. 
   *
   * @param index        The index of the entry for which a
   *                     <tt>BusinessInfo</tt> object shall be written.
   * @param businessInfo The <tt>BusinessInfo</tt> object to be written.
   *
   * @see samples.business.BusinessInfo
   *
   * @exception samples.business.BusinessCardException
   *            Thrown when problems occurr while writing the info
   *            to  card.
   */
  public void setBusinessInfo(int index, BusinessInfo businessInfo) throws BusinessCardException {
	// Check for card presence.
	if (card_ == null)
		throw new BusinessCardException("No card found.");
		
	try {
		// We need mutual exclusion here to prevent other CardServices from
		// modifying data on the card while we write the information fields.
		card_.beginMutex();

		// Write the fields to the card using an AppletProxy.
		businessCardProxy_.setName(index, businessInfo.getName());
		businessCardProxy_.setTitle(index, businessInfo.getTitle());
		businessCardProxy_.setEmail(index, businessInfo.getEmail());
		businessCardProxy_.setPhone(index, businessInfo.getPhone());
		businessCardProxy_.setAddress(index, businessInfo.getAddress());
	} catch (Exception e) {
		e.printStackTrace();
		throw new BusinessCardException(e.getMessage());
	} finally {
		// End mutual exclusion. From now on other CardServices may access
		// the card again.
		card_.endMutex();
	}
  }  
}