package samples.business;

import java.util.Hashtable;

import opencard.core.service.CardChannel;
import opencard.core.service.CardServiceScheduler;
import opencard.core.service.CardServiceException;
import opencard.core.service.CardServiceOperationFailedException;
import opencard.core.service.CardService;
import opencard.core.service.CardServiceScheduler;
import opencard.core.service.SmartCard;
import opencard.core.terminal.CardTerminalException;
import opencard.core.terminal.CardID;
import opencard.core.terminal.CommandAPDU;
import opencard.core.terminal.ResponseAPDU;
import opencard.core.terminal.CHVControl;
import opencard.core.service.CHVDialog;
import opencard.core.terminal.CHVControl;
import opencard.core.util.Tracer;
import opencard.opt.management.ApplicationID;
import opencard.opt.service.CardServiceUnexpectedResponseException;
import opencard.opt.javacard.JavaCardCardService;

import opencard.opt.util.TLV;
import opencard.opt.util.Tag;

/**
 * This service offers a method to applications or proxies derived from it,
 * which sends a given command APDU to the card applet identified by the given
 * application identifier and returns the result APDU. All instances associated
 * with the same physical card share a common state object attached to the the
 * used channel by the first ExecutiveCardService instance. This state object -
 * an instance of CardExecutiveState keeps track of the currently selected
 * applet and is used by Card Executive services to avoid unnecessary selection
 * of applets.
 *
 * @author  Thomas Schaeck (schaeck@de.ibm.com)
 *
 * @version $Id: CardExecutiveCardService.java,v 1.2 1998/09/03 12:17:08 cvsusers Exp $
 *
 * @see CardServiceScheduler
 * @see CardChannel
 */
public class CardExecutiveCardService
             extends CardService
             implements JavaCardCardService
{
  /** The CardExecutive keeps track in an own state object
      of who it selected last. */
  private CardExecutiveState state_ = null;

  private Tracer itracer        = new Tracer(this, CardExecutiveCardService.class);
  private static Tracer ctracer = new Tracer(CardExecutiveCardService.class);

  private CommandAPDU selectAPDU
    = new CommandAPDU(new byte[22]);

  /** Maximum APDU size allowed by card */
  protected final static int MAX_APDU_SIZE = 262;

  /** This is a pseudo AID for the CardExecutive.
      Normal applets can not have the same AID
      accidentially, because real AIDs consist of
      at least 5 bytes. */
  private final static ApplicationID CARD_EXECUTIVE_AID
    = new ApplicationID("EXEC".getBytes());

/**
  * Assures that a CardExecutiveState object is available
  * in the channel through which this service
  * is communicating with the card.  If it is not there yet,
  * create one.
  */
protected void assureCardExecutiveStateExists(CardChannel channel) {

  Hashtable cardState
    = (Hashtable) channel.getState();
  if (cardState == null) {
    cardState = new Hashtable();
    channel.setState(cardState);
  }
  state_ = (CardExecutiveState) cardState.get(
                                      CARD_EXECUTIVE_AID);
  if (state_ == null) {
    cardState.put(CARD_EXECUTIVE_AID,
                  new CardExecutiveState());
  }
}

/**
 * Gets the associated state object. We store the CardExecutive state
 * in the very same way as the state of an applet proxy or applet.
 * This has the advantage that the state is guaranteed to exist
 * only once per card channel and thus per card.
 *
 * @return The state object representing the state of CardExecutive
 */
protected CardExecutiveState getCardExecutiveState(CardChannel channel) {
  if (state_ == null)  {
    assureCardExecutiveStateExists(channel);
  }
  return state_;
}

  /**
   * Instantiates a <tt>CardExecutiveCardService</tt> and tie it both to its
   * scheduler and its using <tt>SmartCard</tt> object.
   *
   * @param scheduler The scheduler of this <tt>CardExecutiveCardService</tt>.
   * @param card      The controlling </tt>SmartCard</tt> object.
   * @param blocking  Specify the wait behavior for obtaining a
   *                  <tt>CardChannel</tt> from the <tt>CardServiceScheduler</tt>.
   */
  protected void initialize(CardServiceScheduler scheduler,
				            SmartCard card, boolean blocking)
  throws CardServiceException
  {
    super.initialize (scheduler, card, blocking);
    ctracer.debug("<init>", "(" + scheduler + "," + card + "," + blocking + ")");
    try {
      allocateCardChannel();
      assureCardExecutiveStateExists(getCardChannel());
    } finally {
      releaseCardChannel();
    }
  }

/**
 * Selects the JavaCard applet with the given application ID using the given
 * channel for communication with the card.
 *
 * @param channel   The <tt>CardChannel</tt> to be used for sending the select
 *                  command to the JavaCard.
 * @param appletAID The application identifier of the applet to be selected.
 *
 * @return The response APDU returned by the select command. Usually, it
 *         contains information about the selected applet.
 */
protected ResponseAPDU selectApplet(CardChannel channel,
                                    ApplicationID appletAID)
     throws CardServiceException, CardTerminalException {

  final byte[] SELECT_HEADER
    = {(byte)0x00, (byte)0xA4, (byte)0x04, (byte)0x00};
  CardExecutiveState state
    = getCardExecutiveState(getCardChannel());

  if ((state.getSelectedAppletAID() == null)
     || (!state.getSelectedAppletAID().equals(appletAID)))
  {
    itracer.debug("selectApplet", "selecting " + appletAID);
    byte[] appletAIDBytes = appletAID.getAID();

    // Build the SELECT command APDU
    selectAPDU.setLength(0);
    selectAPDU.append(SELECT_HEADER); // Class, Instruction, P1, P2
    selectAPDU.append((byte) appletAIDBytes.length); // Lc
    selectAPDU.append(appletAIDBytes); // Application Identifier
    selectAPDU.append((byte) 0x00); // Le

    ResponseAPDU responseAPDU
      = getCardChannel().sendCommandAPDU(selectAPDU);
    itracer.debug("selectApplet", "Selection response sw = 0x" + Integer.toHexString((short) (responseAPDU.sw() & 0xFFFF)));

    switch ((short) (responseAPDU.sw() & 0xFFFF)) {
      case (short) 0x9000 :
        ApplicationID previouslySelectedAID
          = state.setSelectedAppletAID(appletAID);

    	 // Notify the state associated with AID prev...
        if (previouslySelectedAID != null) {
          Hashtable cardState = (Hashtable) getCardChannel().getState();
          itracer.debug("selectApplet", "previouslySelectedAID = " + previouslySelectedAID);
 	   itracer.debug("selectApplet", "cardState = " + cardState);
          AppletState appState
            = (AppletState)(cardState.get(
                                 previouslySelectedAID));
   	   appState.appletDeselected();
        }
        break;
      default :
    	 throw new CardServiceUnexpectedResponseException(
          "Status Words are 0x" + Integer.toHexString(
                   (short) (responseAPDU.sw() & 0xFFFF)));
    }
    return responseAPDU;
  } else { // Was already selected, we did nothing.
    return null;
  }
}

/**
 * Sends a <tt>CommandAPDU</tt> to the smart card.
 *
 * @param     channel      channel to be used for sending APDUs to the smart card
 * @param     appletAID    application identifier of destination applet
 * @param     commandAPDU  <tt>CommandAPDU</tt> to send
 *
 * @return    The resulting <tt>ResponseAPDU</tt> as
 *            received from the card.
 */
public ResponseAPDU sendCommandAPDU(CardChannel channel,
                ApplicationID appletAID, CommandAPDU commandAPDU)
     throws CardTerminalException, CardServiceException {

  itracer.debug("sendCommandAPDU(channel,...)", "sending " + commandAPDU + " to " + appletAID);
  selectApplet(channel, appletAID);
  return channel.sendCommandAPDU(commandAPDU);
}

/**
 * Sends a <tt>CommandAPDU</tt> to the smart card.
 *
 * @param appletAID    application identifier of destination applet
 * @param commandAPDU  <tt>CommandAPDU</tt> to send
 *
 * @return The resulting <tt>ResponseAPDU</tt> as received from the card.
 */
public ResponseAPDU sendCommandAPDU(ApplicationID appletAID, CommandAPDU commandAPDU) throws CardTerminalException, CardServiceException {
  itracer.debug("sendCommandAPDU(...)", "sending " + commandAPDU + " to " + appletAID);
  try {
     allocateCardChannel();
     return sendCommandAPDU(getCardChannel(), appletAID, commandAPDU);
  } finally {
     releaseCardChannel();
  }
}

/**
 * Send a verify CHV command APDU to the JavaCard applet with the given
 * application identifier after filling in the password obtained from the
 * CHV dialog currently associated with this card service.
 *
 * @param channel          The <tt>CardChannel</tt> to be used for sending the
 *                         command APDU.
 * @param appletID         The application identifier of the applet to which the
 *                         verification APDU shall be sent.
 * @param verificationAPDU The command APDU for password verification into which
 *                         the password shall be inserted.
 * @param chvControl       The CHV control to be used for password input.
 * @param timeout          The timeout to be used.
 *
 * @return The response APDU returned by the JavaCard as response to the
 *         verify password command.
 */
public ResponseAPDU sendVerifiedAPDU(CardChannel channel, ApplicationID appletID,
	                                 CommandAPDU verificationAPDU,
	                                 CHVControl chvControl, int timeout)
throws CardServiceException, CardTerminalException {
  // If necessary, select the applet first
  ResponseAPDU selectResponse = selectApplet(channel, appletID);
  return channel.sendVerifiedAPDU(verificationAPDU, chvControl, getCHVDialog(), -1);
}
};