package samples.stockbroker;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.security.InvalidKeyException;

import opencard.core.OpenCardException;
import opencard.core.service.SmartCard;
import opencard.core.service.CardRequest;
import opencard.core.service.CardServiceException;
import opencard.core.service.CardRequest;
import opencard.core.event.CardTerminalEvent;
import opencard.core.event.CTListener;
import opencard.core.terminal.CardTerminalException;
import opencard.core.terminal.CardTerminalRegistry;
import opencard.core.terminal.Slot;
import opencard.core.util.Tracer;

import opencard.opt.iso.fs.CardFile;
import opencard.opt.iso.fs.CardFilePath;
import opencard.opt.iso.fs.FileAccessCardService;
import opencard.opt.iso.fs.CardFileOutputStream;
import opencard.opt.iso.fs.CardFileInputStream;
import opencard.opt.security.PrivateKeyFile;

import com.ibm.opencard.script.MFCScriptService;
import com.ibm.opencard.buffer.TLVBuffer;
import opencard.opt.signature.JCAStandardNames;
import opencard.opt.signature.SignatureCardService;

import samples.stockbroker.SBCHVDialog;
import samples.stockbroker.SignatureCardException;

/****************************************************************************
* This class encapsulates all card related functionality required by the
* InternetBroker Demo. Only this class uses the OpenCardFramework.
*
* This is a version of SignatureCard.java, which does only read and write
* the Card Holder Data. We have created this subset of the original
* demo code because it can be used with cards not capable of public key
* cryptography.  In addition, this subset it is easier to understand.
*
* In case you admire the artifical linebreaks in the code  -  they are
* needed to make the code fit within the margins for printing in the book.
*
* For "the real InternetBroker demo", which requires an IBM MFC 4.0 card,
* please see the free OpenCard download on www.opencard.org.
*
* @author Thomas Schaeck (schaeck@de.ibm.com)
* @author Frank Seliger  (seliger@de.ibm.com)
*****************************************************************************/
public class SignatureCard implements CTListener
{
  /** The smart card object used to get access to a smart card */
  private SmartCard card = null;
  /** The file access card service used to access files on the card */
  private FileAccessCardService fileService = null;
  /** The signature service used for signature generation and key import */
  private SignatureCardService signatureService = null;
  /** The slot where the card is currently inserted in */
  private Slot currentSlot = null;
  /** The last exception not yet handled */
  private Exception earlierException = null;
  /** The tracer object for tracing a la OpenCard */
  private Tracer iTracer
                   = new Tracer(this, SignatureCard.class);

  /****************************************************************************
  * Construct a SignatureCard object and make sure that OpenCard is being
  * initialized properly.
  ****************************************************************************/
  SignatureCard ()  throws SignatureCardException
  {
    try {
      // Start the OpenCard Framework
      SmartCard.start ();
      // Register the new SignatureCard as a Card Terminal Event Listener
      CardTerminalRegistry.getRegistry().addCTListener(this);
      // Let the registry create events for cards which are already present
      CardTerminalRegistry.getRegistry()
                        .createEventsForPresentCards(this);
      iTracer.info("SignatureCard", "OCF is up and running.");
    }
    catch (OpenCardException e) {
      iTracer.critical("SignatureCard", e);
      throw new SignatureCardException(
            "Exception from SignatureCard constructor", e);
    }
    catch (ClassNotFoundException e) {
      iTracer.critical("SignatureCard", e);
      throw new SignatureCardException(
            "Exception from SignatureCard constructor", e);
    } // Non-OpenCard exceptions pass as they are.
  }

  /****************************************************************************
  * React on card removed events sent by OCF: Invalidate card and card service
  * @param ctEvent The card inserted event.
  ****************************************************************************/
  public synchronized void cardRemoved(
                               CardTerminalEvent ctEvent)
  {
    iTracer.info("cardRemoved",
                 "Got a CARD_REMOVED event.");
    if (ctEvent.getSlot().equals(currentSlot))  {
      iTracer.info("cardRemoved",
                   "The removed card was ours.");
      card = null;
      currentSlot = null;
      fileService = null;
      signatureService  = null;
      earlierException = null;
    }
  }

  /****************************************************************************
  * React on card inserted events sent by OCF: Get new card and card service
  * @param ctEvent The card inserted event.
  ****************************************************************************/
  public void cardInserted(CardTerminalEvent ctEvent)
  {
    iTracer.info("cardInserted",
                 "Got a CARD_INSERTED event.");
    try {
      CardRequest cr = new CardRequest(
                            FileAccessCardService.class);
      card = SmartCard.getSmartCard(ctEvent);
      if (card != null)
        allocateServices(card, ctEvent.getSlot());
    }
    catch (OpenCardException e) {
        iTracer.critical("cardInserted", e);
        earlierException = e;
    }
    catch (ClassNotFoundException e) {
        iTracer.critical("cardInserted", e);
        earlierException = e;
    }
  }

  /****************************************************************************
  * Check presence of a signature card.  A signature card has a file system too.
  * @return true, if a signature card is present in a slot, false otherwise
  ****************************************************************************/
  public boolean signatureCardPresent()
  {
    return fileSystemCardPresent()
           && (signatureService != null);
  }

  /****************************************************************************
  * Check presence of a file system card.
  * @return true, if a file system card is present in a slot, false otherwise
  ****************************************************************************/
  public boolean fileSystemCardPresent()
  {
    return cardPresent() && (fileService != null);
  }

  /****************************************************************************
  * Check presence of an (eventually uninitialized) smart card.
  * @return true, if a smart card is present in a slot, false otherwise
  ****************************************************************************/
  public boolean cardPresent()
  {
    return (card != null);
  }

  /****************************************************************************
  * Clean up SignatureCard, i.e. in this case shut down OpenCard.
  ****************************************************************************/
  void close () throws SignatureCardException
  {
    iTracer.debug("close", "Stopping OpenCard ...");
    try {
      SmartCard.shutdown ();
    }
    catch (OpenCardException e) {
        iTracer.critical("close", e);
        throw new SignatureCardException(
           "Exception from closing OCF", e);
    }
  }

  /****************************************************************************
  * Sign given data using the given key.
  * @param keyNumber the number if the private key on the SmartCard to be
  *                  used for generating the signature
  * @param data      the data to be signed
  * @return           the signature
  ****************************************************************************/
  byte[] sign(int keyNumber, byte[] data)
         throws SignatureCardException, InvalidKeyException
  {
    byte[]   signature = null;
    try {
      propagateAnEarlierException();

      // If no card is present, indicate to application
      // that it must prompt for card
      if (!signatureCardPresent())
        return null;

      // specify the key used for signing
      PrivateKeyFile kf = new PrivateKeyFile(
                      new CardFilePath(":C110"), keyNumber);

      // Let the card generate a signature
      signature = signatureService.signData(
                     kf, JCAStandardNames.SHA1_RSA,
                     JCAStandardNames.ZERO_PADDING, data);
    }
    catch (OpenCardException e) {
        iTracer.critical("sign", e);
        throw new SignatureCardException(
              "Exception from signature operation", e);
    }

    return signature;
  }

  /****************************************************************************
  * Get the card holder data from the SmartCard.
  * @return The card holder data read from the card.
  ****************************************************************************/
  public byte[] getCardHolderData()
                 throws SignatureCardException,
                   IOException, FileNotFoundException
  {
    try {
      propagateAnEarlierException();

      // If no card is present, indicate to the application
      // that it must prompt for card
      if (!fileSystemCardPresent())
        return null;

      // The file holding card holder name and e-Mail address:
      CardFile file = new CardFile(fileService, ":C009");

      // Create a CardFileInputStream for file
      DataInputStream dis = new DataInputStream(
                             new CardFileInputStream(file));

      // Read in the owner's name
      byte[] cardHolderData = new byte[file.getLength()];
      iTracer.info("getCardHolderData", "Reading data");
      dis.read(cardHolderData);

      // Explicitly close the InputStream to yield the card
      // to other applications
      dis.close();
      return cardHolderData;
    }
    catch (OpenCardException e) {
        iTracer.critical("getCardHolderData", e);
        throw new SignatureCardException(
              "Exception from getting cardholder data", e);
    }
  }

  /****************************************************************************
  * Set the given card holder data (write them to the SmartCard).
  * @param cardHolderData - data to be written to the card
  ****************************************************************************/
  public boolean setCardHolderData(String cardHolderData)
                  throws SignatureCardException,
                   IOException, FileNotFoundException
  {
    try  {
      propagateAnEarlierException();

      // If no card is present, indicate to application
      // that it must prompt for card
      if (!fileSystemCardPresent())
        return false;

      // This is the file holding card holder name and e-Mail address
      CardFile file = new CardFile(fileService, ":C009");

      // Create a CardFileInputStream for the file
      DataOutputStream dos = new DataOutputStream(
                           new CardFileOutputStream(file));

      // Write the owner's name
      if (cardHolderData == null)
         cardHolderData = " ";
      byte[] temp = new byte[file.getLength()];
      byte[] chd = cardHolderData.getBytes();
      System.arraycopy(chd, 0, temp, 0, chd.length);
      dos.write(temp, 0, temp.length);

      // Explicitly close the InputStream to yield the smart card
      // to other applications
      dos.close();
      return true;
    }
    catch (OpenCardException e) {
        throw new SignatureCardException(
              "Exception from setting cardholder data", e);
    }
  }

  /**
   * Allocate a file access service and a signature service.
   *
   * Failure to allocate a file access service will throw an exception.
   * Failure to allocate a signature service is merely creating a
   * trace entry.  At least we can do file I/O with this card then.
   *
   * @param card   the smartcard for which to allocate the services
   */
  private void allocateServices(SmartCard card, Slot slot)
         throws ClassNotFoundException,
                CardServiceException
  {
    fileService = (FileAccessCardService)
                    card.getCardService(
                        FileAccessCardService.class, true);

    // If we get here, we need to remember the slot for card removal
    currentSlot = slot;

    SBCHVDialog dialog = new SBCHVDialog();
    fileService.setCHVDialog(dialog);

    try {
      signatureService = (SignatureCardService)
                        card.getCardService(
                        SignatureCardService.class, true);
      signatureService.setCHVDialog(dialog);
    }              // Special handling of failure to allocate
                   // a signature service:
    catch (ClassNotFoundException e)  {
      iTracer.critical("allocateServices", e);
    }
  }

  /**
   * Propagate (= rethrow) an earlier exception that we did not
   * throw because we were in a method that must return immediately.
   * If there was no earlier exception, this method has no effect.
   */
  private void propagateAnEarlierException()
                      throws SignatureCardException
  {
      if (earlierException == null)  return;

      SignatureCardException signCardExcp
         = new SignatureCardException("Exception of "
             + earlierException.getClass().toString()
             + " from earlier method execution:",
             earlierException);
      earlierException = null;
      throw signCardExcp;
  }

} // class SignatureCard