(* :Title: JavaView with JLink *)

(* :Context: JavaView`JLink` *)

(* :Author:
        Konrad Polthier
        polthier@math.tu-berlin.de
        http://www.javaview.de/
*)

(* :Package Version: 0.0.7 *)
(* :History 0.0.7, 27.03.01: Set codebase inside Mathematica tree. *)
(* :History 0.0.6, 21.03.01: Code beautifications. *)
(* :History 0.0.5, 04.03.01: RunJavaView option Animate implemented. *)
(* :History 0.0.4, 10.01.01: Comments of Todd incorporated. *)
(* :History 0.0.1, **.07.00: First version. *)

(* :Mathematica Version: 4.0 *)
		     
(* :Keywords: Graphics, JavaView, Geometry, Applets, Animation, J/Link *)

(*:Requirements:
Commands launching JavaView require the JavaView archives
installed within Mathematica, and J/Link to be installed.
*)

(*:Summary:
This package provides commands to setup and attach JavaView displays to
Mathematica for display and interaction with Mathematica graphics.
Events issued from the JavaView display may be received in Mathematica.
Further, export of Mathematica graphics for online display in web pages
using JavaView applets.
*)

BeginPackage["JavaView`JLink`","JLink`","JavaView`RunThrough`"]

InstallJavaView::usage =
"InstallJavaView[] calls JLink`InstallJava[] to launch the Java runtime and prepares it
to be used from Mathematica. InstallJavaView has the same options as InstallJava
to control the name of the Java runtime program, the classpath used, etc.
Only one Java runtime is ever launched; subsequent calls to InstallJavaView after the
first have no effect. InstallJavaView[] assumes that the JavaView`JLink` package is loaded.
\n\n
InstallJavaView[] creates a JavaView viewer manager and a new display inside a new frame.
The frame is set visible in the first invocation of JavaView[g].
\n\n
Some JavaView classes are loaded to simplify access to static methods and variables.
\n\n
Return value of this command is the JavaView viewer manager which is instance of class
jv.viewer.PvViewer. The viewer can be queried for the current display, the current geometry
or current project.
\n\n
See in package JavaView`RunThrough`: JavaView[g], WriteMgs[g,file], WriteHtml[g,file]"

JavaView::usage =
"JavaView[g] displays a Mathematica graphics object in JavaView.
Graphics objects may be Graphics, Graphics3D, SurfaceGraphics, and others.
\n\n
The graphics object is formatted by replacing infinite precision with standard double precision
in order to avoid parsing of Mathematica specific number representations.
\n\n
In contrast to the command 'RunJavaView[g]' this JLink connection to JavaView does not block
further execution of Mathematica.
\n\n
Return value is the JavaView display which is instance of class jv.viewer.PvDisplay.
\n\n
See: InstallJavaView[]
See in package JavaView`RunThrough`: JavaView[g], WriteMgs[g,file], WriteHtml[g,file]"

JavaViewGeom::usage =
"JavaViewGeom[geom] displays a JavaView geometry in a JavaView display.
Geometry must implement the interface jv.project.PgGeometryIf.
\n\n
See: JavaView[]"

(* Avoid at least some annoying warning messages. *)
Off[General::"spell1"];
Off[General::"privv"];
Off[ParametricPlot3D::"ppcom"];

(* Load Java runtime into Mathematica. *)
Needs["JLink`"]

(* Export functionality is provided with JavaView`RunThrough`. *)
Needs["JavaView`RunThrough`"]

Begin["`Private`"]

(* Same options as InstallJava, but different default values.
   Options are just forwarded to InstallJava[]. *)
Options[InstallJavaView] = {ClassPath->None, CommandLine->wjview, MicrosoftJava->True}

InstallJavaView[opts___?OptionQ]:=
  Module[{CP, cmdLine, msJava},
		(* Forward options to InstallJava[] but use our default values. *)
		{cmdLine, CP, msJava} =
			{CommandLine, ClassPath, MicrosoftJava} /. Flatten[{opts}] /. Options[InstallJavaView];
      InstallJava[CommandLine->cmdLine, ClassPath->CP, MicrosoftJava->msJava];
		(* Simply quit if this method was called before. *)
      If[ValueQ[viewer]==True,
    		Return[viewer]
    	];
    	viewer = JavaNew["jv.viewer.PvViewer"];
    	(* Create a JavaView 3d-display, do not show it yet *)
    	disp = viewer@newDisplay["JavaView: From Mathematica", True];
    	frame = disp@getFrame[];
    	frame@setSize[400,400];
    	(* Load some basic JavaView interfaces, sometimes useful. *)
		LoadClass["jv.object.PsConfig"];
		LoadClass["jv.object.PsObject"];
		LoadClass["jv.object.PsDebug"];
		LoadClass["jv.object.PsViewerIf"];
		LoadClass["jv.project.PgGeometryIf"];
		LoadClass["jv.project.PvGeometryIf"];
		LoadClass["jv.project.PvDisplayIf"];
		LoadClass["jv.project.PvViewerIf"];
		LoadClass["jv.vecmath.PdVector"];
		LoadClass["jv.vecmath.PiVector"];
    	LoadClass["jv.loader.PgLoader"];
    	LoadClass["jv.loader.PvDisplayOption"];
		(* Compute codebase. *)
		thisPkgPath = findCurrentPkg[];
		If[thisPkgPath == "",
			Message[JavaView::path];
			Return[$Failed],
			sep 		= $PathnameSeparator;
			codebase	= StringReplace[thisPkgPath, $PathnameSeparator -> "/"];
  		  	PsConfig`setCodeBase[codebase];
		];
		viewer
    ]

JavaView::inst =
"JavaView does not seem to be properly installed. InstallJavaView[] must be
called prior to JavaView[g]."

JavaView::load =
"JavaView: loading failed, graphics="

JavaView::path =
"Path to JavaView package not found, empty string as path."

Options[ JavaView ] = {
	Animatable -> False,
	Debug -> False	
}

(* View a JavaView geometry produced in Mathematica in a JavaView. *)
JavaView[geom_?JavaObjectQ]:= JavaViewGeom[geom]

(* View a Mathematica graphics in a JavaView display using JLink. *)
JavaView[g_Graphics]:= JavaViewGraphics[g]
JavaView[g_Graphics3D]:= JavaViewGraphics[g]
JavaView[g_SurfaceGraphics]:= JavaViewGraphics[g]
JavaView[g_DensityGraphics]:= JavaViewGraphics[g]
JavaView[g_ContourGraphics]:= JavaViewGraphics[g]
JavaView[]:=
  	Module[{},
      If[ValueQ[frame]==False,
			Message[JavaView::inst];
			Return[$Failed]
    	];
		JavaShow[frame];
    ]
(* View a List of Mathematica graphics in a JavaView display using JLink. *)
JavaView[g_List,opts___?OptionQ] := 
	Module[{gs,animation},
    	{animation} = {Animatable}/. Flatten[{opts, Options[JavaView]}];
    	If[animation === True,PgLoader`setAnimation[True];,PgLoader`setAnimation[False];];
    	gs=FormatGraphics[g];
    	geomArr = PgLoader`load[Null,gs,"mgs"];
    	If[geomArr==Null || geomArr[[1]]==Null,
    		Message[JavaView::load<>gs];
    		Return[]
    	];
    	RemoveGeometries[];
		dispOption = PgLoader`getDisplayOption[];
    	If[dispOption =!= Null,
    		dispOption@configure[disp]
    	];

    	For[i=1,i<=Length[geomArr],i++,
			disp@addGeometry[geomArr[[i]]];
			geomArr[[i]]@update[geomArr[[i]]];
		];
    	JavaShow[frame];
    	PgLoader`setAnimation[False];
		geomArr
    ]
JavaViewGraphics[g_]:=
  	Module[{gs,sort},
      If[ValueQ[frame]==False,
			Message[JavaView::inst];
			Return[$Failed]
    	];
  		sort = True;
    	If[Head[g] === ContourGraphics,
  			gs = FormatGraphics[Graphics[g]];
  			sort = False;,
  			gs = FormatGraphics[g];
  		];
    	geomArr = PgLoader`load[Null,gs,"mgs"];
    	If[geomArr==Null || geomArr[[1]]==Null,
    		Message[JavaView::load<>gs];
    		Return[]
    	];
    	RemoveGeometries[];
		dispOption = PgLoader`getDisplayOption[];
    	If[dispOption =!= Null,
    		dispOption@setOption[PvDisplayOption`SHOWUSORTING,sort];
    		dispOption@configure[disp]
    	];

    	For[i=1,i<=Length[geomArr],i++,
			disp@addGeometry[geomArr[[i]]];
			geomArr[[i]]@update[geomArr[[i]]];
		];
		JavaShow[frame];
    	geomArr[[1]]
    ]
(* View a JavaView geometry produced in Mathematica in a JavaView display using JLink. *)
JavaViewGeom[geom_]:=
  	Module[{},
      If[ValueQ[frame]==False,
			Message[JavaView::inst];
			Return[$Failed]
    	];
    	RemoveGeometries[];
    	disp@addGeometry[geom];
    	JavaShow[frame];
    	geom@update[geom];
    	geom
    ]
    
(* Remove all geometries from JavaView display. *)
RemoveGeometries[]:=
	Module[{},
		geometries=disp@getGeometries[];
    	For[i=1,i<=Length[geometries],i++,
    		parent=geometries[[i]]@getFather[];
    		If[parent=!=Null,
    			If[parent@hasAnimation[]===True,
    				parent@getAnimation[]@getAnimationPanel[]@setVisible[False];
    			];
    		];
    	];
    	disp@removeGeometries[];
    ]
    	
End[]

EndPackage[]
