(* :Title: JavaView via RunThrough *)

(* :Context: JavaView`RunThrough` *)

(* :Author:
        Konrad Polthier
        polthier@math.tu-berlin.de
        http://www.javaview.de/
*)

(* :Package Version: 0.0.7 *)
(* :History 0.0.7, 22.04.01: Options for Html export and animations. *)
(* :History 0.0.6, 27.03.01: Set codebase inside Mathematica tree. *)
(* :History 0.0.5, 04.03.01: RunJavaView option Animate implemented. *)
(* :History 0.0.4, 15.02.01: New module writeAnim implemented. *)
(* :History 0.0.1, **.07.00: First version. *)

(* :Mathematica Version: 4.0 *)
		     
(* :Keywords: Graphics, JavaView, Geometry, Applets, Animation *)

(*:Requirements:
Commands launching JavaView require a JavaView executable
reps. a Java runtime environment and the JavaView archives to run.
Other commands to save and format Mathematica graphics are self-contained
within this package.
*)

(*:Summary:
This package contains various functions to view Mathematica graphics
with JavaView and to export Mathematica graphics for online display
in JavaView applets of web pages.
*)

BeginPackage["JavaView`RunThrough`"]

RunJavaView::usage =
"RunJavaView[g, options...] displays a Mathematica graphics object in \
JavaView.
Graphics objects may be Graphics, Graphics3D, SurfaceGraphics, and others.
\n\n
The method uses the RunThrough[g] command of Mathematica to invoke JavaView.
Therefore, on windows the executable 'javaview.exe' respectively on Unix the
shell file 'javaview' must be in the path such that Mathematica is able to \
find it.
\n\n
The graphics object is formatted by replacing infinite precision with \
standard double
precision in order to avoid parsing of Mathematica specific number \
representations.
\n\n
This command blocks further execution of Mathematica until JavaView exits. \
During
the lifetime of the JavaView process the kernel is waiting for JavaView to \
finish.
\n\n
See: WriteMgs[g,file], WriteAnim[g,file], WriteHtml[g,file]
See in package JavaView`JLink`: JavaView[g]"

WriteMgs::usage =
"WriteMgs[g,file] saves a Mathematica graphics object in the given file, \
which may
later be viewed with JavaView from the command line or read from an applet.
The file name should have the extension 'file.mgs' to be identified as \
Mathematica
graphics file. To view the saved file, type at the command prompt either
'javaview file' or 'java javaview file' depending on the Java runtime system,
or import the file back into Mathematika using '<<filename.mgs'.
\n\n
The command uses FormatGraphics[g] to format a Graphics object into a string
and saves it in the given file. The file must be specified relative to \
Mathematica
or with an absolute path.
\n\n
See: RunJavaView[g], WriteAnim[g,file], WriteHtml[g,file]
See in package JavaView`JLink`: JavaView[g]"

WriteAnim::usage =
"WriteAnim[g,file] saves a list of Mathematica graphics objects in the given \
file
as an animation. The animation may later be viewed with JavaView from the \
command
line or read from an applet.
The file name should have the extension 'file.mgs' to be identified as \
Mathematica
graphics file. To view the saved file, type at the command prompt either
'javaview file' or 'java javaview file' depending on the Java runtime system,
or import the file back into Mathematika using '<<filename.mgs'.
\n\n
The command uses FormatGraphics[g] to format a Graphics object into a string
and saves it in the given file. The file must be specified relative to \
Mathematica
or with an absolute path.
\n\n
See: RunJavaView[g], WriteMgs[g,file], WriteHtml[g,file]
See in package JavaView`JLink`: JavaView[g]"

WriteHtml::usage =
"WriteHtml[g,file,options...] saves a Mathematica graphics object as applet in an Html \
file,
which may later be viewed with any Java enabled web browser. The applet uses
JavaView to display the graphics. The command uses FormatGraphics[g] to \
format a
Graphics object into a string and saves it in the given HTML file. The file \
must
be specified relative to Mathematica or with an absolute path.
\n\n
The applet contains no codebase setting, it relies on the JavaView archives
being included in the classpath.
\n\n
See: RunJavaView[g], WriteMgs[g,file], WriteAnim[g,file]
See in package JavaView`JLink`: JavaView[g]"

FormatGraphics::usage =
"FormatGraphics[g] formats a graphics object by replacing infinite precision \
with
standard double precision in order to avoid parsing of Mathematica specific \
number
representations. Further, the graphics is converted to InputForm, CForm and
finally returned as String."

findCurrentPkg::usage =
"findCurrentPkg[] return the path to the current package."

RunJavaView::path =
"Path to JavaView package not found, empty string as path."

Options[ RunJavaView ] = {
	Animatable -> False,
	Debug -> False,
	Verbose -> False
	}
Options[ WriteHtml ] = {
	Codebase -> "./",
	JvxArchive -> False,
	PathToArchive -> "Java/",
	Verbose -> False
	}

Begin["`Private`"]

Off[General::"spell1"];
Off[General::"privv"];
Off[ParametricPlot3D::"ppcom"];
(* Off[ParametricPlot3D]; *)

(* findCurrentPkg will return a full pathname to the current package file if its \
location fits the following rule.
   For a context named Foo`Bar`, the file can be in either *\Foo\Bar.m or \
*\Foo\Bar\Bar.m, where * stands for
   any directory on $Path. The second form is intended to catch cases where \
the developer relies on the idiom
   of putting a Kernel\init.m file in the directory Foo\Bar, which loads the \
package Foo\Bar\Bar.m.
   It returns "" if the file cannot be found. The returned dir name, if not \
"", ends with $PathnameSeparator.
   This module is taken from JLink.m following the kind advice of Todd \
Gayley.
*)
findCurrentPkg[] := 
	Module[{ctxt, pkgPath, pkgPath2, shortCtxt, correctedPaths, fullPaths, res},
		ctxt = StringDrop[Context[findCurrentPkg], -1];  (* Take off ` *)
		(* Print[ctxt]; *)
		pkgPath = StringReplace[ctxt, "`" -> $PathnameSeparator] <> ".m";
		(* Print[pkgPath]; *)
		shortCtxt = StringDrop[ctxt, Last[{0} ~Join~ Flatten[StringPosition[ctxt, "`"]]]];
		(* Print[shortCtxt]; *)
		pkgPath2 = StringReplace[ctxt <> $PathnameSeparator <> shortCtxt, "`" -> $PathnameSeparator] <> ".m";
		(* Print[pkgPath2]; *)
		correctedPaths = If[StringTake[#, -1] =!= $PathnameSeparator, # <> $PathnameSeparator, #]& /@ $Path;
		(* Print[correctedPaths]; *)
		fullPaths = Join[(# <> pkgPath)& /@ correctedPaths, (# <> pkgPath2)& /@ correctedPaths];
		(* Print[fullPaths]; *)
		res = DeleteCases[File /. (FileInformation /@ fullPaths), File];
		(* Print["res = " <> res]; *)
		If[Length[res] == 0,
			"",
		(* else *)
			(* Drop filename, leaving just dir. +2 is for .m *)
			StringDrop[First[res], -(StringLength[shortCtxt] + 2)] //
				(If[StringTake[#, -1] =!= $PathnameSeparator, # <> $PathnameSeparator, #]&)
		]
	]

(* Use RunThrough to view a Mathematica graphics object in JavaView.
Requires existence of an executable shell script or binary "javaview.exe" in the path. *)
RunJavaView[g_,opts___?OptionQ] :=
	Module[{gs,animation,debug,verbose,sort,thisPkgPath,codebase,binbase,jvBin,sep,fmtGeom,jvOptions},
		{animation,debug,verbose} = {Animatable,Debug,Verbose}/. Flatten[{opts, Options[RunJavaView]}];
		jvOptions = " file=stdin";
		If[ animation === True,
			jvOptions = jvOptions <> " Animate=Show";
		];
		If[Head[g] === ContourGraphics,
			gs = Graphics[g];
			jvOptions = jvOptions <> " Sorting=Hide";,
			gs=g;
		];
		(* Compute codebase. *)
		thisPkgPath = findCurrentPkg[];
		If[thisPkgPath == "",
		 	Message[RunJavaView::path];
			Return[$Failed]
		];
		sep 			= $PathnameSeparator;
		binbase 	= thisPkgPath <> $PathnameSeparator <> "bin" <> $PathnameSeparator;
		codebase	= StringReplace[thisPkgPath, $PathnameSeparator -> "/"];
		jvBin 		= binbase <> "javaview.exe" <> " codebase=" <> codebase;
		If[verbose === True,
			Print["calling: " <> jvBin <> jvOptions];
		];
  		fmtGeom = InputForm[N[gs] /. x_Real :> CForm[Chop[x]]];
		RunThrough[jvBin <> jvOptions, fmtGeom];
	]

(* Format graphics objects by reducing precision to machine accuracy, 
and inserting line breaks for better readability of ascii files. *)
FormatGraphics[g_] :=
	Module[{}, StringReplace[ToString[N[g] /. x_Real :> CForm[Chop[x]],
		FormatType -> InputForm, PageWidth -> Infinity], {"}," -> "},\n", "]," -> "],\n"}]
	]

(* Write a graphics object to a geometry file. 
The file filename.mgs on a local disk may be displayed with JavaView
using "javaview filename.mgs" or "java javaview filename.mgs",
or it maybe imported back into Mathematika using "<<filename.mgs". *)
WriteMgs[g_, filename_String] :=
	Module[{strm = OpenWrite[filename, PageWidth -> Infinity]},
		WriteString[strm, FormatGraphics[g]];
		Close[strm]
	]

(* Write a list of graphics object to into an ascii file as argument of \
ShowAnimation[].
The file filename.mgs on a local disk may be displayed with JavaView
using "javaview filename.mgs" or "java javaview filename.mgs",
or it maybe imported back into Mathematika using "<<filename.mgs". *)
WriteAnim[g_, filename_String] := 
	Module[{strm = OpenWrite[filename, PageWidth -> Infinity]},
		WriteString[strm, "ShowAnimation["]; 
		WriteString[strm, FormatGraphics[g]];
		WriteString[strm, "]"];
		Close[strm]
	]

(* Write a graphics object into an Html file as parameter of a JavaView \
applet.
The applet contains no codebase setting, therefore it relies on the JavaView \
archives
being included in the classpath. *)
WriteHtml[g_, filename_String ,opts___?OptionQ] :=
  	Module[{strm = OpenWrite[filename, PageWidth -> Infinity],geom,archive,htmlstart,htmlend},
		{codebase,jvxarchive,pathtoarchive,verbose} = {Codebase,JvxArchive,PathToArchive,Verbose}/. Flatten[{opts, Options[WriteHtml]}];

		(* Replace inner "-quotes to avoid interference with outer quotes of applet parameter. *)
		geom = FormatGraphics[g];
		geom = StringReplace[geom, "\""-> "'"];

		archive = pathtoarchive<>"javaview.jar";
		If[ jvxarchive === True,
			archive = archive <> "," <> pathtoarchive <> "jvx.jar";
		];
		(* Text for Html Output *)
		htmlstart = "<html>\n
		<head><title>Demo: Mathematica Graphics via Applet Parameter</title></head>\n
		<body>\n
		<h2>Demo: Mathematica Graphics via Applet Parameter</h2>\n
		<P>\n
		This Html page contains a demonstration applet which uses an applet\n
		parameter to pass a Mathematica graphics objects to\n
		<EM><FONT color=#004080>JavaView</FONT></EM> for display</P>\n
		<p>Note, the applet contains a codebase setting which must be adjusted to your settings.</p>
		<p>\n
		<APPLET code=javaview.class name=\"Parameter Demo\"\n
			codebase=\""<>codebase<>"\" height=300 width=400\n
			archive=\""<>archive<>"\">\n
			<PARAM NAME=control VALUE=Hide>\n
			<PARAM NAME=panel VALUE=Material>\n
			<PARAM NAME=language VALUE=en>\n
			<PARAM NAME=menu-new-project VALUE=disable>\n
			<PARAM NAME=mathematica VALUE=\"\n";
		htmlend = "\">\n</APPLET>\n</p>\n</body>\n</html>";

		WriteString[strm, htmlstart <> geom <> htmlend];
		Close[strm]
    ]
End[]

EndPackage[]   (* JavaView`RunThrough` *)
