package vgp.tutor.eventCamera;

import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Point;

import jv.object.PsConfig;
import jv.object.PsDebug;
import jv.geom.PgElementSet;
import jv.project.PvDisplayIf;
import jv.project.PvCameraIf;
import jv.project.PvCameraEvent;
import jv.project.PvCameraListenerIf;
import jv.project.PjProject;
import jv.project.PvViewerIf;
import jv.vecmath.PdVector;
import jv.vecmath.PuVectorGeom;

/**
 * Demo project showing how to handle camera events issued from
 * a JavaView display. In this example the events are used to
 * steer the camera in a second window.
 * <p>
 * The camera events from one window are used to steer the
 * camera in the other window such that a stereo effect is produced.
 * Note, the viewing direction of the left camera is slightly rotated
 * around the up-vector of the camera by about 6 degrees.
 * 
 * @see			jv.project.PvCameraEvent
 * @see			jv.project.PvCameraListenerIf
 * @see			jv.project.PvDisplayIf
 * @author		Konrad Polthier
 * @version		25.07.00, 1.00 revised (kp) <br>
 *					25.07.00, 1.00 created (kp)
 */
public class PjEventCamera extends PjProject implements PvCameraListenerIf {
	/** Torus geometry added to both displays. */
	protected	PgElementSet		m_geom;
	/** Left display which is steered. */
	protected	PvDisplayIf			m_dispRight;
	/** Right display which issues camera events. */
	protected	PvDisplayIf			m_dispLeft;

	public PjEventCamera() {
		super("Camera Events");
		m_geom = new PgElementSet(3);
		if (getClass() == PjEventCamera.class)
		  init();
	}
	public void init() {
		super.init();
		m_geom.setName("Torus");
		m_geom.computeTorus(10, 10, 2., 1.);
		m_geom.setGlobalVertexSize(4);
		m_geom.showVertices(true);
		m_geom.showElements(false);
		m_geom.update(m_geom);
	}
	public void start() {
		// Add geometry to default window, ie. the right window.
		addGeometry(m_geom);
		selectGeometry(m_geom);

		getLeftDisplay().addGeometry(m_geom);
		getRightDisplay().addGeometry(m_geom);
		super.start();
	}
	public PvDisplayIf getRightDisplay() {
		if (m_dispRight != null)
			return m_dispRight;
		// Get viewer and ask for another display
		PvViewerIf viewer = getViewer();
		// Create right window and add a clone of the torus geometry.
		m_dispRight = viewer.newDisplay("Right Display", PsConfig.isApplication());
		m_dispRight.addCameraListener(this);
		if (PsConfig.isApplication()) {
			Frame frame = m_dispRight.getFrame();
			frame.setTitle("Right Eye");
			frame.setBounds(500, 50, 250, 250);
			frame.validate();
			frame.setVisible(true);
		}
		return m_dispRight;
	}
	public PvDisplayIf getLeftDisplay() {
		if (m_dispLeft != null)
			return m_dispLeft;
		// Get viewer and ask for another display
		PvViewerIf viewer = getViewer();
		// Create left window and add a clone of the torus geometry.
		m_dispLeft = viewer.newDisplay("Left Display", PsConfig.isApplication());
		m_dispLeft.addCameraListener(this);
		if (PsConfig.isApplication()) {
			Frame frame = m_dispLeft.getFrame();
			frame.setTitle("Left Eye");
			frame.setBounds(250, 50, 250, 250);
			frame.validate();
			frame.setVisible(true);
		}
		return m_dispLeft;
	}
	/**
	 * Get camera events resulting from picking the mouse.
	 * Use information about camera in one display to
	 * adjust the camera in the other window. But previously
	 * rotate the camera position around the line given by
	 * interest+t*upVector by -6 degrees.
	 */
	public void pickCamera(PvCameraEvent pos) {
		if (m_bUpdating)
			return;
		// Do nothing if not both display have been created yet
		if (m_dispLeft==null || m_dispRight==null)
			return;
				
		// Point location			= pos.getLocation();
		// PdVector vertex		= pos.getVertex();
		PdVector interest		= pos.getInterest();
		PdVector upVector		= pos.getUpVector();
		// PdVector viewDir		= pos.getViewDir();
		PdVector position		= pos.getPosition();
		double scale			= pos.getScale();

		// Find out which camera has issued the event,
		// and then steer the other
		PvDisplayIf dispSrc		= pos.getSource();
		PvDisplayIf dispSteer	= null;
		PvCameraIf	cam			= null;
		double angle;
		if (dispSrc == m_dispRight) {
			cam			= m_dispLeft.getCamera();
			dispSteer	= m_dispLeft;
			angle			= -6.*Math.PI/180.;
		} else {
			cam			= m_dispRight.getCamera();
			dispSteer	= m_dispRight;
			angle			= +6.*Math.PI/180.;
		}
		// Adjust the camera in the steered display
		cam.setInterest(interest);
		cam.setUpVector(upVector);
		PdVector pRot = (PdVector)position.clone();
		PuVectorGeom.rotatePointAroundLine(pRot, position, interest, upVector, angle);
		cam.setPosition(pRot);
		cam.setScale(scale);
		// Update the camera in the steered display
		// but block this current method to avoid an infinite loop
		// since the steered camera will also issue a camera event.
		m_bUpdating = true;
		dispSteer.update(cam);
		m_bUpdating = false;
	}
	boolean m_bUpdating = false;
	/**
	 * Get camera events resulting from dragging the mouse.
	 * This method just calls the method pickCamera which does all the work.
	 */
	public void dragCamera(PvCameraEvent pos) {
		pickCamera(pos);
	}
}

