/*
 * Copyright (c) year living systems (R) AG, Germany.
 * All rights reserved.
 * Original Author: Slava Litskevich
 *
 * $Archive:   P:/java/pvcs/archives/Source/com/ls/research/loadbalance/LoadBalance.java-arc  $
 * $Date:   11 Oct 2001 14:20:02  $
 */

package com.ls.demo.capabilities.loadbalance;

import com.ls.logiccontrol.capabilities.LogicCapabilities;
import com.ls.service.log.ILogger;
import com.ls.util.xml.XMLFragment;
import com.ls.lars.communication.Message;
import com.ls.lars.communication.SingleMessage;
import com.ls.TimeoutException;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.List;
import java.util.Vector;
import java.io.RandomAccessFile;
import java.io.IOException;
//import java.util.Hashtable;
//import java.util.Random;

import com.ls.service.log.ILogger;
import com.ls.logiccontrol.LogicConfigurationException;

/**
 * Used to determin distance between two cities in Europe
 * @author  Last modified by $Author:   OHils  $
 * @version $Revision:   1.6  $
 */
public class LoadBalance extends LogicCapabilities
{
	/** constant for name of this class */
	static private final String CLASS_NAME 				= "LoadBalance";
	/** value to be returned by the perception */
	static private final double TRUE					= 1.0;
	/** value to be returned by the perception */
	static private final double FALSE					= 0.0;
	/** timeout in ms for synchronous communication */
	static private final int	SYNCHRONOUS_TIMEOUT		= 5000;
	// !!! dirty. also defined in AgentLoadBalancer
	/** tag used in workflowModel for the number of agents on this platform */
	static private final String TAG_NUMBER_OF_AGENTS	= "numberOfAgents";
	/** tag used in workflowModel for the CPU load on this platform */
	static private final String TAG_CPU_LOAD			= "cpuLoad";

	/** unique id for replyWith attribute */
	static private long uniqueReplyID	 	   			= 0L;


	/** number of agents for which highLoad starts to get true */
	private double highLoadByNumberLow;

	/** number of agents for which highLoad is true */
	private double highLoadByNumberHigh;

	/** cpu usage in percent for which highLoad starts to get true */
	private double highLoadByCPULow;

	/** cpu usage in percent for for which highLoad is true */
	private double highLoadByCPUHigh;

	/** filename where to get the cpu load from */
	private String dataCPUFilename;

	/** Default constructor
	 */
	public LoadBalance()
	{
		highLoadByNumberLow		= 10.0;
		highLoadByNumberHigh	= 20.0;
		highLoadByCPULow		= 50.0;
		highLoadByCPUHigh		= 90.0;
	}

	/**
     * @param configuration The configuration of the  (usually  a XMLFragment)
	 * @throws LogicConfigurationException If the  could not be configured
     */
    public void configure(Map configuration) throws LogicConfigurationException
	{
		final String methodName = "configure";

		try {
			// configure the number of agents for which high load is true
			XMLFragment subTag			= (XMLFragment)((XMLFragment)configuration).getChild("highLoadByNumber");
			String		lowValueString	= (String) ((XMLFragment) subTag.getChild("lowValue")).getContent();
			String		highValueString	= (String) ((XMLFragment) subTag.getChild("highValue")).getContent();
			highLoadByNumberLow			= (Double.valueOf(lowValueString)).doubleValue();
			highLoadByNumberHigh		= (Double.valueOf(highValueString)).doubleValue();
			iLogger.log(CLASS_NAME, ILogger.TRACE1, methodName,
						"configured highLoadByNumber low: " + highLoadByNumberLow +
						" high: " + highLoadByNumberHigh);

		} catch (NumberFormatException e) {
			throw new LogicConfigurationException("highLoadByNumber lowValue or highValue are no numbers");
		} catch (RuntimeException e) {
			throw new LogicConfigurationException("No highLoadByNumber specified or missing lowValue or highValue");
		}

		try {
			// configure the cpu load (percentage)for which high load is true
			XMLFragment subTag			= (XMLFragment)((XMLFragment)configuration).getChild("highLoadByCPU");
			String		lowValueString	= (String) ((XMLFragment) subTag.getChild("lowValue")).getContent();
			String		highValueString	= (String) ((XMLFragment) subTag.getChild("highValue")).getContent();
			this.dataCPUFilename		= (String) ((XMLFragment) subTag.getChild("dataFile")).getContent();
			highLoadByCPULow			= (Double.valueOf(lowValueString)).doubleValue();
			highLoadByCPUHigh			= (Double.valueOf(highValueString)).doubleValue();
			iLogger.log(CLASS_NAME, ILogger.TRACE1, methodName,
						"configured highLoadByCPU low: " + highLoadByCPULow +
						" high: " + highLoadByCPUHigh);

		} catch (NumberFormatException e) {
			throw new LogicConfigurationException("highLoadByCPU lowValue or highValue is no number");
		} catch (RuntimeException e) {
			throw new LogicConfigurationException("No highLoadByCPU specified or missing lowValue or highValue");
		}

	}

	//******************************** Perceptions ***************************************

	/**
	 * for goal configuration only
	 *
	 * @param workflowModel represents the context of the agent
	 * @return TRUE if the the system is load balanced with other platforms
	 */
	public double perceptionBalancedLoad(Map workflowModel)
	{
        return 0.0;
	}

	/**
	 * TRUE if the the systems load is high with respect to the number of agents on the platform
	 *
	 * @param workflowModel represents the context of the agent
	 * @return TRUE if the the systems load is high with respect to the number of agents on the platform
	 */
	public double perceptionHighLoadByAgentNumber(Map workflowModel)
	{
		final String	methodName	= "perceptionHighLoadByAgentNumber";
		double			load		= 0.0;
		Map				agents		= null;

		try {
			agents	= getAgentsOnPlatform("");
			load	= getLinearFuzzyValue(agents.size(), highLoadByNumberLow, highLoadByNumberHigh);
			workflowModel.put(TAG_NUMBER_OF_AGENTS, new Integer(agents.size()));
			iLogger.log(CLASS_NAME, ILogger.TRACE3, methodName, "load: " + load);

		} catch (TimeoutException e) {
			// we did not get an answer
			iLogger.log(CLASS_NAME, ILogger.ERROR, methodName, "timeout when asking for agents", e);
		}
		return load;
	}

	/**
	 * TRUE if the the systems load is high with respect to the cpu usage
	 *
	 * @param workflowModel represents the context of the agent
	 * @return TRUE if the the systems load is high with respect to the cpu usage
	 */
	public double perceptionHighLoadByCPUTime(Map workflowModel)
	{
		final String	methodName	= "perceptionHighLoadByCPUTime";
		double			cpuLoad		= 0.0;
		double			load		= 0.0;

		cpuLoad	= getLocalCPULoad();
		workflowModel.put(TAG_CPU_LOAD, new Integer((int)cpuLoad));
		load	= getLinearFuzzyValue(cpuLoad, highLoadByCPULow, highLoadByCPUHigh);
		iLogger.log(CLASS_NAME, ILogger.TRACE3, methodName, "load: " + load);
		return load;
	}

	//******************************** Actions ***************************************

	/**
	 * Sends one of the mobile agents to a system with less agents
	 * @param workflowModel represents the context of the agent
	 */
	public void actionBalanceLoad(Map workflowModel)
	{
		String	methodName = "actionBalanceLoad";
		int		agentNumber;

		try {
			iLogger.log(CLASS_NAME, ILogger.TRACE3, methodName, "balancing started");

			// search for an agent which can be moved
			String agentToMove = lookupAgentToMove();
			if (agentToMove != null) {

				// lookup the local agent number
				if (workflowModel.containsKey(TAG_NUMBER_OF_AGENTS)) {
					agentNumber = ((Integer) workflowModel.get(TAG_NUMBER_OF_AGENTS)).intValue();
				} else {
					Map	agents  = getAgentsOnPlatform("");
					agentNumber = agents.size();
					workflowModel.put(TAG_NUMBER_OF_AGENTS, new Integer(agentNumber));
				}

				// search for a less loaded platform
				String platformToMoveTo = lookupPlatformToMoveToByNumber(agentNumber);
				if (platformToMoveTo != null) {
					// we move an agent to another platform
					moveAgentTo(agentToMove, platformToMoveTo);
				} else {
					iLogger.log(CLASS_NAME, ILogger.INFO, methodName, "no platform found or none better");
				}

			} else {
				iLogger.log(CLASS_NAME, ILogger.INFO, methodName, "no agent found to move");
			}

		} catch (TimeoutException e) {
			iLogger.log(CLASS_NAME, ILogger.ERROR, methodName, "timeout when asking for platforms", e);

		} catch (RuntimeException e) {
			iLogger.log(CLASS_NAME, ILogger.ERROR, methodName, e);
		}
	}

	/**
	 * Sends one of the mobile agents to a system with less cpu load
	 * @param workflowModel represents the context of the agent
	 */
	public void actionBalanceLoadByCPU(Map workflowModel)
	{
		String	methodName = "actionBalanceLoadByCPU";
		int		load;

		try {
			iLogger.log(CLASS_NAME, ILogger.TRACE3, methodName, "balancing started");

			// search for an agent which can be moved
			String agentToMove = lookupAgentToMove();
			if (agentToMove != null) {

				// lookup the local load
				if (workflowModel.containsKey(TAG_CPU_LOAD)) {
					// we did this already in a perception
					load = ((Integer) workflowModel.get(TAG_CPU_LOAD)).intValue();
				} else {
					load  = (int) getLocalCPULoad();
					workflowModel.put(TAG_CPU_LOAD, new Integer((int)load));
				}

				// search for a less loaded platform
				String platformToMoveTo = lookupPlatformToMoveToByCPU(load);
				if (platformToMoveTo != null) {
					// we move an agent to another platform
					moveAgentTo(agentToMove, platformToMoveTo);
				} else {
					iLogger.log(CLASS_NAME, ILogger.INFO, methodName, "no platform found or none better");
				}

			} else {
				iLogger.log(CLASS_NAME, ILogger.INFO, methodName, "no agent found to move");
			}

		} catch (TimeoutException e) {
			iLogger.log(CLASS_NAME, ILogger.ERROR, methodName, "timeout when asking for platforms", e);

		} catch (RuntimeException e) {
			iLogger.log(CLASS_NAME, ILogger.ERROR, methodName, e);
		}
	}


	//******************************** Helper ***************************************


	/**
	 *  @param		platform	the name of the platform of which to get the list of agents from.
	 *							for the local platform "" can be passed
	 * 	@return		a Map with all agents on this platform
	 * 	@throws		TimeoutException	   if no response is gotten from AgentManager in time
	 */
	private Map getAgentsOnPlatform(String platform)
	throws TimeoutException
	{
		// ask for a list of agents
		String	  methodName   	= "getLoadByAgentNumber";

		if (platform.length() > 0)
			platform = "@" + platform;

		Message   message 	   	= new SingleMessage("list_agents", "am" + platform, new HashMap());
		message.setReplyWith(String.valueOf(getUniqueReplyID()));
		Message   replyMessage	= iCommunication.sendSynchronousRequest(message, SYNCHRONOUS_TIMEOUT);
		// !!! dirty. do not check subject of incoming message
		Map   	  agents  	   	= (Map) replyMessage.getContent();

		return agents;
	}

	/**
	 *  Reads the local CPU load from a comma separeted logfile of a separately running performance tool
	 * 	@return		the cpu load in percent
	 */
	private double getLocalCPULoad()
	{
		String	methodName   		= "getLoadByAgentNumber";
		String	filename			= this.dataCPUFilename;
		String	line				= null;
		int		noOfTimesToAccount	= 2;
		double	overalLoad			= 0.0;
		int		noOfValues			= 0;
		int		value				= 0;
		double	average				= 0.0;

		try {
			// the load is in a file
			RandomAccessFile	file		= new RandomAccessFile(filename, "r");

			file.seek(file.length() - 50 * noOfTimesToAccount);
			file.readLine();

			// read last load times
			while ((line = file.readLine()) != null) {
				line = line.substring(line.indexOf(',') + 2);
				line = line.substring(0, line.indexOf(','));
				value = (Integer.valueOf(line)).intValue();
				overalLoad += value;
				noOfValues++;
			}

			if (noOfValues > 0) {
				// and take the average
				average = overalLoad / noOfValues;
			} else {
				iLogger.log(CLASS_NAME, ILogger.ERROR, methodName, "could not read a cpu load value");
			}

		} catch (IOException e) {
			iLogger.log(CLASS_NAME, ILogger.ERROR, methodName, "could not read file: " + filename, e);
		} catch (NumberFormatException e) {
			iLogger.log(CLASS_NAME, ILogger.ERROR, methodName, "invalid number: " + line, e);
		} catch (RuntimeException e) {
			iLogger.log(CLASS_NAME, ILogger.ERROR, methodName, e);
		}

		return average;
	}

	/**
	 *	Asks the AgentLoadBalancer of the specified platform for its current cpu load
	 *  @param		platform	the name of the platform of which to get the cpu load from.
	 * 	@return		the cpu load in percent
	 * 	@throws		TimeoutException	   if no response is gotten from AgentLoadBalancer in time
	 */
	private int getCPULoadOnPlatform(String platform)
	throws TimeoutException
	{
		// ask for a load of platform
		String	methodName	= "getCPULoadOnPlatform";
		String	loadString	= null;
		int		load		= 100;

		try {
			Message   message		= new SingleMessage("load_by_cpu", "AgentLoadBalancer@" + platform, new HashMap());
			message.setReplyWith(String.valueOf(getUniqueReplyID()));
			Message   replyMessage	= iCommunication.sendSynchronousRequest(message, SYNCHRONOUS_TIMEOUT);

			// !!! dirty. do not check subject of incoming message
			loadString = (String) replyMessage.getContent();
			load = (Integer.valueOf(loadString)).intValue();
			// if the load is unknown, we have to assume high load
			if (load < 0)
				load = 50;

		} catch (NumberFormatException e) {
			iLogger.log(CLASS_NAME, ILogger.ERROR, methodName, "invalid number: " + loadString, e);
		}

		return load;
	}

	/** Checks which agent can be moved to another platform.
	 * 	@return name of the agent that should be moved
	 * 	@throws	   TimeoutException	   if no response is gotten from AgentManager in time
	 */
	private String lookupAgentToMove()
	throws TimeoutException
	{
		// ask for a list of agents
		String	  methodName   	= "lookupAgentToMove";
		Map   	  agents  	   	= getAgentsOnPlatform("");
		Iterator  allAgents   	= agents.values().iterator();
		String	  agentName	   	= null;
		String	  agentToMove  	= null;

		// look for an agentname with more than n letters
		while (allAgents.hasNext() && agentToMove == null) {
			agentName = (String) allAgents.next();
//			iLogger.log(CLASS_NAME, ILogger.TRACE3, methodName, "agentname: " + agentName);
			// !!! dirty. I can not ask an agent if it is mobile. Therefore I look for "mobile" in the name
			if (agentName.toLowerCase().indexOf("mobile") != -1) {
				agentToMove = agentName;
			}
	   }

	   return agentToMove;
	}

	/** Checks which platform is the least loaded
	 *  @param		agentsOnThisPlatform	number of agents on the local platform
	 * 	@return		name of the platform with least load
	 * 	@throws		TimeoutException	   if no response is gotten from AgentManager in time
	 */
	private String lookupPlatformToMoveToByNumber(int agentsOnThisPlatform)
	throws TimeoutException
	{
		// ask for a list of platforms
		String		methodName   	   	= "lookupPlatformToMoveToByNumber";
		Iterator	allPlatforms 	  	= getListOfPlatforms().iterator();
		int			minAgentsNumber		= agentsOnThisPlatform - 1;
		String		platformToMoveTo 	= null;
		String		platformName 	   	= null;
		Map			agents				= null;

		iLogger.log(CLASS_NAME, ILogger.TRACE3, methodName, "agents here: " + agentsOnThisPlatform);

		// look for a platform that has less agents
		while (allPlatforms.hasNext()) {

			// ask all platforms for list of agents
			platformName	= (String) allPlatforms.next();
			agents			= getAgentsOnPlatform(platformName);

			// find minimally loaded platform and compare to own load (which must be at least 2 higher)
			if (minAgentsNumber > agents.size()) {
			 	minAgentsNumber = agents.size();
				platformToMoveTo = platformName;
			}
			iLogger.log(CLASS_NAME, ILogger.TRACE3, methodName, "platformname: " + platformName + " no of agents: " + agents.size());
		}

		return platformToMoveTo;
	}

	/** Checks which platform is the least loaded by percentage of CPU Load
	 *  @param		cpu load of local system in percent
	 * 	@return		name of the platform with least load
	 * 	@throws		TimeoutException	   if no response is gotten from AgentManager in time
	 */
	private String lookupPlatformToMoveToByCPU(int cpuLoad)
	throws TimeoutException
	{
		// ask for a list of platforms
		String		methodName   	   	= "lookupPlatformToMoveToByCPU";
		Iterator	allPlatforms 	  	= getListOfPlatforms().iterator();
		int			minCPU				= cpuLoad - 10;
		String		platformToMoveTo 	= null;
		String		platformName 	   	= null;
		int			loadOnPlatform;

		iLogger.log(CLASS_NAME, ILogger.TRACE3, methodName, "load here: " + cpuLoad);

		// look for a platform that has less load
		while (allPlatforms.hasNext()) {

			try {
				// ask all platforms for list of agents
				platformName	= (String) allPlatforms.next();
				loadOnPlatform	= getCPULoadOnPlatform(platformName);

				// find minimally loaded platform and compare to own load (which must be at least 10 percent higher)
				if (minCPU > loadOnPlatform) {
			 		minCPU = loadOnPlatform;
					platformToMoveTo = platformName;
				}
				iLogger.log(CLASS_NAME, ILogger.TRACE3, methodName, "platformname: " + platformName + " load: " + loadOnPlatform);

			} catch (TimeoutException e) {
				iLogger.log(CLASS_NAME, ILogger.ERROR, methodName, "timeout when asking for platform load", e);
			}
		}

		return platformToMoveTo;
	}

	/**
	 *  asks the ass for a list of platforms to which we are synchronized
	 * 	@return list with all names of platforms
	 * 	@throws	   TimeoutException	   if no response is gotten from AgentSynchronizer in time
	 */
	private List getListOfPlatforms()
	throws TimeoutException
	{
		// ask for a list of platforms
		String		methodName   	   	= "getListOfPlatforms";
		// !!! dirty constants
	   	Message		message  	   	   	= new SingleMessage("list_platforms", "ass", new HashMap());
		message.setReplyWith(String.valueOf(getUniqueReplyID()));
		Message		replyMessage    	= iCommunication.sendSynchronousRequest(message, SYNCHRONOUS_TIMEOUT);
		// !!! dirty. do not check subject of incoming message
		Map			platforms   	   	= (Map) replyMessage.getContent();
		Iterator	allPlatforms 	  	= platforms.values().iterator();
		Vector		allPlatformNames	= new Vector();
		Map			singlePlatform   	= null;
		String		platformName 	   	= null;

		// look for an agentname with more than n letters
		while (allPlatforms.hasNext()) {
			singlePlatform = (Map) allPlatforms.next();
			platformName = (String) singlePlatform.get("ipAddress") + "-" +
			 	   	   	   (String) singlePlatform.get("platformId");
			allPlatformNames.add(platformName);
			iLogger.log(CLASS_NAME, ILogger.TRACE3, methodName, "platformname: " + platformName);
		}

		return allPlatformNames;
	}

	/** Moves the specified agent to the specified platform
	 *  @param agent   	name of the agent that should be moved to the other platform
	 * 	@param platform	name of the platform to move the agent to
	 */
	private void moveAgentTo(String agent, String platform)
	{
		String	  methodName   	   	= "moveAgentTo";
		iLogger.log(CLASS_NAME, ILogger.TRACE1, methodName, "moving agent: " + agent + " to platform: " + platform);

		// migrate agent
		iCommunication.sendMessage(new SingleMessage("migrate", agent, platform));
	}

	/**
	 * @return class unique id for reply messages
	 */
	synchronized private long getUniqueReplyID()
	{
		return ++uniqueReplyID;
	}

	/**
	 * Calculates the truth value of the passed property by linearly interpolating between falseValue and trueValue.
	 * If falseValue is less than or equal trueValue, all values less than falseValue are considered to be false, all values
	 * greater than trueValue are considered to be true and all values in between are linearly interpolated.
	 * If falseValue is greater than trueValue, all values greater than falseValue are considered to be false, all values
	 * less than trueValue are considered to be true and all values in between are linearly interpolated.
	 *
	 * @param   property    the source property for which the fuzzy truth value has to be calculated
	 * @param   falseValue  the value up to which the property is considered to be false
	 * @param   trueValue   the value down to which the property is considered to be true
	 * @return              the fuzzy truth value of the property as value in [0..1]
	 */
	private double getLinearFuzzyValue(double property, double falseValue, double trueValue)
	{
		// do not really know where to put this constant
		final double    D_ZERO_P    = 0.000001;

		if (falseValue < trueValue - D_ZERO_P) {
		    // low values are false, big values true
			if (property < falseValue)
				return 0.0;
			else if (property > trueValue)
				return 1.0;
			else
				return (property - falseValue) / (trueValue - falseValue);

		} else if (falseValue > trueValue + D_ZERO_P) {
		    // low values are true, big values false
			if (property < trueValue)
				return 1.0;
			else if (property > falseValue)
				return 0.0;
			else
				return (falseValue - property) / (falseValue - trueValue);

		} else {
		    // true and false value are (almost) the same
			if (property < falseValue)
				return 0.0;
			else
				return 1.0;
		}
	}
}

