#ifndef _TOrderedObjectDLList_h
#define _TOrderedObjectDLList_h 1

#include "Globals.h"
#include "TObjects.h"
#include "PixVec.h"
#include "MemoryBlocks.h"

typedef class TOrderedObjectDLListNode TOrderedObjectDLListNode;
 
#ifndef _NodeRank_typedefs
#define _NodeRank_typedefs 1
typedef unsigned int  (*NodeRankFn)(TOrderedObjectDLListNode& );
#endif

static const byte DLListNodMask1 = ~(~0 << 1);

class TOrderedObjectDLListNode {
  friend const char* node_name(TOrderedObjectDLListNode*);
  friend class TOrderedObjectDLList;
  friend class DLLinkIterator;

  TOrderedObjectDLListNode*         bk;
  TOrderedObjectDLListNode*         fd;
  TOrderedObjectDLListNode**        links;
  int                               nlinks;
  unsigned int                      fIndex;
  byte                              info[32];
  TOrderedObject*                   hd;
  const TOrderedObjectDLList*       oList;

public:

  TOrderedObjectDLListNode( TOrderedObject* h, const TOrderedObjectDLList* mylist, unsigned int i, 
			   TOrderedObjectDLListNode* p=0, TOrderedObjectDLListNode* n = 0 ) 
    : bk(p), fd(n), hd(h), oList(mylist), fIndex(i)
  { 
    links = NULL;
    nlinks=0;
    memset(info,0,32); 
  }
  
  TOrderedObjectDLListNode()  { 
		hd = NULL;
    links = NULL;
    nlinks=0;
    memset(info,0,32); 
  }

  ~TOrderedObjectDLListNode() { if(links) free(links); }

  TOrderedObjectDLListNode* Init( TOrderedObject* h, 
					const TOrderedObjectDLList* mylist, unsigned int i, 
					TOrderedObjectDLListNode* p=0, TOrderedObjectDLListNode* n = 0 )  { 
		bk=p; fd=n; hd=h; oList=mylist; fIndex=i;
		return this;
  }

	inline TOrderedObject& Object() { return *hd; }  

  inline int link ( TOrderedObjectDLListNode* lnode, int lindex ) {
    if( lindex >= nlinks ) {
			int ls = sizeof(TOrderedObjectDLListNode*);
      if(links) {
				TOrderedObjectDLListNode** newlinks =  new TOrderedObjectDLListNode* [lindex+4];
				memset(newlinks,0,(lindex+4)*ls);
				memcpy( newlinks, links, nlinks*ls );
				delete[] links; nlinks = lindex+4;
				links = newlinks;
      }
      else {
				links =  new TOrderedObjectDLListNode* [nlinks=lindex+4];
				memset(links,0,nlinks*ls);
			}
    }
    links[lindex] =  lnode;
    return lindex;
  } 

  inline int unlink ( int lindex ) { 
    links[lindex] = 0;
    return lindex;
  }

  inline int slink ( TOrderedObjectDLListNode* lnode, int startIndex=0 ) {
    int lindex=startIndex;
    while(1) {
      if( lindex >= nlinks ) {
				int ls = sizeof(TOrderedObjectDLListNode*);
				if(links) {
					TOrderedObjectDLListNode** newlinks = new TOrderedObjectDLListNode* [lindex+4];
					memset(newlinks,0,(lindex+4)*ls);
					memcpy(newlinks,links,nlinks*ls);
					delete[] links; nlinks = lindex+4;
					links = newlinks;
				}
				else {
					links = new TOrderedObjectDLListNode* [nlinks=lindex+4];
					memset(links,0,nlinks*ls);
				}
      }
      if( links[ lindex ] == lnode ) return lindex;
      else if( links[ lindex ] == 0 )   {
				links[lindex] = lnode;
				return lindex;
      }
      lindex++;
    }
  }

  inline int unlink ( TOrderedObjectDLListNode* lnode ) {
    int lindex=0;
    while(1) {
      if( lindex == nlinks ) { return 0; }
      if( links[ lindex ] == lnode )   {
	links[lindex] = 0;
	return 1;
      }
      lindex++;
    }
  }

  inline int NLinks() { return nlinks; }

  inline TOrderedObject& link ( byte lindex ) const {
    if( lindex >= nlinks ) {  gFatal("Bad Link."); }
    return *links[lindex]->hd;
  }

   inline  TOrderedObjectDLListNode* tlink ( byte lindex ) const {
    if( lindex >= nlinks ) { gFatal("Bad Link.");  }
    return links[lindex];
  }

  inline Pix plink ( byte lindex ) const {
    if( lindex < nlinks ) { return Pix( links[lindex] ); }
    else return 0;
  }

 inline unsigned int GetMaxIndex ( ) const {
   unsigned int maxIndex = 0;
   TOrderedObjectDLListNode**  l = links;
   TOrderedObjectDLListNode**  e = links+nlinks;
   do {
     if( (*l)->fIndex > maxIndex ) maxIndex = (*l)->fIndex;
   } while ( ++l != e );
   return maxIndex;
  }

 inline unsigned int GetMinIndex ( ) const {
   unsigned int minIndex = UINT_MAX;
   TOrderedObjectDLListNode**  l = links;
   TOrderedObjectDLListNode**  e = links+nlinks;
   do {
     if( (*l)->fIndex < minIndex ) minIndex = (*l)->fIndex;
   } while ( ++l != e );
   return minIndex;
  }

  inline unsigned int Index() { return fIndex; }
  inline int SetIndex(unsigned int index) { return (fIndex = index); }

  inline const TOrderedObjectDLList* List() { return oList; }

  inline void set_info( byte index ) { 
    info[index >> 3] |= ( 01 << (index & 07) ); 
  }
  inline void clear_info( byte index ) { 
    info[index >> 3] &= ~( 01 << (index & 07) ); 
  }
  inline byte test_info( byte index ) const { 
    return ( ( info[index >> 3] >> (index & 07) ) & DLListNodMask1 ); 
  }

  inline void Dump(FILE* oFile=NULL) const { 
    CString cTemp; int i;
    sprintf(gMsgStr," L%d:{ ",hd->Index()); cTemp += gMsgStr;
    for(i=0; i<nlinks; i++) { sprintf(gMsgStr,"%p ",links[i]); cTemp += gMsgStr; }
    sprintf(gMsgStr,"} : { "); cTemp += gMsgStr;
    for(i=0; i<32; i++) { sprintf(gMsgStr,"%x ",info[i]); cTemp += gMsgStr; } 
#ifdef DEBUG   
    sprintf(gMsgStr,"} { "); cTemp += gMsgStr;
    for(i=0; i<8; i++) { sprintf(gMsgStr,"%x ",test_info(i)); cTemp += gMsgStr; } 
#endif   
    sprintf(gMsgStr,"}\n"); cTemp += gMsgStr;
    if(oFile) fprintf(oFile,"%s",cTemp.chars());  
    else gPrintLog(cTemp.chars()); 
  }
};

class DLLinkIterator {
  TOrderedObjectDLListNode* fNode;
  int fCurrent;
public:
  DLLinkIterator(TOrderedObjectDLListNode* n) { fNode = n; }
  DLLinkIterator( Pix p ) { fNode = (TOrderedObjectDLListNode*)p; fCurrent=0; }
  inline Pix first() { return ( fNode && fNode->nlinks > 0 ) ? (Pix) fNode->links[fCurrent=0] : (Pix)0; }

  inline Pix next() { 
    ++fCurrent; 
    if( fNode->nlinks > fCurrent ) 
      return (Pix) fNode->links[fCurrent]; 
    else 
      return (Pix)0; 
  }

  inline Pix index(int i) { return ( fNode && fNode->nlinks > i ) ? (Pix) fNode->links[fCurrent=i] : (Pix)0; }
  inline int index() { return fCurrent; }
  inline int test_info() { return fNode->test_info(fCurrent); }
  inline int set_info() { fNode->set_info(fCurrent); return fCurrent; }
  inline int NLinks() { return ( fNode ) ? fNode->nlinks : 0; }
};

class TOrderedObjectDLList {
  friend class          TOrderedObjectDLListTrav; 

  TOrderedObjectDLListNode*        h;
  int                              mIndex;
  MemMgr<TOrderedObjectDLListNode> _MemMgr;
  
protected:

  Bool 														checkPixOwnership;


public:

  enum ESortType { kPixIndex, kObjectIndex };

	TOrderedObjectDLList(int initialSize=32);
	virtual ~TOrderedObjectDLList();

  TOrderedObjectDLList&            operator = (const TOrderedObjectDLList& a);
  TOrderedObjectDLList&            operator += (const TOrderedObjectDLList& a);
  int                              CopyToPixVec( PixVec& a, int start = 0 );
  int                              CopyFromPixVec( PixVec& a, int start = 0 );

  inline int            empty() const { return h == 0; }
  int                   length() const;
	inline int 						one_node() const { return  ( h && (h == h->fd) ); }

  void                  clear();
  void                  free();

  Pix                   prepend(TOrderedObject& item);
  Pix                   append(TOrderedObject& item);
  Pix                   append(TOrderedObject& item, unsigned int index);
  void                  join(TOrderedObjectDLList&);
  Pix                   move( Pix pt, Pix pm0, Pix pm1=0 );  // Move pm0-pm1 in front of pt
  void                  swap( Pix p0, Pix p1 ); 

  TOrderedObject&       front() const;
  TOrderedObject*       firstObject() const;
  TOrderedObject&       remove_front();
  void                  del_front(int freeObj=0);

  TOrderedObject&       rear() const;
  TOrderedObject&       remove_rear();
  void                  del_rear(int freeObj=0);

  TOrderedObject&                  operator () (Pix p) const;
  TOrderedObject&                  elem (Pix p) const;
  inline int            index (const Pix p) const;
  inline int            set_index (Pix p, int index) const;
  inline Pix            first() const { return Pix(h); }
  inline Pix            last() const { return (h == 0)? 0 : Pix(h->bk); }
  inline Pix           next(Pix& p) const { 
                           p = (p == 0 || p == h->bk)? 0 : Pix(((TOrderedObjectDLListNode*)p)->fd); return p; }
  inline Pix           prev(Pix& p) const {
                           p = (p == 0 || p == h)? 0 : Pix(((TOrderedObjectDLListNode*)p)->bk); return p; }
  Pix                   ins_after(Pix p, TOrderedObject& item);
  Pix                   ins_before(Pix p, TOrderedObject& item);
  Pix                   find( TOrderedObject* item ) const;
  Pix  									find( int index ) const;
  Pix                   findBefore( Pix p, TOrderedObject* item ) const;
  Pix                   findAfter( Pix p, TOrderedObject* item ) const;
  Pix                   find( TOrderedObject& item ) const;

  void                  del(Pix& p, int dir = 1, int free_obj=0);
  void                  del_after(Pix p, int free_obj);
  void                  del_next_node(Pix& p, int free_obj=0);
  void                  del_upto ( Pix p, int free_obj);

  void                  error(char* msg) const;
  int                   OK() const;
  int                   owns(Pix p) const;

  int                   link ( Pix p0, Pix pL, byte lindex );   // Creates link p0 -> pL via link lindex 
  int                   slink ( Pix p0, Pix pL, int startIndex=0 );    // creates link from p0 -> pL using first available slot after startIndex
 TOrderedObject&       link ( Pix i, byte lindex ) const;
  Pix                   plink ( Pix i, byte lindex ) const;

  void                  subset_add( Pix p, byte setIndex );
  void                  subset_delete( Pix p, byte setIndex );
  byte                  subset_contains( Pix p, byte setIndex ) const;
  void                  DumpLink( Pix  p, FILE* oFile ) const;
  void                  DumpLinks(FILE* ofile, char* lable );
  int                   sort( NodeRankFn nrf );
  int                   qsort( NodeRankFn nrf );
  int                   is_sortable( NodeRankFn nrf );
  int                   sort( ESortType st = kPixIndex );
	inline void           SetOwnershipCheck( Bool val ) { checkPixOwnership = val; }

  virtual inline TOrderedObjectDLListNode* GetNode(TOrderedObject* obj, const TOrderedObjectDLList* mylist, unsigned int i, 
			   TOrderedObjectDLListNode* p=0, TOrderedObjectDLListNode* n = 0) {
    return _MemMgr.Allocate()->Init(obj,mylist,i,p,n);
  }

  inline int index_test( ESortType st, TOrderedObjectDLListNode* u0, TOrderedObjectDLListNode* u1 ) {
    if( st == kPixIndex ) { return ( u0->Index() > u1->Index() ); }
    else { return ( u0->hd->Index() > u1->hd->Index() ); }
  }

};


inline TOrderedObjectDLList::~TOrderedObjectDLList()
{
  clear();
}

inline TOrderedObjectDLList::TOrderedObjectDLList(int initialSize) : _MemMgr(initialSize)
{
  h = 0; mIndex = 0; 	checkPixOwnership = (gDebug) ? True : False;
}

inline  int  TOrderedObjectDLList::owns( Pix p ) const
{
  if (p == 0) return 0;
  else return (((TOrderedObjectDLListNode*)p)->oList == this);
}

inline TOrderedObject& TOrderedObjectDLList::operator () (Pix p) const
{
#ifdef DEBUG
  if (p == 0) error("null Pix");
  if(  checkPixOwnership && !owns(p) ) error( " Pix not owned by Set" );
#endif
  return *((TOrderedObjectDLListNode*)p)->hd;
}

inline TOrderedObject& TOrderedObjectDLList::elem (Pix p) const
{
#ifdef DEBUG
  if (p == 0) error("null Pix");
  if( checkPixOwnership && !owns(p) ) error( " Pix not owned by Set" );
#endif
  return *((TOrderedObjectDLListNode*)p)->hd;
}

inline int TOrderedObjectDLList::index (const Pix p) const
{
#ifdef DEBUG
  if (p == 0) error("null Pix");
  if(  checkPixOwnership && !owns(p) ) error( " Pix not owned by Set" );
#endif
  return ((TOrderedObjectDLListNode*)p)->Index();
}
inline int TOrderedObjectDLList::set_index (Pix p, int index) const
{
#ifdef DEBUG
  if (p == 0) error("null Pix");
  if(  checkPixOwnership && !owns(p) ) error( " Pix not owned by Set" );
#endif
  return ((TOrderedObjectDLListNode*)p)->SetIndex(index);
}

inline void TOrderedObjectDLList::DumpLink( Pix  p, FILE* oFile ) const 
{
#ifdef DEBUG  
  if (p == 0) error("null Pix");
  if(  checkPixOwnership && !owns(p) ) error( " Pix not owned by Set" );
#endif
  ((TOrderedObjectDLListNode*)p)->Dump(oFile);
}

inline TOrderedObject& TOrderedObjectDLList::front() const
{
  if (h == 0) error("front: empty list");
  return *h->hd;
}

inline TOrderedObject* TOrderedObjectDLList::firstObject() const
{
  if (h == 0) { return NULL; }
  else { return h->hd; }
}

inline TOrderedObject& TOrderedObjectDLList::rear() const
{
  if (h == 0) error("rear: empty list");
  return *h->bk->hd;
}

inline int TOrderedObjectDLList::link ( Pix p0, Pix pL, byte lindex )
{
#ifdef DEBUG
  if (p0 == 0) error("null Pix");
  if(  checkPixOwnership && !owns(p0) ) error( " Pix not owned by Set" );
#endif
  TOrderedObjectDLListNode* lNode =  (TOrderedObjectDLListNode*)(pL);
  return ((TOrderedObjectDLListNode*)(p0))->link(lNode,lindex);
}

inline void TOrderedObjectDLList::DumpLinks(FILE* ofile, char* label ) {
  TOrderedObjectDLListNode* n = h;
  fprintf(ofile,"\n%s:",label);
  if(n==NULL) { fprintf(ofile,"\nempty."); return; }
  while(1) {
    fprintf(ofile,"\n%p: >%p <%p : %p",n,n->fd,n->bk,n->hd);
    n = n->fd;
    if( n == h ) break;
  }
  fflush(ofile);
}
inline int TOrderedObjectDLList::slink ( Pix p0, Pix pL, int startIndex )  // creates link from p0 -> pL using first available slot after startIndex
{
#ifdef DEBUG
  if (p0 == 0) error("null Pix");
  if(  checkPixOwnership && !owns(p0) ) error( " Pix not owned by Set" );
#endif
  TOrderedObjectDLListNode* lNode =  (TOrderedObjectDLListNode*)(pL);
  return ((TOrderedObjectDLListNode*)(p0))->slink(lNode,startIndex);
}
 
inline  TOrderedObject& TOrderedObjectDLList::link ( Pix p0, byte lindex ) const
{
#ifdef DEBUG
  if (p0 == 0) error("null Pix");
  if(  checkPixOwnership && !owns(p0) ) error( " Pix not owned by Set" );
  return ((TOrderedObjectDLListNode*)(p0))->link(lindex);
#else
  return *(((TOrderedObjectDLListNode*)(p0))->links[lindex])->hd;
#endif
}

inline Pix  TOrderedObjectDLList::plink ( Pix p0, byte lindex ) const 
{
#ifdef DEBUG
  if (p0 == 0) error("null Pix");
  if(  checkPixOwnership && !owns(p0) ) error( " Pix not owned by Set" );
#endif
  return ((TOrderedObjectDLListNode*)(p0))->plink(lindex);
}

inline  void TOrderedObjectDLList::subset_add( Pix p, byte setIndex ) {
#ifdef DEBUG
  if (p == 0) error("null Pix");
  if(  checkPixOwnership && !owns(p) ) error( " Pix not owned by Set" );
#endif
  ((TOrderedObjectDLListNode*)(p))->set_info(setIndex);
}

inline  void TOrderedObjectDLList::subset_delete( Pix p, byte setIndex ) {
#ifdef DEBUG
  if (p == 0) error("null Pix");
  if(  checkPixOwnership && !owns(p) ) error( " Pix not owned by Set" );
#endif
  ((TOrderedObjectDLListNode*)(p))->clear_info(setIndex);
}

inline  byte TOrderedObjectDLList::subset_contains( Pix p, byte setIndex ) const {
#ifdef DEBUG
  if (p == 0) error("null Pix");
  if(  checkPixOwnership && !owns(p) ) error( " Pix not owned by Set" );
#endif
  return ((TOrderedObjectDLListNode*)(p))->test_info(setIndex);
}

#endif
