//----------------------------------------------------------------------------------------
  //	Env.cc
  //	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
  //	Change History:
  //----------------------------------------------------------------------------------------
#include "Environ.h"
#include "Utilities.h"
#include "viewserver.h"
#include "controlsocket.h"

#ifdef HAS_SOCKETS

extern "C" {
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <unistd.h>
#include <arpa/inet.h>

// int gethostname( char* name, int nl );
}
#endif

extern "C" {
#ifdef HAS_HDF
#include "mfhdf.h"
#else
#include "sme_types.h"
#endif
}

#include <stdarg.h>
#define buildmsg() char msg[256];va_list argptr;va_start(argptr,fmt);vsprintf(msg,fmt,argptr);va_end(argptr)

enum EEnvirTags { kScreenTag };
char gMsgStr[MAX_MSG_SIZE];

#ifdef DEBUG
int gDebug = 2;    // initial value, overridden by d(...) command in Config file.
#else
int gDebug = 0;
#endif

int gWildCardIndex = -99;
byte* gWatchPtr = NULL;
FILE* gLogFile = NULL;
int gSpeedOp = 0;
int gGlobalFileSystem = 1;

#ifdef USE_MPI  
int gMPI = 1;
#else 
int gMPI = 0;
#endif

int gNProc=1;
int gIProc=0;
int gInterface=0;

byte Env::fEnvInfo[8] = { 0,0,0,0,0,0,0,0 };
byte Env::fByteOrdering;
byte* Env::fMPI_buffer = NULL;
int Env::fMPI_buffer_size = 0;
FILE* Env::fLogFile = NULL;
FILE* Env::fOstream;
FILE* Env::fEstream;
FILE* Env::fTimerFile = NULL;

ByteBuffer Env::fTemp(100);
int Env::fStopCnt = 0;

CPathString Env::fModelPath;
CPathString Env::fTmpPath;
CPathString Env::fDataPath;
CPathString Env::fConfigPath;
CPathString Env::fArchivePath;
CString Env::fTclFile;
CString Env::fProjName; 
CString Env::fModelName;
CString Env::fScenarioName;
CString Env::fErrorString;
CString Env::fHDFInFile;
CString Env::fDebugger;
CString Env::fProgram;
CString Env::fDisplay;
CString Env::fTclRunScript;
ArgArray Env::fArgv;
ArgArray Env::fInfoMessages;
ArgArray Env::fWarnMessages;
ArgArray Env::fErrorMessages;
ArgArray Env::fInfoBuffer;
ArgArray Env::fWarnBuffer;
ArgArray Env::fErrorBuffer;
TAssoc_array Env::fLocalEnvVars;
TAssoc_array Env::fGlobalEnvVars;

// network parameters
CString Env::iHost;
int     Env::iPort;
int     Env::iTimeout = 2;
int     Env::iSockBufsize = 4096;
int     Env::iSocket = -1;
int     Env::iSync = 0;
int     Env::iReplicationIndex = 0;
int     Env::iBatchIndex = -1;

/*
clock_t Env::fAccumTime[10]= { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
clock_t Env::fGlobalTime = 0;
clock_t Env::fLocalTime = 0;
*/

double Env::fCommTime = 0;
double Env::fCommStartTime = 0;

/*
struct tms Env::_global_start_time;
struct tms Env::_global_end_time;
struct tms Env::_local_start_time;
struct tms Env::_local_end_time;
*/

// use interpreter instead of CL monitor
Bool    Env::bUseTcl = FALSE;
Bool    Env::bUsePython = FALSE;

// do not read configuration with TCL interpreter if FALSE (bound to -noconf switch)
Bool    Env::bReadConf = TRUE;
Bool    Env::bUseDebugger = FALSE;
Bool    Env::bUseMPE = FALSE;
Bool    Env::bUseTimer = FALSE;
Bool    Env::bGlobalSort = TRUE;
int     Env::iPauseLevel = 1;
Bool    Env::bUseJavaSocket = FALSE;
Bool    Env::bAttachDebugger = FALSE;

// shared (mmap) data space
void*  Env::iDataRep = NULL;
// size of data 'swap' area (def: 128k)
size_t Env::iDataSize = 131072;

#ifdef USE_MPI
MPI_Comm Env::fComm;
#endif

int Env::debugger_stop_here() { fStopCnt++; return 0;}
char * Env::_program_name = NULL;

int Env::Socket(int listen_for_connection )
{
#ifdef HAS_SOCKETS
  struct sockaddr_in sin;
  const char* host = (const char*)Env::Host();
  
  if (iSocket == -1) {
    iPort = 0;
    iSocket = socket(AF_INET, SOCK_STREAM, 0);  
    // tbi set options
    // set sockaddr for bind()
    memset((char*)&sin, '\0', sizeof(sin));
    
    if ((sin.sin_addr.s_addr = inet_addr((char*)host)) == -1)  {
      hostent* hp = gethostbyname((char*)host);
      if (hp != NULL) 
	{
	  memcpy(&(sin.sin_addr), hp->h_addr, hp->h_length);
	  sin.sin_family = hp->h_addrtype;
	}
    }
    else sin.sin_family = AF_INET;  
    
    if (bind(iSocket, (sockaddr*)&sin, sizeof(sin)) >= 0) {
      struct sockaddr_in ls;
      int l = sizeof(struct sockaddr_in);
      if (getsockname(iSocket, (sockaddr*)&ls, (socklen_t*) &l) != -1) 	{
	// set local host 
	static char hostname[64];
	iPort = ntohs(ls.sin_port);
	gethostname(hostname, 63);
	iHost = hostname;
      }
    }
    // listen to connections
    if ( listen_for_connection ) {
      if( listen(iSocket, 5) < 0) 
	iPort = 0;
    }
    sprintf(gMsgStr,"***************** socket %d listening\n",iSocket); gPrintScreen();
  }
  return iSocket;
#endif
  return 0;
}

int Env::Listen() {
#ifdef HAS_SOCKETS
  if( listen(iSocket, 4) < 0) { return 0; }
#endif
  return iSocket;
}

void Env::PauseForResponse( ) { 
		if( bUseDebugger || ( gDebug < iPauseLevel ) || bUseTcl ) return;
		printf("\npause: Please type <Return> to continue\n");
		fflush(stdout);
		getchar();
}

inline byte get_byte_ordering() {
	int32 test = 1;
	byte* btest = (byte*)(&test);
	if( btest[3] == 1 ) {  return 1; }
	else if( btest[0] == 1 ) { return 0; }
	else { gPrintErr( " Unrecognized integer format! " ); return 1; }
}


void Env::BCastTest(int test_index) {
#ifdef USE_MPI 
  if( gMPI ) {
	int test = gIProc;
	MPI_Bcast( &test, 1, MPI_INT, 0, MPI_COMM_WORLD );
	if( gIProc == 0 ) {
	  fprintf(stderr, "\ninfo  Bcast test %d initiated\n",test_index); 
	} else {
	  fprintf(stderr, "\ninfo ************ P(%d): Bcast test %d result: %d\n",gIProc,test_index,test); 
	}
	if( test != 0 ) { gFatal("BCastTest Failure"); }
  }
#endif
}
 
void Env::Startup( int  argc, char* argv[], EAppType type ) {

  Bool redirectScreenOutput = FALSE;
  SetInfo(kAppType, type );
  fOstream = stdout;
  fEstream = stderr;
  fDebugger = "gdb";
  fScenarioName = "def";
  fByteOrdering = get_byte_ordering();
    
#ifdef USE_MPI  
  if( gMPI ) {
//		sleep(60);
	int flag;
	int err = MPI_Initialized(&flag);
	if( flag == 0 ) { 
	  MPI_Init(&argc,&argv);
	}
//	SME_set_error_break();
    int size;
    MPI_Comm_size(MPI_COMM_WORLD,&size);
    gNProc = size;
    int rank;
    MPI_Comm_rank(MPI_COMM_WORLD,&rank);
    gIProc = rank;
    int test = rank;
	MPI_Bcast( &test, 1, MPI_INT, 0, MPI_COMM_WORLD );
    fprintf( stderr, "\ninfo: P(%d): Initialized MPI: %d procs, Bcast test result: %d",rank,size,test); 
  }
#endif

  if( gIProc == 0 ) {      
    for( int i = 0; i < argc; i++ ) {  
      if(argv[i] == NULL) break; 
      fArgv.Add(argv[i]);     
    }    
  }	
  
#ifdef USE_MPI 
  if( gMPI ) { fArgv.BroadCast(); } 
#endif
  
  fHDFInFile = "SimData.hdf";    
  int debug=gDebug, pause = 0;
  int Argc = fArgv.NArgs();
  int use_system_classpath = 0;

  for(int i=0; i<Argc; i++) {

    if( fArgv[i] == "-d" ) {
      bUseDebugger = TRUE;
      if( (i+1) < argc && Util::is_integer(fArgv[i+1]) )  { debug=atoi(fArgv[i+1]); } 
    }  else  if( fArgv[i] == "-D" ) {
      bUseDebugger = TRUE; debug=1;
      if( (i+1) < argc )  { fDebugger = fArgv[i+1]; } 
    } else if( fArgv[i] == "-i" ) { 
      gInterface=1; 
    }
    else if (fArgv[i] == "-quiet") {
			redirectScreenOutput = TRUE;
		}
    else if (fArgv[i] == "-java") {
      bUseJavaSocket = TRUE;
#ifdef USE_TCL
      bUseTcl = TRUE; 
#endif
      if ((i+1) < Argc) { Env::Host() = fArgv[++i]; }
    }
	else if (fArgv[i] == "-mpe") {
      bUseMPE = TRUE;
    }
    else if (fArgv[i] == "-notty") {
      /* not using a terminal (run from pipe):
	 disable line buffering on stdin/stdout */
      setvbuf(stdin,  NULL, _IOLBF, 0);
      setvbuf(stdout, NULL, _IOLBF, 0);
    }
    else if ( fArgv[i] ==  "-p" ) {
      if ((i+1) < Argc) {
		fProjName = fArgv[++i];
		sprintf( gMsgStr , "Setting Project Name to  %s ",fProjName.chars()); gPrintScreen(gMsgStr,True);
	  }
    }
    else if ( fArgv[i] ==  "-errstop" ) {
      iPauseLevel = 0;
      gPrintScreen("Pausing at Warnings");
    }
    else if ( fArgv[i] ==  "-nostop" ) {
      iPauseLevel = 1000;
      gPrintScreen("Disabling pausing at Warnings");
    }
    else if ( fArgv[i] ==  "-use_classpath" ) {
      use_system_classpath = 1;
      gPrintScreen("Using system classpath");
    }
    else if ( fArgv[i] ==  "-pause" ) {
      if ((i+1) < Argc ) {
	if( Util::is_integer(fArgv[i+1]) )  { pause = atoi(fArgv[i+1]); } 
	else pause = 60;
      }
    }
    else if ( fArgv[i] == "-m" ) {
      if ((i+1) < Argc) fModelName = fArgv[++i];
    }
    else if ( fArgv[i] == "-scen" ) {
      if ((i+1) < Argc) fScenarioName = fArgv[++i];
    }
    else if ( fArgv[i] == "-ppath" ) {
      if ((i+1) < Argc) { fModelPath.Add(fArgv[++i]); }
    }
    else if ( fArgv[i] == "-tpath" ) {
      if ((i+1) < Argc) { fTmpPath.Add(fArgv[++i]); } 
    }
    else if (fArgv[i] == "-tcl") {
      gPrintScreen("Using TCL interpreter");
      bUseTcl = 1;
    }
    else if (fArgv[i] == "-python") {
      gPrintScreen("Using python interpreter");
      bUsePython = 1;
    }
    else if (fArgv[i] == "-debug") {
      bAttachDebugger = TRUE;
    }
    else if (fArgv[i] == "-exec") {
      bUseTcl = 1;
      if ((i+1) < argc) { 
		fTclFile = argv[++i]; 
	  }
    }
    else if (fArgv[i] == "-noconfig") {
      gPrintScreen("Reading of default config file disabled\n");
      bReadConf = 0;
    }
    else if (fArgv[i] == "-batch") {
      gPrintScreen("Running in batch mode\n");
      iBatchIndex = 0;
    }
    else if( fArgv[i] == "-help" ) {
      gPrintScreen("SME generic command line Arguments:");
      gPrintScreen(" -p NAME        Use NAME as project Name (default: $ProjName)");
      gPrintScreen(" -m NAME        Use NAME as model Name (default: $ModelName)");
      gPrintScreen(" -scen NAME     Use NAME as scenario Name (default: def)");
      gPrintScreen(" -ppath PATH    Use PATH as path as path to project directory (default: $SME3_PROJ)");
      gPrintScreen(" -tpath PATH    Read/write all files to/from PATH");
      gPrintScreen(" -d INT         Set debug level to INT (default: 0) ");
      gPrintScreen(" -D NAME        Set use NAME as debugger (gdb/dbx) ");
      gPrintScreen(" -i             Use CME interface (SME-GUI)");
      gPrintScreen(" -tcl           Use tcl interface");
      gPrintScreen(" -exec FILE     Execute TCL file (implies -tcl)");
      gPrintScreen(" -noconfig      Do not read config files");
      return;
    }  
  }

	fProgram = fArgv[0];
	fDisplay = SME_get_root_display(gIProc);

	if( bAttachDebugger ) {
		Env::AttachDebugger( NULL );
	}

	if( pause > 0 ) { 
#ifdef HAS_SOCKETS
		pid_t pid = getpid();  
		sprintf(gMsgStr, "info: Paused for %d seconds, attach debugger to process %d\n",pause,pid);
		gPrintScreen(gMsgStr,TRUE);
		sleep(pause);
#endif
		debugger_stop_here();
	}
  
#ifdef USE_MPI 
  if(gMPI) { 
#ifdef DEBUG
    if( bUseDebugger ) {
			int dbg_proc = -1;
			char* tmp = GetEnv("DBG_PROC");
			if( (tmp != NULL) && Util::is_integer(tmp) ) { 
				dbg_proc = atoi(tmp);
			}
			if( ( dbg_proc < 0 ) || ( dbg_proc == gIProc ) ) {
				fprintf(stderr,"\ninfo: (P%d) Starting debugger %s in xterm",gIProc,fDebugger.chars()); 
				if( fDebugger == "gdb" ) {
					SME_Start_gdb_in_xterm ( fProgram, fDisplay );
					SME_Errors_print(MPI_COMM_WORLD,fProgram,fDisplay);
				} else {
					SME_Start_dbx_in_xterm ( fProgram, fDisplay );
					SME_Errors_print(MPI_COMM_WORLD,fProgram,fDisplay);
				}
			}
    } else {
//		SME_Errors_dump_to_debugger(MPI_COMM_WORLD,fProgram,fDisplay,fDebugger);
		SME_Errors_print(MPI_COMM_WORLD,fProgram,fDisplay);
    }
#endif
    int err; 
    if( err = GetComm(&fComm,0) ) { 
      fprintf(stderr,"\nError %d in Env::Startup\n",err); 
      exit(err); 
    } 
  }  
#endif
  fEnvInfo[kUseDebugger] = bUseDebugger;
  SetDebugLevel(debug);
  SetEnvVars();

  sprintf(gMsgStr,"%s/Driver.Log.%d", fArchivePath.chars(), gMyID() );
  fLogFile = fopen(gMsgStr,"w");
  if(fLogFile==NULL) {
    if(gMyID()==0) fprintf(stderr,"\nWARNING: Unable to open Log file Driver.Log.%d\n",gMyID()); 
    fEnvInfo[kLog] = 0;
  }
  else {
		gLogFile = fLogFile;
		if( redirectScreenOutput ) {
			  fOstream = fEstream = fLogFile;
		}
		fEnvInfo[kLog] = 1;
	}
	
//    gPrintScreen(" *** Spatial Modeling Environment, Copyright (C) 1995 (TXU-707-542), Tom Maxwell");
//    gPrintScreen(" *** SME comes with ABSOLUTELY NO WARRANTY");
//    gPrintScreen(" *** This is free software, and you are welcome to redistribute it");
//    gPrintScreen(" *** under the terms of the GNU General Public License.");
	
#ifdef JAVA
	if( gIProc==0 ) {
	  if( bUseJavaSocket ) { JavaSocketReader::init( argc, argv ); }
	  ViewServer::init( argc, argv );
	  ViewServer::setLogStream(fLogFile); 
	}
#endif
	if( gMPI ) {  fprintf(stderr,"\nP%d: Done Env::Startup",gIProc); }
}

int Env::AttachDebugger( const char* display ) {	
		sprintf(gMsgStr,"\ninfo: (P%d) Starting debugger.",gIProc); gPrintLog(); fprintf(stderr,"%s",gMsgStr);
		int rv;
		if( display == NULL ) display = fDisplay.chars(); 
		if( fDebugger == "gdb" ) {
			rv = SME_Start_gdb_in_xterm ( fProgram, display );
		} else if ( fDebugger == "dbx" ) {
			rv = SME_Start_dbx_in_xterm ( fProgram, display );
		}
		gPrintLog(SME_debug_get_result_string());
		return rv;
}

void Env::Finalize() {	

#ifdef USE_MPI
  if(gMPI) MPI_Finalize();
#endif

 }

int Env::FindEnvValue(FILE *infile)
{
  int ch; int got_eq = 0;
  
  while( isspace(ch=fgetc(infile)) || ch == '=' ) { 
    if( ch == '=' ) got_eq = 1;
  }
  if( (ch == '\n' && !got_eq) || ch==EOF ) return 0;
  ungetc(ch,infile);
  return 1;		
} 

char* Env::GetEnvValFromFile( CPathString fileName, const char* varName ) {

  CString envVal;
  FILE *infile = fopen(fileName,"r");
  if(infile == NULL) { 
    return NULL; 
  }
  int found=0;
  while( !found ) {
    if( Util::scan_forward(infile,varName) == 0 ) { 
      fprintf(stderr,"Warning: Can't can't find environment variable: %s\n",varName); 
      fflush(stderr);  
      return 0; 
    }
    if( FindEnvValue(infile) ) {
      char ch; 
      while( !isspace(ch=fgetc(infile)) ) { envVal += ch; }
      fclose(infile);
      found = 1;
    }
  }
  if(found) {
    int l = envVal.length();
    char* rv = new char[l+1];
    Util::ncopy0(envVal.chars(), rv, l);
    return rv;
  } else return NULL;
}

void Env::ReadEnvValsFromFile( CPathString fileName ) {

  CString key, value;
  FILE *infile = fopen(fileName,"r");
  if(infile == NULL) { 
    return; 
  }
  int ch; 
  int more_data=1; 
  while( more_data ) {
	int got_key = 0;
	while( isspace(ch=fgetc(infile))) {;} 
	if( ch==EOF ) more_data = 0;
	else {
	  key.clear(); value.clear();
	  while( (ch != '\n') && (ch != '=') && (ch != EOF) ) { key += (char)ch; ch=fgetc(infile); }
	  key.trim(); 
	  if( (ch == '=') && !key.empty() ) { got_key = 1; }
	  while( ((ch=fgetc(infile)) != '\n') && (ch != EOF) ) { value += (char)ch; }
	  value.trim();
	  if( got_key  && !value.empty() ) {
		int cutindex0=0, cutindex1;
		CString sval, skey;
		while( (cutindex0 = value.index("${",cutindex0)) >= 0 ) {
		  if( (cutindex1 = value.index("}",cutindex0)) >= 0 ) {
			CString subkey( value.sub(cutindex0+2,cutindex1) ); 
			if( !subkey.empty() ) {
			  sval = GetEnv(subkey);
			  skey = "${" + subkey + "}";
			}
		  }
		  if( sval.empty() ) {
			gPrintErr( format( "Illegal syntax in Environment File: %s %s", key.chars(), value.chars() ) );
		  } else {
			value.gsub(skey,sval);
		  }
		}
		fLocalEnvVars.add( key, new CString(value) ); 
		gPrintScreen( format( "Adding local Environment Variable: %s = %s ", key.chars(), value.chars() ) );
	  } else {
		 gPrintErr( format( "Illegal line in Environment File: %s %s", key.chars(), value.chars() ) );
	  }
	}
  }  
}
 
char* Env::GetEnv(const char* varName) {

  char* data = NULL;
  if( gMyID() == 0 || gGlobalFileSystem ) {
	CString* gcdata = (CString*) fGlobalEnvVars.objptr(varName);
	CString* lcdata = (CString*) fLocalEnvVars.objptr(varName);
	if( gcdata != NULL ) { data = (char*)gcdata->chars(); }
	else if (lcdata != NULL ) { data = (char*)lcdata->chars(); }
	else {
	  data = getenv(varName); 
	  if( data == NULL ) {
		CPathString globalEnvFile( SME_LIBRARY ); 
		globalEnvFile.Add("Environment");
		data = GetEnvValFromFile( globalEnvFile, varName );
		if( data != NULL ) { 
		  fGlobalEnvVars.add( varName, new CString(data) ); 
		  gPrintScreen( format( "Adding global Environment Variable: %s = %s ",varName,data) );
		}
		else {
		  CPathString localEnvFile( Env::DataPath() ); 
		  localEnvFile.Add("Environment");
		  data = GetEnvValFromFile( localEnvFile, varName );
		  if( data != NULL ) { 
			fLocalEnvVars.add( varName, new CString(data) ); 
			gPrintScreen( format( "Adding local Environment Variable: %s = %s ",varName,data) );
		  }
		}
	  }
	}
  }
  int size = Util::slen(data);
 
  if( !gGlobalFileSystem  && gMPI ) {
#ifdef USE_MPI
	if( gMPI ) {
	  MPI_Bcast( &size, 1, MPI_INT, 0, fComm );
	  if( size > 0 ) {
		if( gMyID() > 0 ) { data = new char[size+1]; }
		MPI_Bcast( data, size+1, MPI_CHAR, 0, fComm );
	  }
	}
#endif
  }
  if(gDebug) { 
    if( data ) { sprintf(gMsgStr,"\nDoing GetEnv: name = %s,data = %s",varName,data); gPrintLog(); }
    else { sprintf(gMsgStr,"\nGetEnv: Variable not found: %s",varName); gPrintLog(); }
  }
  return data;
}

void Env::PrintScreen( const char* astr, int async ) {
  const char* str = (astr) ? astr : gMsgStr;
  if( (gIProc == 0) || async ) {
		if( !ArchiveMessage( str, kInfo, async  ) ) {
			fputs(str,fOstream);
			if (bUseTcl) fputc('\n', fOstream);
			fflush(fOstream); 
		}
	}   
}

int Env::QueryUser( const char* query ) {
	char tst[10];
	while(1) {
		sprintf(gMsgStr,"%s (y/n)?\n>> ", query);
		PrintScreen();
		if( gMyID() == 0 ) {
			gets(tst); 
			if(tst[0] == 'y') return 1;
			if(tst[0] == 'n') return 0; 
		}	
	}
}
 
void Env::PrintLog( const char* str, int async ) {
  if(!fEnvInfo[kLog]) return;
  if( async || (gIProc == 0) ) { 
		fprintf(fLogFile,"\n %s",str); 
		fflush(fLogFile); 
	}
}

void Env::DumpToDebug() {
#ifdef unix
  debugger_stop_here(); 
#else
  BreakToSourceDebugger_();
#endif
}

void Env::fatal(const char* fmt, ...)
{
  buildmsg();
  gPrintErr(msg,1,"error");
	gPrintLog(msg);
	Finalize();
	fprintf(stderr,"\n\n");
  exit(1);
}

void Env::Fatal( const char* msg, int async ) {
	gPrintErr(msg,async,"error");
	gPrintLog(msg);
#ifdef JAVA
	JavaSocketReader::sendErrorMessage( (char*) msg );
#endif
	fprintf(stderr,"\n\n");
	Finalize();
	exit(1);
}

#ifdef USE_MPI  
int Env::GetComm(MPI_Comm* commP, int newC ) {
	int flag, err;
	if( gMPI ) {
		err = MPI_Initialized(&flag);
		if( flag == 0 ) { fprintf(stderr, "MPI Not Initialized! error= %d\n",err); return 1; }
		if( newC ) { MPI_Comm_dup(MPI_COMM_WORLD,commP); }
		else *commP = fComm; 
	}
	return 0;	
}
#endif

void Env::SetEnvVars( ) {

	int useProject = fTmpPath.empty();     
    
  if ( fProjName == ""  )  {
		if( useProject ) {
			char* tmp = GetEnv("ProjName");
			if( tmp == NULL) { 
				gFatal("Can't find Required Env Var: ProjName"); 
			}
			else fProjName = tmp;
		} else {
			fProjName = "DEFAULT";
		}
	}
  
  if (fModelName == "") {
		char* tmp = GetEnv("ModelName");
		if(tmp == NULL) { fModelName = fProjName; }
		else fModelName = tmp;
	}
  
	if( useProject ) { 
		if ( fModelPath.empty() ) { 
			char* proj_tmp = GetEnv("SME3_PROJ");
			// if not supplied, get project dir and add project name
			// if supplied on commandline, project name must be included
			if( proj_tmp == NULL ) {
				gPrintErr("Can't find ModelPath");
			} else { fModelPath.Add(proj_tmp); }
		} 
		fModelPath.Add(fProjName);
	} else {
		fModelPath.Add(fTmpPath);
	}

	if( useProject ) { 
		fConfigPath = fModelPath; fConfigPath.Add("Config");
		fArchivePath = fModelPath; fArchivePath.Add("DriverOutput");
		fDataPath = fModelPath; fDataPath.Add("Data");
	} else {
		fConfigPath = fTmpPath; 
		fArchivePath = fTmpPath;
		fDataPath = fTmpPath;
	}
  
  if( gDebug && gMyID() == 0 ) {
    sprintf(gMsgStr,"Model Path = %s",  fModelPath.chars()); gPrintScreen();
    sprintf(gMsgStr,"Config Path = %s", fConfigPath.chars()); gPrintScreen();
    sprintf(gMsgStr,"Archive Path = %s",fArchivePath.chars()); gPrintScreen();
    sprintf(gMsgStr,"Data Path = %s",   fDataPath.chars()); gPrintScreen(); 
    sprintf(gMsgStr,"Project Name = %s",fProjName.chars()); gPrintScreen();
    sprintf(gMsgStr,"Model Name = %s\n",fModelName.chars()); gPrintScreen();
  }

  CPathString localEnvFile( Env::DataPath() ); 
  localEnvFile.Add("Environment");
  ReadEnvValsFromFile( localEnvFile );
}



/* Sets name of program. */

void  Env::set_program_name (char *prog_name) 
{ 
  _program_name = prog_name;
}

void Env::error(const char* fmt, ...)
{
  buildmsg();
  gPrintErr(msg,1,"error");
	gPrintLog(msg);
}

void Env::log(const char* fmt, ...)
{
  buildmsg();
	gPrintLog(msg);
}

void Env::screen(const char* fmt, ...)
{
  buildmsg();
	gPrintScreen(msg);
}
/*
void Env::StartTimer() {
	CPathString path(fArchivePath);
	CString name(fModelName); 
	name += ".timer.";
	name.appendIndex(gIProc);
	path.Add(name);
	fTimerFile = fopen(path,"w");
	if( fTimerFile != NULL ) { 
		bUseTimer = TRUE;
		gPrintScreen("Starting timer."); 
	} else { sprintf(gMsgStr,"Unable to open Timer file: %s",path.chars()); gPrintScreen(gMsgStr,TRUE); }
	fGlobalTime = times(&_global_start_time);
}

void Env::StopTimer() {
	if( bUseTimer ) {
		bUseTimer = FALSE;
		fGlobalTime = times(&_global_end_time) - fGlobalTime;
		fprintf( fTimerFile, "\n ******** GLOBAL ELAPSED TIME *****************");
		PrintElapsedTime( fGlobalTime, _global_start_time, _global_end_time, TRUE );
		fclose(fTimerFile);
		gPrintScreen("Stoping timer."); 
	}
}
*/
void Env::SetMPIBufferSize( int min_size, int new_size ) {
#ifdef USE_MPI
	if( gMPI ) {
	  if( min_size > fMPI_buffer_size ) {
		if( fMPI_buffer_size > 0 ) {
			MPI_Buffer_detach(fMPI_buffer,&fMPI_buffer_size);
			delete[] fMPI_buffer;
		}
		fMPI_buffer = new byte[new_size];
		fMPI_buffer_size = new_size;
		MPI_Buffer_attach(fMPI_buffer,fMPI_buffer_size);
	  }
	}
#endif
}	

Bool Env::ArchiveMessage( const char* s1, EMessageType mt, int async ) {
	CString* s = NULL;
	if( bUseJavaSocket ) { 
		switch( mt ) {
			case kInfo  : s = fInfoMessages.Add(s1);  break;
			case kWarn  : s = fWarnMessages.Add(s1);  break;
			case kError : s = fErrorMessages.Add(s1); break;
		}
/*
		if( gDebug && ((gIProc == 0) || async ) ) {		
			fputs(s1,stdout);
			if (bUseTcl) fputc('\n', stdout);
		}
*/
		if( s!= NULL ) { s->clean_control_chars(); }
		return True;
	}
	return False;
}

FILE*  Env::GetBiFlowFile() {
  static FILE* biFlowArchiveFile = NULL;
  if( biFlowArchiveFile == NULL ) {
	CPathString path(Env::ConfigPath());
	path.Add( Env::ModelName() );
	biFlowArchiveFile = fopen(path.Path(".biflows"),"w");
  }
  return biFlowArchiveFile;
}

void  Env::ArchiveBiFlow( const char* module_name, const char* variable_name ) {
  FILE* f = GetBiFlowFile();
  if( f ) {
	fprintf(f, "%s %s\n",module_name,variable_name);
  }
}

const char*  Env::RetrieveBiFlow() {
  FILE* f = GetBiFlowFile();
  static char name[100];
  if( f ) {
	return fgets(name,100,f);
  }
  return NULL;
}

/*
clock_t Env::PrintElapsedTime( clock_t rt, struct tms& st, struct tms& et, Bool print_accumulation ) {
	static long clktck = 0;	
	clock_t tott = 0;
#ifdef HAS_SOCKETS
	if( clktck == 0 ) {
		if( (clktck = sysconf( _SC_CLK_TCK ) ) < 0 ) { gPrintErr("sysconf error"); }
	}
	clock_t usrt = ( et.tms_utime - st.tms_utime );
	clock_t syst = ( et.tms_stime - st.tms_stime );
	tott = usrt+syst;

	sprintf( gMsgStr, " ( real: %7.2f, user: %7.2f, sys: %7.2f, tot: %7.2f  ) ", 
		rt / (double) clktck,  usrt / ((double) clktck), syst / ((double) clktck), tott / ((double) clktck) );
	fprintf( fTimerFile,"\n %s",gMsgStr);
		
	if( print_accumulation ) {
		gPrintScreen(gMsgStr,TRUE);
//		sprintf( gMsgStr, " ( comm: %7.2f  ) ", fCommTime );
//		fprintf( fTimerFile," %s ",gMsgStr);
//		gPrintScreen(gMsgStr,TRUE);
		long total_tks = 0;
		fprintf( fTimerFile, "\n\tAccumulation by class: ( ");
		for(int i=0; i<10; i++ ) {
			total_tks += fAccumTime[i];
			fprintf( fTimerFile, " %7.2f, ", fAccumTime[i] / ((double) clktck) );
		}
		fprintf( fTimerFile, " total: %7.2f ) ", total_tks / ((double) clktck) );
	}
#endif
	return tott;
}
*/






