//----------------------------------------------------------------------------------------
  //	Env.h
  //	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
  //	Change History:
  //----------------------------------------------------------------------------------------

#ifndef __Env__
#define __Env__

#include "Globals.h"
#include "assoc.h"

#ifndef __CPathString__
#include "PathString.h"
#endif

#ifndef __ByteBuffer__
#include "ByteBuffer.h"
#endif

#ifndef __UTILITY_H
#include "Utility.h"
#endif

#include <stdarg.h>

extern "C" {
#include "ErrHandler.h"
// #include <sys/times.h>
}

#define BreakToSourceDebugger_()		 Env::debugger_stop_here();
#define lprint0(x) if(gDebug){sprintf(gMsgStr,x);gPrintLog();} 
#define lprint1(x,y) if(gDebug){sprintf(gMsgStr,x,y);gPrintLog();} 
#define lprint2(x,y,z) if(gDebug){sprintf(gMsgStr,x,y,z);gPrintLog();} 
#define lprint3(x,y,z,w) if(gDebug){sprintf(gMsgStr,x,y,z,w);gPrintLog();} 
#define lprint4(x,y,z,w,v) if(gDebug){sprintf(gMsgStr,x,y,z,w,v);gPrintLog();} 
#define lprint5(x,y,z,w,v,q) if(gDebug){sprintf(gMsgStr,x,y,z,w,v,q);gPrintLog();} 

class Env {

  static byte fEnvInfo[8];
  static byte fByteOrdering;
  static byte* fMPI_buffer;
  static int fMPI_buffer_size;
  static  FILE* fLogFile;
  static  FILE* fOstream;
  static  FILE* fEstream;
  static  FILE* fTimerFile;
  static ByteBuffer fTemp;
  static int fStopCnt;
  static char* _program_name; /* Records the program name. */
  static CString fErrorString;
  static ArgArray fArgv;
  
  static ArgArray fInfoMessages;
  static ArgArray fWarnMessages;
  static ArgArray fErrorMessages;
  static ArgArray fInfoBuffer;
  static ArgArray fWarnBuffer;
  static ArgArray fErrorBuffer;
  static TAssoc_array fGlobalEnvVars;
  static TAssoc_array fLocalEnvVars;
/*  
  static clock_t fAccumTime[10];
  static clock_t fGlobalTime;
  static clock_t fLocalTime;
*/  
  static double fCommTime;
  static double fCommStartTime;
/*
  static struct tms _global_start_time;
  static struct tms _global_end_time;
  static struct tms _local_start_time;
  static struct tms _local_end_time;
*/
  static CPathString fModelPath;
  static CPathString fTmpPath;
  static CPathString fDataPath;
  static CPathString fConfigPath;
  static CPathString fArchivePath;
  static CString fTclFile;
  static CString fHDFInFile;
  static CString fProjName; 
  static CString fModelName; 
  static CString fScenarioName;
  static CString fDebugger; 
  static CString fProgram; 
  static CString fDisplay; 
  static CString fTclRunScript;
#ifdef USE_MPI
  static MPI_Comm fComm;
#endif

  static int FindEnvValue(FILE *infile);
  static char* GetEnvValFromFile(CPathString fileName, const char* varName);
  static void ReadEnvValsFromFile( CPathString fileName );

  static CString iHost;
  static int     iPort;
  static int     iTimeout;
  static int     iSockBufsize;
  static int     iSocket;
  static void*   iDataRep;
  static size_t  iDataSize;
  static int     iSync;
  static int     iBatchIndex;  

  static Bool    bUseTcl;
  static Bool    bUsePython;
	static Bool    bUseJavaSocket;
	static Bool    bAttachDebugger;
	static Bool    bReadConf;
	static Bool    bUseDebugger;
	static Bool    bUseMPE;
 	static Bool    bUseTimer;
	static Bool    bGlobalSort;
 	static int     iPauseLevel;
	static int     iReplicationIndex;
	static float   fTime;

public:

  enum EEnvInfo { kUseDebugger, kLog, kSimOpen, kEventsDefined, kAppType };
  enum EOSType { kUnix, kMac, kDOS };
  enum EAppType { kUndefined, kDriver, kMCP, kCGP, kSTELLAP, kMMLP }; 
  enum EMessageType { kInfo, kWarn, kError };
	
	static inline EOSType OSType() {
#ifdef dos
		return kDOS;
#else
		return kUnix;
#endif	
	}
	
  static void Startup (int  argc, char* argv[], EAppType type );
  static void Finalize ();
  static void BCastTest(int test_index) ;
  static TAssoc_array& LocalEnvVars() { return fLocalEnvVars; }
  
  static inline byte GetInfo(EEnvInfo index) { return fEnvInfo[index]; }
  static inline void SetInfo(EEnvInfo index, byte val) { fEnvInfo[index] = val; }
  static int AttachDebugger( const char* display ); 
 
  static void PrintScreen( const char* str=gMsgStr, int async = False );
  static void PrintLog( const char* str=gMsgStr, int async = True );
  static inline void PrintDebug( char* str=gMsgStr, int async = True ) { PrintLog( str, async ); }

  static int QueryUser( const char* query );

  static void Fatal( const char* msg = NULL, int async = True );

	static void error(const char* fmt, ...);
	static void fatal(const char* fmt, ...);
	static void screen(const char* fmt, ...);
	static void log(const char* fmt, ...);
	
  static void DumpToDebug();

  static inline void ProgramBreak(const char* msg=NULL, int async = True) { PrintScreen(msg,async); DumpToDebug(); }

  static char* GetEnv(const char* varName);
  static void SetEnvVars( );

#ifdef USE_MPI  
	static int GetComm(MPI_Comm* commP, int newC = 1);
#endif

  static inline FILE* LogFile() { return fLogFile; }
  static void FlushScreen();
  static inline void SetDebugLevel(int dLevel) { gDebug = dLevel; }
  static inline void SetGlobalSort( Bool val ) { bGlobalSort = val; } 

  static const CPathString& ModelPath() { return fModelPath; }
  static const CPathString& DataPath() { return fDataPath; }
  static const CPathString& ArchivePath() { return fArchivePath; }
  static const CPathString& ConfigPath() { return fConfigPath; }
  static const CPathString& TmpPath() { return fTmpPath; }

  static const CString& ProjName() { return fProjName; } 
  static const CString& ModelName() { return fModelName; } 
  static const CString& ScenarioName() { return fScenarioName; } 
  static CString& HDFInputFile() { return fHDFInFile; }
  static const CString& TclFile() { return fTclFile; }
  
  // communication stuff
  static CString& Host()           { return iHost; }
  static int&     Port()           { return iPort; }
  static int&     Timeout()        { return iTimeout; }
  static int&     SockBufsize()    { return iSockBufsize; }
  static int      Socket(int listen_for_connection = 1);
  static int 			Listen();
  static int&     Sync()           { return iSync;  }
  static void*    DataRep()        { return iDataRep; }
  static void     DataRep(void* v) { iDataRep = v;    }
  static size_t&  DataSize()       { return iDataSize; }
	 
	// replication
  static int&     ReplicationIndex() { return iReplicationIndex; } 
	
	// use TCL interpreter
  static Bool&   UseTcl() { return bUseTcl; } 
  static Bool&   UsePython() { return bUsePython; } 
  static Bool&   ReadConf() { return bReadConf; } 
  static Bool&   UseMPE() { return bUseMPE; } 
  static CString&  TclRunScript() { return fTclRunScript; } 
  static int&   BatchIndex() { return iBatchIndex; } 
  static int debugger_stop_here();
  static void set_program_name (char *prog_name);
  static void Error (char *format, ...);
  
  static inline const CString& Argv(int i) { return fArgv[i]; }
  static inline int Argc() { return fArgv.NArgs(); }
	static CString& Program() { return  fProgram; } 
  static CString& Display() { return  fDisplay; } 

	static Bool&   UseJavaSocket() { return bUseJavaSocket; } 
  static inline void SetOutputStream(FILE* stream) { fOstream = stream; }
  static inline void SetErrorStream(FILE* stream) { fEstream = stream; }
  static inline void SetLogStream(FILE* stream) { fLogFile = stream; }
	static void PauseForResponse() ;
	static inline Bool UseTimer() { return bUseTimer; }
	static inline Bool GlobalSort() { return bGlobalSort; }
	static inline FILE* TimerFile() { return fTimerFile; }	
//	static void StartTimer();
//	static void StopTimer() ;
//	static clock_t PrintElapsedTime( clock_t rt, struct tms& st, struct tms& et, Bool print_accumulation );
	static void SetMPIBufferSize( int min_size, int new_size );
	static inline byte ByteOrdering() { return fByteOrdering; }
	static CString& ErrorString() { return fErrorString; }
	
	static inline int getNInfo() { fInfoMessages.move(fInfoBuffer);  return fInfoBuffer.NArgs(); }
	static inline CString& getInfo( int index ) { return fInfoBuffer.elem(index); }

	static inline int getNError() { fErrorMessages.move(fErrorBuffer);  return fErrorBuffer.NArgs(); }
	static inline CString& getError( int index ) { return fErrorBuffer.elem(index); }

	static inline int getNWarn() { fWarnMessages.move(fWarnBuffer);  return fWarnBuffer.NArgs(); }
	static inline CString& getWarn( int index ) { return fWarnBuffer.elem(index); }
	
	static Bool ArchiveMessage( const char* str, EMessageType mt, int async  );
/*	
	static inline void StartLocalTimer() {
			if( bUseTimer ) { fLocalTime = times(&_local_start_time); }
	}
	static inline clock_t StopLocalTimer( int category, const char* msg1, const char* msg2, const char* msg3, Bool print_accumulation ) {
		clock_t cpu_time = 0;
		if( bUseTimer ) {
			fLocalTime = times(&_local_end_time) - fLocalTime;
			fprintf(fTimerFile,"\n  < %s %s %s >: ",msg1,msg2,msg3);
			cpu_time = PrintElapsedTime( fLocalTime, _local_start_time, _local_end_time, print_accumulation );
			fAccumTime[category] += cpu_time;
		}
		return cpu_time;
	}
*/	
	static FILE* GetBiFlowFile() ;	
	static void ArchiveBiFlow( const char* module_name, const char* variable_name );
	static const char* RetrieveBiFlow();

#ifdef USE_MPI
	static inline void StartCommTimer() {
			if( bUseTimer ) { fCommStartTime = MPI_Wtime(); }
	}
	static inline void StopCommTimer() {
		if( bUseTimer ) { fCommTime += (MPI_Wtime() - fCommStartTime); }
	}
#endif
	static inline void EnableDebugger() {
		bAttachDebugger = TRUE;
	}

};

inline void gPrintScreen( const char* str=gMsgStr, int async = False, char* label = "info" ) { 
	const char* s1 = (str) ? (str) : (const char*)gMsgStr; 
	CString s("\n"); s+= label; 
	if( async && gMPI ) { s += ": P"; s.appendIndex(gIProc); }
	s+=": "; s+= s1;
  Env::PrintScreen( s, async ); 
}


inline void gPrintErr( const char* str=gMsgStr, int async = True, char* label = "warning" ) {
	const char* s1 = (str) ? (str) : (const char*)gMsgStr; 
	gPrintScreen(s1,async,label); 
	if( !Env::UseJavaSocket() ) {
		errno=0;
		if( gIProc == 0 ) { Env::PauseForResponse(); }
	}
	Env::DumpToDebug();
}

inline void gProgramBreak(const char* msg=gMsgStr, int async = True) {
  Env::ProgramBreak( msg, async ); }

inline void gPrintLog( const char* str=gMsgStr, int async = True ) {
  Env::PrintLog( str, async ); }

inline void gPrintDebug( char* str=gMsgStr, int async = True ) {
  Env::PrintDebug( str, async ); }

inline void gFatal( const char* msg = gMsgStr, int async = True ) {
  Env::Fatal( msg, async ); }
 

#endif

 
