#ifdef USE_MPI
#include "mpi.h"
#endif

#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>
#include "ErrHandler.h"

char result_string[512];

const char* SME_debug_get_result_string() { return result_string; }

char* SME_get_root_display(int myid) {
  int  str_len;
  char *disp, *display;
#ifdef USE_MPI 
  int flag, err;
#endif
 if (myid == 0) {
    display = getenv( "DISPLAY" );
    if (!display || display[0] == ':') {
      /* Replace display with hostname:0 */
      display = (char *)malloc( 100 );
      disp = getenv( "HOST" ); 
      if( disp ) {
				sprintf( display, "%s:0", disp );
			} else {
				sprintf( display, "flock:0" );
			}
    }
    str_len = strlen( display ) + 1;
  }
#ifdef USE_MPI
  err = MPI_Initialized(&flag);
  if( flag != 0 ) { 
    MPI_Bcast( &str_len, 1, MPI_INT, 0, MPI_COMM_WORLD );
    if (myid != 0) { display = (char *) malloc( str_len ); }
    MPI_Bcast( display, str_len, MPI_CHAR, 0, MPI_COMM_WORLD );
  }
#endif
  return display;
}

/* Place to put varargs code, which should look something like

   void mpir_errors_are_fatal( MPI_Comm *comm, int *code, ... )
   {
   va_list Argp;

   va_start( Argp, code );
   string = va_arg(Argp,char *);
   file   = va_arg(Argp,char *);
   line   = va_arg(Argp,int *);
   va_end( Argp );
   ... 
   }
 */

/* 
   In order to allow more elaborate debuggers, such as starting the
   debugger in an xterm or with a -display option, we provide a
   default argument stream in baseargs.  In this case, the arguments are
   baseargs, followed by the process id of the job to stop.
   If baseargs not set, then the given debugger is used.
 */
static char *program=0;
static char *display=0;
static char *debugger=0;
static char **baseargs=0;
static int  nbaseargs=0;
static int  debugger_started=0;


int SME_Start_debugger(const char* display) {
	int child; 
  child = fork(); 
  if (child) { /* I am the parent will run the debugger */
    char  **args, pid[10];
    char title[100];
		char *host; 
		int i, i0, myid=0, index=0;  
#ifdef USE_MPI 		
		MPI_Comm_rank( MPI_COMM_WORLD, &myid );
#endif
		host =  getenv( "HOST" ); 

    args = (char **)malloc( (25 + nbaseargs) * sizeof(char *) );
    /* PETSc has discovered that many systems DON'T want a sigstop. They
     are described here */
#if  !defined(MPI_rs6000) && !defined(MPI_solaris) && \
    !defined(MPI_IRIX) && !defined(MPI_IRIX64) && !defined(MPI_freebsd) && \
    !defined(MPI_LINUX) && !defined(WIN32)
    kill(child,SIGSTOP);
#endif
    sprintf(pid,"%d",child); 
		if( host != NULL ) { sprintf(title,"debug%d-%s",myid,host); }
		else  { sprintf(title,"debug%d",myid); }
		args[index++] = SME_Copy_string( "/usr/openwin/bin/xterm" );
    args[index++] = SME_Copy_string( "-display" );
    args[index++] = SME_Copy_string( display );
    args[index++] = SME_Copy_string( "-T" );
    args[index++] = SME_Copy_string( title );
		args[index++] = SME_Copy_string( "-e" );

    if (nbaseargs > 0) {
			for (i=0; i<nbaseargs; i++) {
				args[index++] = baseargs[i];
			}
#ifdef DBX_IS_OSF
			args[index++] = "-pid";
#elif defined(MPI_hpux)
			args[index++] = "-P";
#elif defined(MPI_IRIX)
			args[index++] = "-p";
#elif defined(MPI_rs6000)
	/* Note that rs6000 dbx PROHIBITS passing the name of the program(!),
	   at least in some versions! If one of the "baseargs" is
           the program name, we're dead.  Sigh. */
			args[index++] = "-a";
#endif
			args[index++] = pid;
			args[index++] = 0;
			debugger  = args[0];
	}
    else {
			index = 0;
			args[index++] = debugger; args[index++] = program; 
#ifdef DBX_IS_OSF
			args[index++] = "-pid";
#elif defined(MPI_hpux)
			args[index++] = "-P";
#elif defined(MPI_IRIX)
			args[index++] = "-p";
#elif defined(MPI_rs6000)
	/* Note that rs6000 dbx PROHIBITS passing the name of the program(!),
	   at least in some versions! */
			index--;      /* Back up over the program name */
			args[index++] = "-a";
#endif
			args[index++] = pid; args[index++] = 0;
    }
    fprintf(stderr,"info: Attaching debugger %s to %s %s, ", debugger, program, pid);
    for( i0=0; i0<index; i0++ ) if(args[i0]) fprintf(stderr," %s ",args[i0]);
    fprintf(stderr," \n ");  fflush(stderr);
    if (execvp(debugger, args)  < 0) {
      sprintf(result_string, "error: Unable to start debugger: %s \n",  strerror(errno) );
      fprintf(stderr, "%s", result_string );
      return 1;
    } else {
      sprintf(result_string, "info: Running Process %d on %s\n", myid, host); 
      fprintf(stderr,"%s ", result_string );
	}
  }
  else { /* I am the child, continue with user code */
#if defined(MPI_hpux)
      while(1); /* HP cannot attach to sleeper */
#else
    sleep(20);
#endif
#ifdef DBX_IS_OSF
    /* Need to stop the child so that the debugger will respond.  Wierd that
       you need this */
    /* kill(child,SIGTSTP); */
    /* It didn't work.   What to do now? */
#endif
  }
  return 0;
}


/*@
   SME_Errors_call_debugger - On an error, print a message and (attempt) to
   start the specified debugger on the program

   Input Parameters:
.  pgm - Name of the program.
.  dbg - Name of the debugger.  If null, use a default (usually dbx)
.  args - arguments to use in generating the debugger.
   This allows things like "xterm -e dbx pgm pid", or 
   "xdbx -geometry +%d+%d pgm pid".  The list should be null terminated.
   (The %d %d format is not yet supported).

    Notes:
    You may need to ignore some signals, depending on the signals that
    the MPICH and underlying communications code is using.  You can
    do this in dbx by adding "ignore signal-name" to your .dbxinit file.
    For example, to ignore SIGUSR1, use "ignore USR1".

    Currently, there is no Fortran interface for this routine.
@*/



int SME_Start_gdb( const char* pgm )
{
  int child=0, i;
  char* dbg = "/usr/local/bin/gdb";
  
  child = fork(); 
  if (child) { /* I am the parent will run the debugger */
    char  **args, pid[10];
    args = (char **)malloc( 6 * sizeof(char *) );
#if !defined(WIN32)
    kill(child,SIGSTOP);
#endif
    sprintf(pid,"%d",child); 
    program = (char *)malloc( strlen(pgm) + 1 );
    strcpy( program, pgm );
    debugger = (char *)malloc( strlen(dbg) + 1 );
    strcpy( debugger, dbg );
    i = 0;
    args[i++] = debugger; args[i++] = program; 
    args[i++] = pid; args[i++] = 0;
    fprintf(stderr,"info: Attaching %s to %s %s\n", debugger, program, pid);
    if (execvp(debugger, args)  < 0) {
      perror("Unable to start debugger");
      exit(3);
    }
  }
  else { /* I am the child, continue with user code */
    sleep(10);
  }
  return 0;
}


int SME_Start_gdb_in_xterm0( char* pgm )
{
  int child=0, myid= 0;
  char *display, *host;
#ifdef USE_MPI
  MPI_Comm_rank( MPI_COMM_WORLD, &myid );
#endif
  display = SME_get_root_display(myid);
  host =  getenv( "HOST" ); 
    
  child = fork(); 
  if (child) { /* I am the parent will run the debugger */
    char  **args, pid[10];
    char title[100];
#if  !defined(MPI_rs6000) && !defined(MPI_solaris) && \
    !defined(MPI_IRIX) && !defined(MPI_IRIX64) && !defined(MPI_freebsd) && \
    !defined(MPI_LINUX) && !defined(WIN32) 
    kill(child,SIGSTOP);
#endif    
    sprintf(pid,"%d",child); 
		sprintf(title,"debug%d-%s",myid,host);
		
    args    = (char **)malloc( 12 * sizeof(char *) );
    args[0] = SME_Copy_string( "/usr/openwin/bin/xterm" );
    args[1] = SME_Copy_string( "-display" );
    args[2] = SME_Copy_string( display );
    args[3] = SME_Copy_string( "-T" );
    args[4] = SME_Copy_string( title );
		args[5] = SME_Copy_string( "-e" );
    args[6] = SME_Copy_string( "/usr/local/bin/gdb" );
		args[7] = SME_Copy_string( "-x" );
    args[8] = SME_Copy_string( "~/.gdb.startup.sme.MPI" );
    args[9] = SME_Copy_string( pgm );
    args[10] = SME_Copy_string( pid );
    args[11] = 0;

    fprintf(stderr,"info: (%d)Attaching debugger in xterm (%s) to %s %s, running on %s:\n", 
	    myid, display,  pgm, pid, host );
    if (execvp(args[0], args)  < 0) {
      fprintf(stderr,"info: Unable to start debugger: %s",  strerror(errno) );
      exit(3);
    } else fprintf(stderr,"info: Running Process %d on %s\n", myid, host); 
  }
  else { /* I am the child, continue with user code */
    sleep(10);
  }
  return 0;
}

int SME_Start_gdb_in_xterm( const char* pgm, const char* display )
{
	nbaseargs = 4;
	program = (char*)pgm;
	debugger = "/usr/local/bin/gdb";
	baseargs = (char **)malloc( (nbaseargs+1) * sizeof(char *) );
	baseargs[0] = SME_Copy_string( debugger );
	baseargs[1] = SME_Copy_string( "-x" );
	baseargs[2] = SME_Copy_string( "~/.gdb.startup.sme.MPI" );
	baseargs[3] = SME_Copy_string( pgm );
	return SME_Start_debugger(display);
}

int SME_Start_dbx_in_xterm( const char* pgm, const char* display )
{
	nbaseargs = 4;
	program = (char*) pgm;
	debugger = "dbx";
	baseargs = (char **)malloc( (nbaseargs+1) * sizeof(char *) );
	baseargs[0] = SME_Copy_string( debugger );
	baseargs[1] = SME_Copy_string( "-C" );
	baseargs[2] = SME_Copy_string( "-editor" );
	baseargs[3] = SME_Copy_string( pgm );
	return SME_Start_debugger(display);
}

#ifdef USE_MPI 

/*

void SME_Errors_to_dbx( MPI_Comm *comm, int *code, ... )
{
  char buf[MPI_MAX_ERROR_STRING];
  int  myid, result_len; 
  char *string, *file, *display;
  int  *line;
  va_list Argp;
  int child, ncmd = *code;

  va_start( Argp, (void) ncmd );
  string = va_arg(Argp,char *);
  file   = va_arg(Argp,char *);
	display   = va_arg(Argp,char *);
	line   = va_arg(Argp,int *);
  va_end( Argp );

  if (MPI_COMM_WORLD) MPI_Comm_rank( MPI_COMM_WORLD, &myid );
  else myid = -1;
  MPI_Error_string( *code, buf, &result_len );
  fprintf( stderr, "%d -  File: %s   Line: %d\n", myid, 
		   file, *line );
  fprintf( stderr, "%d - %s : %s\n", myid, 
          string ? string : "<NO ERROR MESSAGE>", buf );

  SME_Start_debugger(display);
}

void SME_Errors_call_debugger( char* pgm, char* dbg, char** args )
{
  MPI_Errhandler err;
  int rv, i;

  if (args) {
    while (args[nbaseargs]) nbaseargs++;
    baseargs = (char **)malloc( (nbaseargs+1) * sizeof(char *) );
    for (i=0; i<=nbaseargs; i++) 
      baseargs[i] = args[i];
  }
  else if (!dbg) {
    dbg = "/usr/local/bin/gdb";
  }
  
  if (!pgm) {
    fprintf( stderr, 
	     "Must specify the program name when setting errors-call-debugger\n" );
    return;
  }
  program = (char *)malloc( strlen(pgm) + 1 );
  strcpy( program, pgm );
  if (dbg) {
    debugger = (char *)malloc( strlen(dbg) + 1 );
    strcpy( debugger, dbg );
  }  
	rv = MPI_Errhandler_create( (MPI_Handler_function*) SME_Errors_to_dbx, &err );
  if( rv != MPI_SUCCESS ) { 
		fprintf( stderr, "\nUnable to create error handler!"); 
		exit(rv);
	}
  rv = MPI_Errhandler_set( MPI_COMM_WORLD, err );
  if( rv != MPI_SUCCESS ) { 
		fprintf( stderr, "\nUnable to set error handler!"); 
		exit(rv);
	}
}

void SME_Errors_dump_to_gdb( MPI_Comm comm, int * code, ... ) {
  char dstring[MPI_MAX_ERROR_STRING]; 
  int len, myid = -1;
  if (MPI_COMM_WORLD) MPI_Comm_rank( MPI_COMM_WORLD, &myid );
  MPI_Error_string(*code,dstring,&len);
  fprintf( stderr, "\nerror (MPI)P%d: %s\n",myid,dstring);
  if( debugger_started == 0 ) { 
		SME_Start_gdb_in_xterm( program, display );
		debugger_started = 1;
	}
  SME_stop_here();
}

void SME_Errors_dump_to_dbx( MPI_Comm comm, int * code, ... ) {
  char dstring[MPI_MAX_ERROR_STRING]; 
  int len, myid = -1;
  if (MPI_COMM_WORLD) MPI_Comm_rank( MPI_COMM_WORLD, &myid );
  MPI_Error_string(*code,dstring,&len);
  fprintf( stderr, "\nerror (MPI)P%d: %s\n",myid,dstring);
  if( debugger_started == 0 ) { 
		SME_Start_dbx_in_xterm( program, display );
		debugger_started = 1;
	}
  SME_stop_here();
}


void SME_Errors_dump_to_debugger( MPI_Comm com, const char* pgm, const char* dsp, const char* debugger )
{  
  MPI_Errhandler err;
  int rv;
  if( strcmp(debugger,"gdb") == 0 ) {
	rv = MPI_Errhandler_create( (MPI_Handler_function*) SME_Errors_dump_to_gdb, &err );
  } else {
	rv = MPI_Errhandler_create( (MPI_Handler_function*) SME_Errors_dump_to_dbx, &err );
  }
  if( rv != MPI_SUCCESS ) { 
		fprintf( stderr, "\nUnable to create error handler" ); 
		exit(rv);
	}
  rv = MPI_Errhandler_set( com, err );
  if( rv != MPI_SUCCESS ) { 
		fprintf( stderr, "\nUnable to set error handler!"); 
		exit(rv);
	}
	if( program == 0 ) {
		program = (char *)malloc( strlen(pgm) + 1 );
		strcpy( program, pgm );
	}
	if( display == 0 ) {
		display = (char *)malloc( strlen(dsp) + 1 );
		strcpy( display, dsp );
	}
}


void SME_Errors_call_gdb_in_xterm( char* pgm, char* display )
{
  char **args, *disp;
  int  myid=0, str_len;

  if (!display) {
    MPI_Comm_rank( MPI_COMM_WORLD, &myid );
    if (myid == 0) {
      display = getenv( "DISPLAY" );
      if (!display || display[0] == ':') {
	display = (char *)malloc( 100 );
	disp = getenv( "HOST" ); 
	sprintf( display, "%s:0", disp );
      }
      str_len = strlen( display ) + 1;
    }
    MPI_Bcast( &str_len, 1, MPI_INT, 0, MPI_COMM_WORLD );
    if (myid != 0)  display = (char *) malloc( str_len );
    MPI_Bcast( display, str_len, MPI_CHAR, 0, MPI_COMM_WORLD );
  }
  
  args    = (char **)malloc( 7 * sizeof(char *) );
  args[0] = SME_Copy_string( "xterm" );
  args[1] = SME_Copy_string( "-display" );
  args[2] = SME_Copy_string( display );
  args[3] = SME_Copy_string( "-e" );
  args[4] = SME_Copy_string( "/usr/local/bin/gdb" );
  args[5] = SME_Copy_string( pgm );
  args[6] = 0;
  
  SME_Errors_call_debugger( pgm, (char *)0, args );
}
*/

void SME_set_error_handler( MPI_Handler_function* ehf )
{
  MPI_Errhandler err;
  int rv = MPI_Errhandler_create( ehf, &err );
  if( rv != MPI_SUCCESS ) { 
		fprintf( stderr, "\nUnable to create error handler!"); 
		exit(rv);
	}
  rv = MPI_Errhandler_set( MPI_COMM_WORLD, err );
  if( rv != MPI_SUCCESS ) { 
		fprintf( stderr, "\nUnable to set error handler!"); 
		exit(rv);
	}
}
void SME_Errors_breakpoint( MPI_Comm *comm, int *code, ... )
{
	return;  /* set break here  */
}

void SME_set_error_break()
{
	SME_set_error_handler( (MPI_Handler_function*) SME_Errors_breakpoint );
}

void SME_Errors_print_handler( MPI_Comm comm, int * code, ... ) {
  char dstring[MPI_MAX_ERROR_STRING]; 
  int len, myid = -1;
  if (MPI_COMM_WORLD) MPI_Comm_rank( MPI_COMM_WORLD, &myid );
  MPI_Error_string(*code,dstring,&len);
  fprintf( stderr, "\nerror (MPI)P%d: %s\n",myid,dstring);
  SME_stop_here();
}

void SME_Errors_print( MPI_Comm com, const char* pgm, const char* dsp )
{  
  MPI_Errhandler err;
  int rv = MPI_Errhandler_create( (MPI_Handler_function*) SME_Errors_print_handler, &err );
  if( rv != MPI_SUCCESS ) { 
		fprintf( stderr, "\nUnable to create error handler" ); 
		exit(rv);
	}
  rv = MPI_Errhandler_set( com, err );
  if( rv != MPI_SUCCESS ) { 
		fprintf( stderr, "\nUnable to set error handler!"); 
		exit(rv);
	}
	if( program == 0 ) {
		program = (char *)malloc( strlen(pgm) + 1 );
		strcpy( program, pgm );
	}
	if( display == 0 ) {
		display = (char *)malloc( strlen(dsp) + 1 );
		strcpy( display, dsp );
	}
}

#endif


char *SME_Copy_string( const char* str )
{
  char *snew;
  snew = (char *)malloc( strlen(str) + 1 );
  strcpy( snew, str );
  return snew;
}

/* This routine is collective; all processes in MPI_COMM_WORLD must call */



int SME_stop_here() { return 0; }




