/*-------------------------------------------------------------------------
 *
 *   PGconnection
 *
 *   DESCRIPTION
 *      implementation of the PGconnection class.
 *   PGconnection encapsulates a frontend to backend connection
 *
 *-------------------------------------------------------------------------
 */
#ifdef POSTGRESS

#include <stdlib.h>
#include "PQpp.h"

extern "C" {
#include "libpq/libpq-fs.h"
}

// default constructor
// checks environment variable for database name
PGconnection::PGconnection()
{
  char* name;
  PGenv* newenv;

  conn = NULL;
  result = NULL;
  errorMessage[0] = '\0';

  newenv = new PGenv(); // use reasonable defaults for the environment
  if (!(name = getenv(ENV_DEFAULT_DBASE)))
    return;
  connect(newenv, name);
}

// constructor -- for given environment and database name
PGconnection::PGconnection(PGenv* Env, char* dbName)
{
  conn = NULL;
  result = NULL;
  errorMessage[0] = '\0';
  connect(Env, dbName);
}

// destructor - closes down the connection and cleanup
PGconnection::~PGconnection()
{
  if (result)	PQclear(result);
  if (conn)	PQfinish(conn);
}

// PGconnection::connect
// establish a connection to a backend
ConnStatusType
PGconnection::connect(PGenv* newenv, char* dbName)
{
#if 0
    FILE *debug;
    debug = fopen("/tmp/trace.out","w");
    PQtrace(conn, debug);
#endif

    env = newenv;
    fe_setauthsvc(env->pgauth, errorMessage); 
    conn = PQsetdb(env->pghost, env->pgport, env->pgoption, env->pgtty, dbName);
    if(strlen(errorMessage))
      return CONNECTION_BAD;
    else
      return status();
}

// PGconnection::status -- return connection or result status
ConnStatusType
PGconnection::status()
{
    return PQstatus(conn);
}

// PGconnection::exec  -- send a query to the backend
ExecStatusType
PGconnection::exec(char* query)
{
  if (result)
    PQclear(result); 

  result = PQexec(conn, query);
  if (result)
      return PQresultStatus(result);
  else {
      strcpy(errorMessage, PQerrorMessage(conn));
      return PGRES_FATAL_ERROR;
  }
}


/*-------------------------------------------------------------------------
 *
 *   PGenv
 *
 *   DESCRIPTION
 *      PGenv is the environment for setting up a connection to a 
 *   postgres backend,  captures the host, port, tty, options and
 *   authentication type.
 *
 *-------------------------------------------------------------------------
 */

#define DefaultAuth DEFAULT_CLIENT_AUTHSVC 
#define DefaultPort  "5432"

// default constructor for PGenv
// checks the environment variables
PGenv::PGenv()
{
  char* temp;

  pgauth = NULL;
  pghost = NULL;
  pgport = NULL;
  pgoption = NULL;
  pgtty = NULL;

  setValues(getenv(ENV_DEFAULT_AUTH), getenv(ENV_DEFAULT_HOST),
            getenv(ENV_DEFAULT_PORT), getenv(ENV_DEFAULT_OPTION),
	    getenv(ENV_DEFAULT_TTY));
}

// constructor for given environment
PGenv::PGenv(char* auth, char* host, char* port, char* option, char* tty)
{
  pgauth = NULL;
  pghost = NULL;
  pgport = NULL;
  pgoption = NULL;
  pgtty = NULL;

  setValues(auth, host, port, option, tty);
}

// allocate memory and set internal structures to match
// required environment
void
PGenv::setValues(char* auth, char* host, char* port, char* option, char* tty)
{
  char* temp;

  temp = (auth) ? auth : DefaultAuth;

  if (pgauth)
    free(pgauth);
  pgauth = strdup(temp);

  temp = (host) ? host : DefaultHost;

  if (pghost)
    free(pghost);
  pghost = strdup(temp);

  temp = (port) ? port : DefaultPort;

  if (pgport)
    free(pgport);
  pgport = strdup(temp);
  
  temp = (option) ? option : DefaultOption;

  if (pgoption)
    free(pgoption);
  pgoption = strdup(temp);

  temp = (tty) ? tty : DefaultTty;

  if (pgtty)
    free(pgtty);
  pgtty = strdup(temp);
}

// default destrutor
// frees allocated memory for internal structures
PGenv::~PGenv()
{
  if (pgauth)
    free(pgauth);
  if (pghost)
    free(pghost);
  if (pgport)
    free(pgport);
  if (pgoption)
    free(pgoption);
  if (pgtty)
    free(pgtty);
}


/*-------------------------------------------------------------------------
 *
 *   PGlobj
 *
 *   DESCRIPTION
 *      implementation of the PGlobj class.
 *   PGlobj encapsulates a frontend to backend connection
 *
 *-------------------------------------------------------------------------
 */


// default constructor
// creates a large object in the default database
PGlobj::PGlobj() : PGconnection() {
  object = lo_creat(conn, INV_READ|INV_WRITE);
  if (object == 0) {
    sprintf(errorMessage, "PGlobj: can't create large object");
  }
  fd = lo_open(conn, object, INV_READ|INV_WRITE);
  if (fd < 0) {
    sprintf(errorMessage, "PGlobj: can't open large object %d", object);
  } else
    sprintf(errorMessage, "PGlobj: created and opened large object %d",
	    object);
   
}

// constructor
// open an existing large object in the default database
PGlobj::PGlobj(Oid lobjId) : PGconnection() {
  object = lobjId;
  fd = lo_open(conn, object, INV_READ|INV_WRITE);
  if (fd < 0) {
    sprintf(errorMessage, "PGlobj: can't open large object %d", object);
  } else
    sprintf(errorMessage, "PGlobj: opened large object %d",
	    object);
}

// constructor
// create a large object in the given database
PGlobj::PGlobj(PGenv* Env, char* dbName) : PGconnection(Env,dbName) {
  object = lo_creat(conn, INV_READ|INV_WRITE);
  if (object == 0) {
    sprintf(errorMessage, "PGlobj: can't create large object");
  }
  fd = lo_open(conn, object, INV_READ|INV_WRITE);
  if (fd < 0) {
    sprintf(errorMessage, "PGlobj: can't open large object %d", object);
  } else
    sprintf(errorMessage, "PGlobj: created and opened large object %d",
	    object);
}

// constructor
// open an existing large object in the given database
PGlobj::PGlobj(PGenv* Env, char* dbName, Oid lobjId) : PGconnection(Env,dbName) {
  object = lobjId;
  fd = lo_open(conn, object, INV_READ|INV_WRITE);
  if (fd < 0) {
    sprintf(errorMessage, "PGlobj: can't open large object %d", object);
  } else
    sprintf(errorMessage, "PGlobj: created and opened large object %d",
	    object);
}

// PGlobj::unlink
// destruct large object and delete from it from the database
int
PGlobj::unlink() {
  int temp = lo_unlink(conn, object);
  if (temp) {
    return temp;
  } else {
    delete this;
    return temp;
  }
}

// PGlobj::import -- import a given file into the large object
int
PGlobj::import(char* filename) {
    char buf[BUFSIZE];
    int nbytes, tmp;
    int in_fd;

    // open the file to be read in
    in_fd = open(filename, O_RDONLY, 0666);
    if (in_fd < 0)  {   /* error */
	sprintf(errorMessage, "PGlobj::import: can't open unix file\"%s\"", filename);
	return -1;
    }

    // read in from the Unix file and write to the inversion file
    while ((nbytes = ::read(in_fd, buf, BUFSIZE)) > 0) {
      tmp = lo_write(conn, fd, buf, nbytes);
      if (tmp < nbytes) {
       sprintf(errorMessage, "PGlobj::import: error while reading \"%s\"",
         filename);
	  return -1;
      }
    }
    
    (void) close(in_fd);
    return 0;
}

// PGlobj::export -- export large object to given file
int
PGlobj::export(char* filename) {
    int out_fd;
    char buf[BUFSIZE];
    int nbytes, tmp;

    // open the file to be written to
    out_fd = open(filename, O_CREAT|O_WRONLY, 0666);
    if (out_fd < 0)  {   /* error */
	sprintf(errorMessage, "PGlobj::export: can't open unix file\"%s\"",
		filename);
	return -1;
    }

    // read in from the Unix file and write to the inversion file
    while ((nbytes = lo_read(conn, fd, buf, BUFSIZE)) > 0) {
      tmp = ::write(out_fd, buf, nbytes);
      if (tmp < nbytes) {
        sprintf(errorMessage,"PGlobj::export: error while writing \"%s\"",
	    filename);
	return -1;
      }
    }
    (void) close(out_fd);
    return 0;
}

// default destructor -- closes large object
PGlobj::~PGlobj() {
  if (fd >= 0)
    lo_close(conn, fd);
}


#endif
