#ifndef __PLEX_H
#define __PLEX_H

#include "Globals.h"
// #include <iostream.h>

#ifndef __TObjects__
#include "TObjects.h"
#endif

typedef int (*COMPARE_FUNC)(const VObject**, const VObject**);

class Plex : public TObject
{
  VObject** _data;      // Array of pointers to objects
  int _size;            // Size of the array
  int _index;           // iterator index

	VObject* add( VObject* objptr, int index);
	
protected:

  int _items;           // Number of items in the array
  void resize(int newdim);            

public:

  Plex(int arraysize);      
  Plex();                   
  Plex(const Plex&);    
  
  virtual ~Plex() ;
  virtual const char* class_name() const ;  
  virtual word class_id() const ;            
  virtual Bool ok() const ;                   

  int size() const { return _size; }         
  int items() const { return _items; }       
	inline int low() const { return 0; }
  int high() const;  
  int find(const VObject& v) const ;
	int find(const VObject* v) const ;                                                                                 

  VObject& operator[] (int index) const ;    
  VObject* objptr(int index) const ;      
  Plex& operator= (const Plex& a); 
  Plex& operator+= (const Plex& a);
  Plex& move ( Plex& receiver, Bool overwrite = TRUE );

  virtual Bool destroy(int index = -1, Bool pack = FALSE); 
  virtual inline VObject* dup() const { return new Plex(*this); }

  VObject* add(const VObject& object, int index = -1) ; 
  VObject*  insert(const VObject& object, int index = 0);
  VObject* remove(int index, Bool pack = FALSE);
  void swap(int i1, int i2);
  void pack();      
  void sort(COMPARE_FUNC = NULL); 
  inline VObject** last () const;
	inline VObject**  first () const;
	inline int  valid (int idx) const ;
	inline int   prev(int& idx) const;
	inline int  next(int& idx) const;
	inline int  more(int idx) const;
	inline void       prev(Pix& pix);
	inline void       next(Pix& pix);
	inline int        owns(Pix p) const;
	inline int        Pix_to_index(Pix p) const ;
	inline Pix        index_to_Pix(int idx) const ;
	inline void 		fill(const VObject&  x) ;
	inline void			fill(const VObject&  x, int from, int to) ;
	inline void			clear(); 
	inline VObject*     operator() (int index) const; // fast access with no safety checking.
 
  int             empty () const { return _items == 0; }
  
// Access functions 
    
  inline VObject&      elem (int idx) { return operator [] (idx); }
  inline const VObject&      elem (int idx) const { return (const VObject&) operator [] (idx); }
  inline VObject&      elem (Pix p)  { return **((VObject**)p); }
  inline const VObject&      elem (Pix p) const { return **((const VObject**)p); }  

// read-only versions for const Plexes

  inline const VObject& operator () (Pix p) const  { return **((const VObject**)p); }

// Index functions


// Growth

  inline VObject*       add_high(const VObject&  elem) { return add(elem); }
  inline VObject*       add_low(const VObject&  elem) { return insert(elem,0); }

// Shrinkage

  inline int       del_high() { remove( high(), FALSE); return high(); }
  inline int       del_low() { remove(0, TRUE); return 0; }

// operations on multiple elements

  inline int        length () const { return items(); }

  void		    error(const char* msg) const;
  void		    index_error() const;
};
  
inline VObject** Plex::last () const {
	for (int last = size(); --last >= 0; ) {
		if ( _data[last] ) return (VObject**) &_data[ last ];
	}
}

inline VObject**  Plex::first ()  const { return (VObject**) &_data[0]; }

inline int  Plex::valid (int idx) const { return (_data[idx] != NULL); }

inline int   Plex::prev(int& idx) const {
	return --idx;
}

inline int  Plex::next(int& idx) const {
	return ++idx;
}

inline int  Plex::more(int idx) const {
 return (  (idx >= 0) && (idx < size()) && (_data[idx] != NULL) ); 
}

inline void       Plex::prev(Pix& pix) { if( ((VObject**)pix) > _data ) ((VObject**&)pix)--; else pix = NULL; }
inline void       Plex::next(Pix& pix) { if( ((VObject**)pix) < (_data+_size-1) ) ((VObject**&)pix)++; else pix = NULL;  }
inline int        Plex::owns(Pix p) const { return ((p >= _data) && (p < (_data+_size))); }

// index<->Pix 

inline int        Plex::Pix_to_index(Pix p) const    { return find(*((VObject**)p));  }     
inline Pix        Plex::index_to_Pix(int idx) const {  return ( idx < _size ) ? _data + idx : NULL; }   
 
inline VObject* Plex::objptr(int index) const {
  return (index < _size && index >= 0) ? _data[index] : (VObject*)NULL;
}

inline void Plex::fill(const VObject&  x) {
	VObject* objPtr = x.dup();
	for (int i = 0; i < size(); i++) {
		_data[i] = objPtr;
	}
}

inline void      Plex::fill(const VObject&  x, int from, int to) {
	VObject* objPtr = x.dup();
	for (int i = from; (i < size()) && ( i < to ) ; i++) {
		_data[i] = objPtr;
	}
}

inline void  Plex::clear() { 
	for (int i = 0; i < size(); i++) {
		if( _data[i] != NULL ) {
			delete _data[i]; 
			_data[i] = NULL;
		}
	}
	_items = 0;
}


inline VObject& Plex::operator[] (int index) const
{
  VObject* o = objptr(index);
#ifdef DBG
  if (o == NULL)
    Env::error("Can't access NULL array item: %d of %d", index, size());
#endif
  return *o;
}

inline VObject* Plex::operator() (int index) const { return _data[index]; }  // fast access with no safety checking.

class TArray;

typedef int (*COMPARE_FUNCTION)(const TObject**, const TObject**);

class TArray : public TObject
{
  TObject** _data;      // Array of pointers to objects
  int _size;            // Size of the array
  int _items;           // Number of items in the array
  Bool _autodestroy;    // Destroy objects on delete

protected:
  // @FPROT
  void resize(int newdim);            // Estende l'array

public:
  // @FPUB
  TArray(int arraysize);      // Crea un array (chiama resize)
  TArray();                   // Crea un array (non chiama resize)
  TArray(const TArray&);      // copia tutto e duplica gli elementi 
  
  // (casino se non hanno dup() definita)
  virtual ~TArray() ;
  virtual const char* class_name() const ;    // Ritorna il nome della classe
  virtual word class_id() const ;             // Ritorna il class-id
//  virtual void print_on(ostream& out) const ; // Stampa un array
  virtual Bool ok() const ;                   // Ok se l'array non e' vuoto

  int size() const { return _size; }          // Ritorna grandezza dell'array
  int items() const { return _items; }        // Ritorna numero di oggetti nell'array
  int last() const;                                                                                                               // Ritorna l'indice dell'ultimo oggetto
  Bool autodestroy() const { return _autodestroy; }
  void autodestroy(Bool b) { _autodestroy = b;    }

  TObject& operator[] (int index) const ;     // [] ritorna l'oggetto puntato da index
  TObject* objptr(int index) const ;          // Ritorna l'oggetto di posto [index]
  TArray& operator= (const TArray& a); 

  virtual Bool destroy(int index = -1, Bool pack = FALSE); // Rimuove uno o tutti gli elementi (default)
  virtual int add(TObject* obj, int index = -1) ; // Aggiunge un oggetto ad un array.
  virtual int insert(TObject* obj, int index = 0);

  int add(const TObject& object, int index = -1) ; // Aggiunge un oggetto all'array. L'oggetto viene duplicato
  int insert(const TObject& object, int index = 0);
  TObject* remove(int index, Bool pack = FALSE);
  void swap(int i1, int i2);
  void pack();      // Rende contigui tutti gli elementi non nulli
  void sort(COMPARE_FUNCTION = NULL);  // Ordina i TObject (TSortable per default)
};

// @FIN
inline TObject* TArray::objptr(int index) const
{
  return (index < _size && index >= 0) ? _data[index] : (TObject*)NULL;
}

// @FIN
inline TObject& TArray::operator[] (int index) const
{
  TObject* o = objptr(index);
#ifdef DBG
  if (o == NULL)
    fatal_box("Can't access NULL array item: %d of %d", index, size());
#endif
  return *o;
}


class TBit_Array : public VObject
{
  word _size;
  byte* _bit;

protected:
  virtual Bool ok() const;
  void resize(word size);
  void copy(const TBit_Array& ba);
  word index(long n) const { return word(n >> 3); }
  byte mask(long n) const { return 1 << (n & 0x7); }

public:
  TBit_Array(long size = 0);
  TBit_Array(const TBit_Array& ba);
  ~TBit_Array();

  TBit_Array& operator=(const TBit_Array& ba);
  Bool operator[] (long n) const;
  TBit_Array& operator |=(const TBit_Array& b); 

  long first_one() const;
  long last_one() const;
  long ones() const;

  void set(long n);
  void reset(long n);
  void Not(long n);

  void set(long n, Bool on) { on ? set(n) : reset(n); }
  void set();
  void reset();
  void set(const char* numbers);
};


#endif
