#ifndef _included_Region2_h
#define _included_Region2_h

#include "Point2.h"
#include "Globals.h"
#include "Environ.h"
#include <math.h>

#ifdef HAS_GRASS_4_2
extern "C" {
#include "gis.h"
}
#endif

#ifndef LOWERBOUND
#define LOWERBOUND (-1000000000)
#endif
#ifndef UPPERBOUND
#define UPPERBOUND ( 1000000000)
#endif

//----------------------------------------------------------------------------------------
  //						GeoRegion
  //----------------------------------------------------------------------------------------

class Region2 {
 
 
protected:

   Point2 lwp, upp, inc;

   /*
     Constructors and destructor for class Region2 -- inline these
   */


public:

   inline Region2() : lwp(UPPERBOUND,UPPERBOUND),
                      upp(LOWERBOUND,LOWERBOUND), inc(1) { }

   inline Region2(const Point2& l, const Point2& u) : lwp(l), upp(u), inc(1) { }
   inline Region2(const Point2& l, const Point2& u, const Point2& d) : lwp(l), upp(u), inc(d) { }
   inline Region2(const int i,  const int j, const int ii, const int jj, const int iii=1, const int jjj=1)
     : lwp(i,  j), upp(ii, jj), inc(iii, jjj)  { }
   inline Region2(const Region2& r) { lwp = r.lwp; upp = r.upp; inc = r.inc; }
   inline Region2& operator = (const Region2& r)
     { lwp = r.lwp; upp = r.upp; inc = r.inc; return(*this); }
   inline ~Region2() { }
   
   /*
     Functions which modify regions -- lower and upper bounds
   */

   inline void setempty() {
      lwp.elem(0) = UPPERBOUND; upp.elem(0) = LOWERBOUND;
      lwp.elem(1) = UPPERBOUND; upp.elem(1) = LOWERBOUND;
   }
   
   inline void setlower(const int i,  const int j) {
      lwp.elem(0) = i;
      lwp.elem(1) = j;
   }
   inline void setlower(const Point2& l) { lwp = l; }

   inline void setupper(const int i,  const int j) {
      upp.elem(0) = i;
      upp.elem(1) = j;
   }
   inline void setupper(const Point2& u) { upp = u; }

   inline void setincrement(const int i,  const int j) {
      inc.elem(0) = i;
      inc.elem(1) = j;
   }
   inline void setincrement(const Point2& u) { inc = u; }

   inline void setregion(const int i,  const int j, const int ii, const int jj, const int iii=1, const int jjj=1)
     {
      setlower(i,  j);
      setupper(ii, jj);
      setincrement(iii, jjj);
     }
   inline void setregion(const Point2& l, const Point2& u) { lwp = l; upp = u; }
   inline void setregion(const Point2& l, const Point2& u, const Point2& d) { lwp = l; upp = u; inc = d; }
   inline void setregion(const Region2& r) { lwp = r.lwp; upp = r.upp; inc = r.inc; }
   
   /*
     Some simple inline access functions for class Region2
   */

   inline const Point2& lower() const { return(lwp); }
   inline const Point2& upper() const { return(upp); }
   inline int lower(const int i) const { return(lwp(i)); }
   inline int upper(const int i) const { return(upp(i)); }
   inline const Point2& increment() const { return(inc); }
   inline int increment(const int i) const { return(inc(i)); }

   inline Point2& get_lower() { return(lwp); }
   inline Point2& get_upper() { return(upp); }
   inline int& get_lower(const int i) { return(lwp.elem(i)); }
   inline int& get_upper(const int i) { return(upp.elem(i)); }
   inline Point2& get_increment() { return(inc); }
   inline int& get_increment(const int i) { return(inc.elem(i)); }

   inline int extents(const int i) const { return(upp(i)-lwp(i)+1); }
   inline Point2 extents() const { return(upp-lwp+1); }

   inline int nelem(const int i) const { return (int) Util::ceil( (upp(i)-lwp(i)+1)/((float)inc(i))); }

   /*
     Check to see if this region is empty -- check each of the elements
   */

   inline int empty() const
     { return((upp(0) < lwp(0)) || (upp(1) < lwp(1)));}
     
   inline int size() const
     { return( empty() ? 0 : (upp(0)-lwp(0)+1) * (upp(1)-lwp(1)+1)); }

   inline int nelem() const
     { return( empty() ? 0 : nelem(0)*nelem(1)); }

   /*
     Define the bounding box operators + and += (bounding box union)
     Define the intersection operators * and *=
   */

   Region2& operator += (const Region2& rhs);
   inline Region2 operator + (const Region2& rhs) const
     {
      Region2 bbox(*this);
      bbox += rhs;
      return(bbox);
     }
   inline Region2& operator *= (const Region2& rhs)
     {
      lwp.max(rhs.lwp);
      upp.min(rhs.upp);
      return(*this);
     }
   inline Region2 operator * (const Region2& rhs) const
     {
      Region2 both(*this);
      both.lwp.max(rhs.lwp);
      both.upp.min(rhs.upp);
      return(both);
     }

   /*
     Define the comparison operators == and !=
   */

   int operator == (const Region2& rhs) const;
   inline int operator != (const Region2& rhs) const
     { return(!(*this == rhs)); }

   /*
     Member functions inside() return whether a point is inside the region
   */

   inline int inside(const int i,  const int j) const
     {
      return(((i >= lwp(0)) && (i <= upp(0)))
             && ((j >= lwp(1)) && (j <= upp(1))));
     }
   inline int inside(const Point2& p) const
     {
      
      return(Region2::inside(p(0), p(1)));
      
      
     }
   inline int ghostLayer (const int i,  const int j) const
     {
       if( inside(i,j) ) return 0;
       int di, dj;
       if(i<lwp(0)) di = lwp(0)-i;
       else if (i > upp(0)) di = i - upp(0);
       else di = 0;
       if(j<lwp(1)) dj = lwp(1)-j;
       else if (j > upp(1)) dj = j - upp(1);
       else dj = 0;
       return ((di>dj) ? di : dj);
     }

   /*
     Member function accrete() (also grow()) which grows a region
   */

   inline void accrete(const Point2& p)
     { if (!empty()) { lwp -= p; upp += p; } }
   inline void grow(const Point2& p)
     { if (!empty()) { lwp -= p; upp += p; } }
   inline void accrete(const int i)
     { if (!empty()) { lwp -= i; upp += i; } }
   inline void grow(const int i)
     { if (!empty()) { lwp -= i; upp += i; } }

  inline int bcoord( const int dim, const int gcoord ) const 
     { return ((gcoord-lower(dim))/inc(dim)); }

  inline int bindex( const int c0, const int c1 ) const 
     { return (bcoord(0,c0)*(extents(1)/inc(1)) + bcoord(1,c1)); }
     
  inline int cindex( const int c0, const int c1 ) const 
     { return ((c0-lower(0))*extents(1) + (c1-lower(1))); }

  };

/*
  Function accrete() increases or decreases the size of the region
*/

Region2 accrete(const Region2& region, const Point2& p);
Region2 grow(const Region2& region, const Point2& p);
Region2 accrete(const Region2& region, const int c);
Region2 grow(const Region2& region, const int c);
Region2* accrete(const Region2 *const region, const int n, const int c);
Region2* grow(const Region2 *const region, const int n, const int c);
Region2* accrete(const Region2 *const region, const int n, const Point2& p);
Region2* grow(const Region2 *const region, const int n, const Point2& p);

/*
  Function shift() shifts the  space of the region
*/

inline Region2 shift(const Region2& region, const Point2& p)
  { return(Region2(region.lower()+p, region.upper()+p)); }

//----------------------------------------------------------------------------------------
//						GeoCoord2
//----------------------------------------------------------------------------------------

enum EGeoCoordIndex { k0, k1 };

class GeoCoord2 {

protected:

  byte fInfo[8];
  float fC[2];

public:

  enum ECoordInfo { kCoordSet };
  inline GeoCoord2( ) { memset(fInfo,0,8); }
  inline GeoCoord2( float c0, float c1 ) { fC[0] = c0; fC[1] = c1; memset(fInfo,0,8); fInfo[kCoordSet] = 1; }  
  inline GeoCoord2( const GeoCoord2& g ) { fC[0] = g.fC[0]; fC[1] = g.fC[1]; memcpy(fInfo,g.fInfo,8); }
  inline GeoCoord2& operator = (const GeoCoord2& g) { fC[0] = g.fC[0]; fC[1] = g.fC[1];  memcpy(fInfo,g.fInfo,8); return *this; }
  inline float operator()( EGeoCoordIndex index ) const { return fC[index]; }
  inline float& elem( EGeoCoordIndex index ) { return fC[index]; }
  inline void operator()( float f0, float f1 ) { fC[0] = f0; fC[1] = f1; fInfo[kCoordSet] = 1; }
  byte Info( ECoordInfo index ) const { return fInfo[index]; }
};

//----------------------------------------------------------------------------------------
//						GeoRegion2
//----------------------------------------------------------------------------------------

class GeoRegion2 : public Region2 {
 
 
protected:

   GeoCoord2 fC[2];
   float fItoC[2];
   float fCellsize;
 
public:

   inline GeoRegion2() : Region2() { fCellsize = 1.0; }

   inline GeoRegion2(const Point2& l, const Point2& u) : Region2(l, u) { fCellsize = 1.0; fItoC[0] = fItoC[1] = 1.0;}
   inline GeoRegion2(const Point2& l, const Point2& u, const Point2& d) : Region2(l, u,d) { fCellsize = 1.0; fItoC[0] = fItoC[1] = 1.0; }
   inline GeoRegion2(const int i,  const int j, const int ii, const int jj, const int iii=1, const int jjj=1) 
     : Region2( i,   j,  ii,  jj,  iii,  jjj) { fCellsize = 1.0; fItoC[0] = fItoC[1] = 1.0;}
    inline GeoRegion2(const GeoRegion2& r) { 
     fCellsize = 1.0; lwp = r.lwp; upp = r.upp; inc = r.inc; fC[0] = r.fC[0]; fC[1] = r.fC[1]; CalcFactors(); }
   inline GeoRegion2(const Region2& r) { fCellsize = 1.0; lwp = r.lower(); upp = r.upper(); inc = r.increment(); fItoC[0] = fItoC[1] = 1.0; }
   inline GeoRegion2& operator = (const GeoRegion2& r) { 
     fCellsize = 1.0; lwp = r.lwp; upp = r.upp; inc = r.inc; fC[0] = r.fC[0]; fC[1] = r.fC[1]; CalcFactors(); return(*this); }
   inline GeoRegion2& operator = (const Region2& r) { 
      fCellsize = 1.0; lwp = r.lower(); upp = r.upper(); inc = r.increment(); fItoC[0] = fItoC[1] = 1.0; return(*this); }
   inline GeoRegion2( GeoCoord2 c0, GeoCoord2 c1, const Point2& l, const Point2& u): Region2(l, u) {
     fC[0] = c0; fC[1] = c1; CalcFactors(); fCellsize = 1.0;
   }
   
   inline void CalcFactors() {
     fItoC[0] = ((fC[1](k0) - fC[0](k0))/extents(0));
     fItoC[1] = ((fC[1](k1) - fC[0](k1))/extents(1));
   }

   inline void SetCoords( float r0, float c0, float r1, float c1) {
     fC[0](r0,c0); fC[1](r1,c1); CalcFactors();
   }
   
   inline void SetCellsize( float cs ) { fCellsize = cs; }

#ifdef HAS_GRASS_4_2

   inline GeoRegion2& operator = (const Cell_head& ch) { SetGeoRegion ( ch ); return *this; }

   inline int operator == (const Cell_head& ch) {       
     if( ch.north == fC[0](k0) && ch.east == fC[0](k1) && ch.south == fC[1](k0) && ch.west == fC[1](k1) 
	 && ch.rows == extents(0) && ch.cols == extents(1) ) return 1;
     else return 0;
   }

   inline int operator!=(const Cell_head& ch) { return !operator==(ch); }

   inline int SetGeoRegion (const Cell_head& ch) { 
     lwp = Point2(0,0);
     upp = Point2( ch.rows-1, ch.cols-1 );
     fC[0].elem(k0) = ch.north;
     fC[0].elem(k1) = ch.east;
     fC[1].elem(k0) = ch.south;
     fC[1].elem(k1) = ch.west;
     CalcFactors();
     return 1; 
   }

   inline int SetGeoRegion ( int nrows, int ncols, float west, float east,  float south, float north  ) { 
     lwp = Point2(0,0);
     upp = Point2( nrows-1, ncols-1 );
     fC[0].elem(k0) = north;
     fC[0].elem(k1) = east;
     fC[1].elem(k0) = south;
     fC[1].elem(k1) = west;
     CalcFactors();
     return 1; 
   }

   int SetGRASSRegion( Cell_head& ch ) const {
      ch.north = fC[0](k0);
      ch.east  = fC[0](k1);
      ch.south = fC[1](k0);
      ch.west  = fC[1](k1);
      ch.rows  = extents(0);
      ch.cols  = extents(1);
      return 1;
   }
   
#endif

   inline float Coord( int index, EGeoCoordIndex dim ) const { return index*fItoC[(byte)dim]; }
   inline int CellIndex( float coord, EGeoCoordIndex dim ) const { return (int)(coord/fItoC[(byte)dim]); }
    
   inline int GetCoordOfPoint( Point2& p, GeoCoord2& c ) const {
#ifdef DEBUG
     if( !( fC[0].Info(GeoCoord2::kCoordSet) && fC[1].Info(GeoCoord2::kCoordSet) ) ) { 
       gPrintErr( "Can't get coordinates until spatial Ref is set."); return 0; 
     } 
#endif
     c.elem(k0) =  Coord(p(0),k0);
     c.elem(k1) =  Coord(p(1),k1);
     return 1;
   }
    inline int GetPointFromCoord( Point2& p, GeoCoord2& c ) const {
#ifdef DEBUG
     if( !( fC[0].Info(GeoCoord2::kCoordSet) && fC[1].Info(GeoCoord2::kCoordSet) ) ) { 
       gPrintErr( "Can't get coordinates until spatial Ref is set."); return 0; 
     } 
#endif
     p.elem(0) =  CellIndex(c(k0),k0);
     p.elem(1) =  CellIndex(c(k1),k1);
     return 1;
   }
  inline const GeoCoord2& operator[]( EGeoCoordIndex index ) const { return fC[index]; }
};   

#endif
